﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_UserModel.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_ShaderUtility.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_ModelUtility.h>
#include <nw/g3d/g3d_edit.h>

namespace nw { namespace g3d { namespace demo {

namespace {

const char* s_BlockNameTable[NUM_BLOCK] = { "view", "env", "outline", "context" };

}

MaterialAssign::MaterialAssign(nw::g3d::ResMaterial* pResMaterial)
{
    NW_G3D_ASSERT_NOT_NULL(pResMaterial);

    m_pResMaterial = pResMaterial;
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        Pass& pass = m_PassArray[idxPass];
        pass.pShadingModelObj = NULL;
        pass.pShadingModel = NULL;
        memset(pass.pSamplerArray, 0, sizeof(pass.pSamplerArray));
        memset(pass.blockIndexArray, -1, NUM_BLOCK);
    }
}

nw::g3d::BindResult MaterialAssign::BindShader(
    PassType type, nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    Pass& pass = m_PassArray[type];
    pass.pShadingModel = pShadingModel;
    const nw::g3d::ResShaderAssign* pShaderAssign = m_pResMaterial->GetShaderAssign();

    // UniformBlock のインデックスを初期化します。
    for (int idxBlock = 0; idxBlock < NUM_BLOCK; ++idxBlock)
    {
        int blockIndex = pShadingModel->GetUniformBlockIndex(s_BlockNameTable[idxBlock]);
        pass.blockIndexArray[idxBlock] = static_cast<s8>(blockIndex);
    }

    // シェーダパラメータの変換コンバータを設定します。
    int idxMatBlock = pShadingModel->GetSystemBlockIndex(
        nw::g3d::ResUniformBlockVar::TYPE_MATERIAL);
    if (idxMatBlock >= 0)
    {
        nw::g3d::ResUniformBlockVar* pMaterialBlock = pShadingModel->GetUniformBlock(idxMatBlock);
        for (int idxShaderParam = 0, numShaderParam = m_pResMaterial->GetShaderParamCount();
            idxShaderParam < numShaderParam; ++idxShaderParam)
        {
            nw::g3d::ResShaderParam* pShaderParam = m_pResMaterial->GetShaderParam(idxShaderParam);
            if (nw::g3d::ResUniformVar* pUniform = pMaterialBlock->GetUniform(pShaderParam->GetId()))
            {
                if (const char* pConverter = pUniform->GetConverter())
                {
                    if (strcmp(pConverter, "texsrt_ex") == 0)
                    {
                        pShaderParam->SetConvertParamCallback(
                            nw::g3d::ResShaderParam::ConvertTexSrtExCallback);
                        m_pResMaterial->SetVolatile(idxShaderParam);
                    }
                }
            }
        }
    }

    // サンプラーを初期化します。
    const nw::g3d::ResSampler** ppSamplerArray = pass.pSamplerArray;

    int numSampler = pShadingModel->GetSamplerCount();
    NW_G3D_ASSERT(numSampler <= MAX_SAMPLER);

    for (int idxSampler = 0; idxSampler < numSampler; ++idxSampler)
    {
        const char* id = pShadingModel->GetSamplerName(idxSampler);
        // id で assign をルックアップします。
        const char* name = pShaderAssign->GetSamplerAssign(id);
        if (name == NULL)
        {
            // ツールでサンプラーが関連付けられていない場合。
            ppSamplerArray[idxSampler] = NULL; // 決め打ちのサンプラーを挿入する運用も可。
        }
        else if (const ResSampler* pResSampler = m_pResMaterial->GetSampler(name))
        {
            // ツールでモデル内のサンプラーを関連付けた場合。
             ppSamplerArray[idxSampler] = pResSampler;
        }
        else
        {
            // ツールでモデル内のサンプラーを選択しなかった場合。
            // シェーダアノテーションで指定した alt に応じた処理を行うことができます。
        }
    }

    // マテリアルのシェーダパラメータの初期化はデフォルトパスのシェーダでのみ行います。
    if (type != PASS_DEFAULT)
    {
        return nw::g3d::BindResult::Bound();
    }

    nw::g3d::ShaderUtility::BindShaderParam(m_pResMaterial, pShadingModel);
    return nw::g3d::BindResult::Bound();
}

void MaterialAssign::BindDebugShader(nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    Pass& pass = m_PassArray[PASS_DEBUG];
    pass.pShadingModel = pShadingModel;

    // UniformBlock のインデックスを初期化します。
    for (int idxBlock = 0; idxBlock < NUM_BLOCK; ++idxBlock)
    {
        int blockIndex = pShadingModel->GetUniformBlockIndex(s_BlockNameTable[idxBlock]);
        pass.blockIndexArray[idxBlock] = static_cast<s8>(blockIndex);
    }

    // サンプラーを初期化します。
    static const char* samplerName[] = { "_a0", "_n0" };
    for (int idxSampler = 0, numSamplerVar = sizeof(samplerName) / sizeof(*samplerName);
        idxSampler < numSamplerVar; ++idxSampler)
    {
        const char* name = samplerName[idxSampler];
        if (const nw::g3d::ResSampler* pResSampler = m_pResMaterial->GetSampler(name))
        {
            int idxSamplerVar = pShadingModel->GetSamplerIndex(name);
            NW_G3D_ASSERT(idxSamplerVar >= 0);
            pass.pSamplerArray[idxSamplerVar] = pResSampler;
        }
    }
}

void MaterialAssign::Setup()
{
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        Pass& pass = m_PassArray[idxPass];
        if (pass.pShadingModel == NULL)
        {
            pass.pShadingModelObj = NULL;
            continue;
        }

        nw::g3d::ShadingModelObj::InitArg arg(pass.pShadingModel);
        pass.pShadingModelObj = CreateShadingModelObj(arg);

        // マテリアルが持っているシェーダオプションを反映します。
        // マテリアルの設定がシェーダと一致する場合は次の変数を true にすることで
        // 完全性をチェックできます。
        bool materialPass = false;
        if (idxPass == PASS_DEFAULT || idxPass == PASS_OPAQUE || idxPass == PASS_TRANSLUCENT)
        {
            materialPass = true;
        }
        nw::g3d::ResShaderAssign* pResShaderAssign = m_pResMaterial->GetShaderAssign();
        nw::g3d::ShaderUtility::InitShaderKey(pass.pShadingModelObj, pResShaderAssign, materialPass);

        pass.pShadingModelObj->UpdateShaderRange();
        pass.pShadingModelObj->CalcOptionBlock();
    }
}

void MaterialAssign::Cleanup()
{
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        Pass& pass = m_PassArray[idxPass];
        if (pass.pShadingModelObj == NULL)
        {
            continue;
        }

        DestroyShadingModelObj(pass.pShadingModelObj);
    }
}

//--------------------------------------------------------------------------------------------------

ShapeAssign::ShapeAssign(nw::g3d::ResShape* pResShape, nw::g3d::ResMaterial* pResMaterial)
{
    NW_G3D_ASSERT_NOT_NULL(pResShape);

    m_pResShape = pResShape;
    m_pResMaterial = pResMaterial;

    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        Pass& pass = m_PassArray[idxPass];
        pass.pShadingModel = NULL;
        memset(pass.pVtxAttribArray, 0, sizeof(pass.pVtxAttribArray));
        memset(pass.pVtxBufferArray, 0, sizeof(pass.pVtxBufferArray));
        pass.vtxBufferCount = 0;
    }

    // "_t0" 及び "_n0" の有無で法線と接線のストリームアウトの要否を判定します。
    nw::g3d::ResVertex* pResVertex = m_pResShape->GetVertex();
    if (pResVertex->GetVtxAttrib("_t0"))
    {
        m_StreamType = STREAM_PNT;
    }
    else if (pResVertex->GetVtxAttrib("_n0"))
    {
        m_StreamType = STREAM_PN;
    }
    else
    {
        m_StreamType = STREAM_P;
    }
    m_pStreamOutShader = NULL;
}

void ShapeAssign::BindShader(PassType type, nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    Pass& pass = m_PassArray[type];
    pass.pShadingModel = pShadingModel;
    const nw::g3d::ResShaderAssign* pShaderAssign = m_pResMaterial->GetShaderAssign();

    // 頂点属性の関連付けを記録します。
    const nw::g3d::ResVtxAttrib** ppVtxAttribArray = pass.pVtxAttribArray;
    const nw::g3d::ResBuffer** ppVtxBufferArray = pass.pVtxBufferArray;

    int numAttribVar = pShadingModel->GetAttribCount();
    NW_G3D_ASSERT(numAttribVar <= MAX_ATTRIB);

    const nw::g3d::ResVertex* pResVertex = m_pResShape->GetVertex();
    for (int idxAttribVar = 0; idxAttribVar < numAttribVar; ++idxAttribVar)
    {
        const char* id = pShadingModel->GetAttribName(idxAttribVar);
        // id で assign をルックアップする。
        const char* name = pShaderAssign->GetAttribAssign(id);
        if (name == NULL)
        {
            // ツールで頂点属性が関連付けられていない場合。
            ppVtxAttribArray[idxAttribVar] = NULL; // 決め打ちの頂点属性を挿入する運用も可。
        }
        else
        {
            // ツールでモデル内の頂点属性を関連付けた場合。
            const nw::g3d::ResVtxAttrib* pVtxAttrib = pResVertex->GetVtxAttrib(name);
            ppVtxAttribArray[idxAttribVar] = pVtxAttrib;
            if (pVtxAttrib == NULL)
            {
                continue; // 同一マテリアルの異なるシェイプには存在しない場合があります。
            }

            const nw::g3d::ResBuffer* pBuffer =
                pResVertex->GetVtxBuffer(pVtxAttrib->GetBufferIndex());
            int idxBuffer = 0;
            for (; idxBuffer < MAX_BUFFER; ++idxBuffer)
            {
                if (ppVtxBufferArray[idxBuffer] == pBuffer)
                {
                    break; // 既に登録済みであれば何もしません。
                }
                else if (ppVtxBufferArray[idxBuffer] == NULL)
                {
                    ppVtxBufferArray[idxBuffer] = pBuffer;
                    ++pass.vtxBufferCount;
                    break; // 未登録であれば登録します。
                }
            }
            NW_G3D_ASSERT(idxBuffer != MAX_BUFFER);
        }
    }

    // フェッチシェーダの構築はメモリの確保とハンドルの取得が必要なため Setup で行います。
}

void ShapeAssign::BindStreamOutShader(nw::g3d::ResShaderArchive* pShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL(pShaderArchive);

    NW_G3D_TABLE_FIELD char* tblShadingModelName[] = { "P", "PN", "PNT" };
    m_pStreamOutShader = pShaderArchive->GetShadingModel(tblShadingModelName[m_StreamType]);
}

void ShapeAssign::BindDebugShader(nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    Pass& pass = m_PassArray[PASS_DEBUG];
    pass.pShadingModel = pShadingModel;

    // 頂点属性の関連付けを記録します。
    int numAttribVar = pShadingModel->GetAttribCount();
    NW_G3D_ASSERT(numAttribVar <= MAX_ATTRIB);
    const nw::g3d::ResVtxAttrib** ppVtxAttribArray = pass.pVtxAttribArray;
    const nw::g3d::ResBuffer** ppVtxBufferArray = pass.pVtxBufferArray;

    NW_G3D_TABLE_FIELD char* attribName[] = {
        "_p0", "_n0", "_t0", "_b0", "_i0", "_i1", "_w0", "_w1", "_c0", "_u0"
    };

    // オプションに依存しない割り当てを行います。
    for (int idxAttrib = 0, numAttrib = sizeof(attribName) / sizeof(*attribName);
        idxAttrib < numAttrib; ++idxAttrib)
    {
        const char* name = attribName[idxAttrib];
        if (nw::g3d::ResVtxAttrib* pResVtxAttrib = m_pResShape->GetVertex()->GetVtxAttrib(name))
        {
            int idxAttribVar = pShadingModel->GetAttribIndex(name);
            NW_G3D_ASSERT(idxAttribVar >= 0);
            ppVtxAttribArray[idxAttribVar] = pResVtxAttrib;
        }
    }

    for (int idxAttribVar = 0; idxAttribVar < numAttribVar; ++idxAttribVar)
    {
        if (const nw::g3d::ResVtxAttrib* pVtxAttrib = ppVtxAttribArray[idxAttribVar])
        {
            nw::g3d::ResBuffer* pBuffer =
                m_pResShape->GetVertex()->GetVtxBuffer(pVtxAttrib->GetBufferIndex());
            int idxBuffer = 0;
            for (; idxBuffer < MAX_BUFFER; ++idxBuffer)
            {
                if (ppVtxBufferArray[idxBuffer] == pBuffer)
                {
                    break; // 既に登録済みであれば何もしません。
                }
                else if (ppVtxBufferArray[idxBuffer] == NULL)
                {
                    ppVtxBufferArray[idxBuffer] = pBuffer;
                    ++pass.vtxBufferCount;
                    break; // 未登録であれば登録します。
                }
            }
            NW_G3D_ASSERT(idxBuffer != MAX_BUFFER);
        }
    }

    // フェッチシェーダの構築はメモリの確保とハンドルの取得が必要なため Setup で行います。
}

void ShapeAssign::Setup()
{
    MaterialAssign* pMatAssign = m_pResMaterial->GetUserPtr<MaterialAssign>();
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        Pass& pass = m_PassArray[idxPass];
        if (pass.pShadingModel == NULL)
        {
            pass.pShadingModelObj = NULL;
            continue;
        }
        pass.pShadingModelObj =
            pMatAssign->GetPass(static_cast<PassType>(idxPass)).pShadingModelObj;

        nw::g3d::GfxFetchShader& fetchShader = pass.fetchShader;
        NW_G3D_ASSERT(fetchShader.GetShaderPtr() == NULL);

        nw::g3d::GfxFetchShader& fetchShaderSO = pass.fetchShaderSO;
        NW_G3D_ASSERT(fetchShaderSO.GetShaderPtr() == NULL);

        // フェッチシェーダを構築します。
        const nw::g3d::ResShadingModel* pShadingModel = pass.pShadingModel;
        const nw::g3d::ResVtxAttrib** ppVtxAttribArray = pass.pVtxAttribArray;
        const nw::g3d::ResBuffer** ppVtxBufferArray = pass.pVtxBufferArray;
        int vtxBufferCount = pass.vtxBufferCount;

        int numNotUsed = std::count(ppVtxAttribArray, ppVtxAttribArray + MAX_ATTRIB,
            static_cast<nw::g3d::ResVtxAttrib*>(NULL));
        int numAttribValid = MAX_ATTRIB - numNotUsed;

        fetchShader.SetAttribCount(numAttribValid);
        fetchShader.CalcSize();
        fetchShader.SetShaderPtr(AllocMem2(fetchShader.GetShaderSize(), GX2_SHADER_ALIGNMENT));
        fetchShader.SetDefault();
        fetchShaderSO.SetAttribCount(numAttribValid);
        fetchShaderSO.CalcSize();
        fetchShaderSO.SetShaderPtr(AllocMem2(fetchShaderSO.GetShaderSize(), GX2_SHADER_ALIGNMENT));
        fetchShaderSO.SetDefault();

        int idxAttribValid = 0;
        for (int idxAttribVar = 0, numAttribVar = pShadingModel->GetAttribCount();
            idxAttribVar < numAttribVar; ++idxAttribVar)
        {
            const nw::g3d::ResVtxAttrib* pAttrib = ppVtxAttribArray[idxAttribVar];
            if (pAttrib == NULL)
            {
                continue;
            }
            const nw::g3d::ResAttribVar* pAttribVar = pShadingModel->GetAttrib(idxAttribVar);
            int bufferIndex = pAttrib->GetBufferIndex();
            const nw::g3d::ResBuffer* pResBuffer =
                m_pResShape->GetVertex()->GetVtxBuffer(bufferIndex);
            const nw::g3d::ResBuffer** ppFound = std::find(
                ppVtxBufferArray, ppVtxBufferArray + vtxBufferCount, pResBuffer);
            NW_G3D_ASSERT(ppFound != ppVtxBufferArray + vtxBufferCount);

            u32 bufferSlot = std::distance(ppVtxBufferArray, ppFound);
            GX2AttribFormat format = pAttrib->GetFormat();
            u32 offset = pAttrib->GetOffset();
            u32 location = pAttribVar->GetLocation();
            const nw::g3d::GfxBuffer* pVertexBuffer = pResBuffer->GetGfxBuffer();

            fetchShader.SetBufferSlot(idxAttribValid, bufferSlot);
            fetchShader.SetFormat(idxAttribValid, format);
            fetchShader.SetOffset(idxAttribValid, offset);
            fetchShader.SetLocation(idxAttribValid, location);
            fetchShader.SetVertexBuffer(idxAttribValid, pVertexBuffer);

            // ストリームアウトした位置と法線、接線を入力するフェッチシェーダを別に作成します。
            // リトルエンディアンで書き込まれているため、EndianSwapMode を None に設定します。
            // プログラムで制御するためフォーマットとオフセットは決め打ちです。
            if (0 == strcmp(pAttrib->GetName(), "_p0"))
            {
                fetchShaderSO.SetBufferSlot(idxAttribValid, pass.vtxBufferCount);
                fetchShaderSO.SetFormat(idxAttribValid, GX2_ATTRIB_FORMAT_32_32_32_FLOAT);
                fetchShaderSO.SetEndianSwapMode(idxAttribValid, GX2_ENDIANSWAP_NONE);
                fetchShaderSO.SetOffset(idxAttribValid, 0);
            }
            else if (0 == strcmp(pAttrib->GetName(), "_n0"))
            {
                fetchShaderSO.SetBufferSlot(idxAttribValid, pass.vtxBufferCount);
                fetchShaderSO.SetFormat(idxAttribValid, GX2_ATTRIB_FORMAT_8_8_8_8_SNORM);
                fetchShaderSO.SetEndianSwapMode(idxAttribValid, GX2_ENDIANSWAP_NONE);
                fetchShaderSO.SetOffset(idxAttribValid, 12);
            }
            else if (0 == strcmp(pAttrib->GetName(), "_t0"))
            {
                fetchShaderSO.SetBufferSlot(idxAttribValid, pass.vtxBufferCount);
                fetchShaderSO.SetFormat(idxAttribValid, GX2_ATTRIB_FORMAT_8_8_8_8_SNORM);
                fetchShaderSO.SetEndianSwapMode(idxAttribValid, GX2_ENDIANSWAP_NONE);
                fetchShaderSO.SetOffset(idxAttribValid, 16);
            }
            else
            {
                fetchShaderSO.SetBufferSlot(idxAttribValid, bufferSlot);
                fetchShaderSO.SetFormat(idxAttribValid, format);
                fetchShaderSO.SetOffset(idxAttribValid, offset);
            }

            fetchShaderSO.SetLocation(idxAttribValid, location);
            fetchShaderSO.SetVertexBuffer(idxAttribValid, pVertexBuffer);

            ++idxAttribValid;
        }

        fetchShader.Setup();
        fetchShader.DCFlush();
        fetchShaderSO.Setup();
        fetchShaderSO.DCFlush();
    }
}

void ShapeAssign::Cleanup()
{
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        nw::g3d::GfxFetchShader& fetchShader = m_PassArray[idxPass].fetchShader;
        if (void* shaderPtr = fetchShader.GetShaderPtr())
        {
            fetchShader.Cleanup();
            FreeMem2(shaderPtr);
            fetchShader.SetShaderPtr(NULL);
        }
        nw::g3d::GfxFetchShader& fetchShaderSO = m_PassArray[idxPass].fetchShaderSO;
        if (void* shaderPtr = fetchShaderSO.GetShaderPtr())
        {
            fetchShaderSO.Cleanup();
            FreeMem2(shaderPtr);
            fetchShaderSO.SetShaderPtr(NULL);
        }
    }
}

//--------------------------------------------------------------------------------------------------

ModelAssign::ModelAssign(nw::g3d::ResModel* pResModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResModel);

    m_pResModel = pResModel;
    m_pModelShader = NULL;
}

nw::g3d::BindResult ModelAssign::BindShader(
    PassType type, nw::g3d::ResShaderArchive* pShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL(pShaderArchive);

    nw::g3d::BindResult result;

    for (int idxMat = 0, numMat = m_pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        if (pMatAssign->GetPass(type).pShadingModel)
        {
            continue; // 割り当て済みなのでスキップします。
        }

        ResShaderAssign* pShaderAssign = pResMaterial->GetShaderAssign();
        if (0 == strcmp(pShaderAssign->GetShaderArchiveName(), pShaderArchive->GetName()))
        {
            const char* name = pShaderAssign->GetShadingModelName();
            nw::g3d::ResShadingModel* pShadingModel = pShaderArchive->GetShadingModel(name);
            if (pShadingModel)
            {
                result |= pMatAssign->BindShader(type, pShadingModel);
                continue;
            }
        }
        result |= nw::g3d::BindResult::NotBound();
    }

    BindShaderShape(type);

    return result;
}

nw::g3d::BindResult ModelAssign::BindShader(
    PassType type, nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    nw::g3d::BindResult result;

    for (int idxMat = 0, numMat = m_pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        if (pMatAssign->GetPass(type).pShadingModel)
        {
            continue; // 割り当て済みなのでスキップします。
        }

        result |= pMatAssign->BindShader(type, pShadingModel);
    }

    BindShaderShape(type);

    return result;
}

nw::g3d::BindResult ModelAssign::BindShader(
    PassType type, nw::g3d::ResShaderArchive** ppShaderArchives, int shaderArchiveCount)
{
    NW_G3D_UNUSED(shaderArchiveCount);
    NW_G3D_ASSERT_NOT_NULL(ppShaderArchives);

    nw::g3d::BindResult result;

    int numMat = m_pResModel->GetMaterialCount();
    NW_G3D_ASSERT(shaderArchiveCount == numMat);
    for (int idxMat = 0; idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = ppShaderArchives[idxMat];
        if (!pShaderArchive)
        {
            result |= nw::g3d::BindResult::NotBound();
            continue;
        }

        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        if (pMatAssign->GetPass(type).pShadingModel)
        {
            continue; // 割り当て済みなのでスキップします。
        }

        ResShaderAssign* pShaderAssign = pResMaterial->GetShaderAssign();
        NW_G3D_ASSERT(0 == strcmp(
            pShaderAssign->GetShaderArchiveName(), pShaderArchive->GetName()));
        const char* name = pShaderAssign->GetShadingModelName();
        nw::g3d::ResShadingModel* pShadingModel = pShaderArchive->GetShadingModel(name);
        NW_G3D_ASSERT_NOT_NULL(pShadingModel);
        result |= pMatAssign->BindShader(type, pShadingModel);
    }

    BindShaderShape(type);

    return result;
}

void ModelAssign::BindStreamOutShader(nw::g3d::ResShaderArchive* pShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL(pShaderArchive);

    for (int idxShape = 0, numShape = m_pResModel->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = m_pResModel->GetShape(idxShape);
        ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShapeAssign);
        pShapeAssign->BindStreamOutShader(pShaderArchive);
    }
}

void ModelAssign::BindDebugShader(nw::g3d::ResShadingModel* pShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModel);

    for (int idxMat = 0, numMat = m_pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        pMatAssign->BindDebugShader(pShadingModel);
    }

    for (int idxShape = 0, numShape = m_pResModel->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = m_pResModel->GetShape(idxShape);
        ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShapeAssign);
        pShapeAssign->BindDebugShader(pShadingModel);
    }
}

void ModelAssign::Setup()
{
    for (int idxMat = 0, numMat = m_pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        pMatAssign->Setup();
    }

    for (int idxShape = 0, numShape = m_pResModel->GetShapeCount();
        idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = m_pResModel->GetShape(idxShape);
        ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShapeAssign);
        pShapeAssign->Setup();
    }
}

void ModelAssign::Cleanup()
{
    for (int idxMat = 0, numMat = m_pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        pMatAssign->Cleanup();
    }

    for (int idxShape = 0, numShape = m_pResModel->GetShapeCount();
        idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = m_pResModel->GetShape(idxShape);
        ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShapeAssign);
        pShapeAssign->Cleanup();
    }
}

void ModelAssign::BindShaderShape(PassType type)
{
    for (int idxShape = 0, numShape = m_pResModel->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = m_pResModel->GetShape(idxShape);
        int idxMat = pResShape->GetMaterialIndex();
        nw::g3d::ResMaterial* pResMaterial = m_pResModel->GetMaterial(idxMat);
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        NW_G3D_ASSERT_NOT_NULL(pMatAssign);
        nw::g3d::ResShadingModel* pShadingModel = pMatAssign->GetPass(type).pShadingModel;
        if (pShadingModel == NULL)
        {
            continue; // マテリアルにシェーダが割り当てられていない場合はスキップします。
        }

        ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShapeAssign);
        pShapeAssign->BindShader(type, pShadingModel);
    }
}

//--------------------------------------------------------------------------------------------------

UserView::UserView()
{
}

void UserView::Setup()
{
    size_t size = sizeof(View);
    size_t alignedSize = nw::g3d::Align(size, GX2_UNIFORM_BLOCK_ALIGNMENT);
    void* ptr = AllocMem2(alignedSize * NUM_VIEW, GX2_UNIFORM_BLOCK_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
    {
        m_ViewBlock[idxView].SetData(ptr, size);
        m_ViewBlock[idxView].Setup();
        ptr = AddOffset(ptr, alignedSize);
    }
}

void UserView::Cleanup()
{
    for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
    {
        m_ViewBlock[idxView].Cleanup();
    }
    FreeMem2(m_ViewBlock[0].GetData());
}

void UserView::Calc(int viewIndex)
{
    View* pView = static_cast<View*>(m_ViewBlock[viewIndex].GetData());
    nw::g3d::Mtx44& projMtx = m_View[viewIndex].projMtx;
    nw::g3d::Mtx34& viewMtx = m_View[viewIndex].viewMtx;
    nw::g3d::Mtx34& invViewMtx = m_View[viewIndex].invViewMtx;
    float det;
    invViewMtx.Inverse(&det, viewMtx);
    nw::g3d::Copy32<true>(&pView->projMtx, &projMtx, sizeof(projMtx) >> 2);
    nw::g3d::Copy32<true>(&pView->viewMtx, &viewMtx, sizeof(viewMtx) >> 2);
    nw::g3d::Copy32<true>(&pView->invViewMtx, &invViewMtx, sizeof(invViewMtx) >> 2);
    m_ViewBlock[viewIndex].DCFlush();
}

//--------------------------------------------------------------------------------------------------

UserMaterial::UserMaterial(MaterialObj* pMaterialObj)
{
    NW_G3D_ASSERT_NOT_NULL(pMaterialObj);

    m_pMaterialObj = pMaterialObj;

    nw::g3d::ResRenderInfo* pInfo = pMaterialObj->GetResource()->GetRenderInfo("dynamic_shadow");
    m_RenderShadow = ( pInfo && *pInfo->GetInt() != 0 );
}

//--------------------------------------------------------------------------------------------------

UserShape::UserShape(ShapeObj* pShapeObj)
{
    NW_G3D_ASSERT_NOT_NULL(pShapeObj);

    m_pShapeObj = pShapeObj;

    // TODO: どこかに移動を検討

    // ストリームアウト用の設定を行います。
    nw::g3d::ResShape* pResShape = m_pShapeObj->GetResource();
    nw::g3d::ResVertex* pResVertex = pResShape->GetVertex();
    ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
    StreamType streamType = pShapeAssign->GetStreamType();
    {
        NW_G3D_TABLE_FIELD size_t tblStride[] = {
            sizeof(float) * 3,
            sizeof(float) * 3 + sizeof(int),
            sizeof(float) * 3 + sizeof(int) + sizeof(int)
        };
        size_t stride = tblStride[streamType];
        size_t size = stride * pResVertex->GetCount();
        size_t bufferSize = nw::g3d::Align(size + sizeof(GX2StreamOutContext), LL_CACHE_FETCH_SIZE);
        void* pBuffer = AllocMem2(bufferSize, GX2_STREAMOUT_BUFFER_ALIGNMENT);

        GX2StreamOutContext* pSOCtx = nw::g3d::AddOffset<GX2StreamOutContext>(pBuffer, size);
        nw::g3d::CPUCache::Flush(pSOCtx, sizeof(*pSOCtx));

        nw::g3d::GfxBuffer& streamOutBuffer = m_StreamOutPass.streamOutBuffer;
        streamOutBuffer.SetData(pBuffer, size);
        streamOutBuffer.SetStride(stride);
        streamOutBuffer.SetStreamOutContext(pSOCtx);
        streamOutBuffer.Setup();
        streamOutBuffer.DCFlush();

        nw::g3d::GfxStreamOut& streamOut = m_StreamOutPass.streamOut;
        streamOut.Setup();
        streamOut.SetPrimitiveType(GX2_PRIMITIVE_POINTS);
        streamOut.SetStreamOutBuffer(0, &streamOutBuffer);
    }

    // ストリームアウト用フェッチシェーダ
    {
        // シェイプアニメに対応するため、リソースではなくインスタンス毎に作成します。
        nw::g3d::GfxFetchShader& fetchShader = m_StreamOutPass.fetchShader;
        fetchShader.SetAttribCount(16); // シェイプアニメ用に最大数確保しておく。
        fetchShader.CalcSize();
        void* shaderPtr = AllocMem2(fetchShader.GetShaderSize(), GX2_SHADER_ALIGNMENT);
        fetchShader.SetShaderPtr(shaderPtr);
        fetchShader.SetAttribCount(0);

        m_StreamOutPass.vtxAttribCount = 0;
        m_StreamOutPass.vtxBufferCount = 0;
        memset(m_StreamOutPass.pVtxAttribArray, 0, sizeof(m_StreamOutPass.pVtxAttribArray));
        memset(m_StreamOutPass.pVtxBufferArray, 0, sizeof(m_StreamOutPass.pVtxBufferArray));
        memset(m_StreamOutPass.bufferSlotArray, 0, sizeof(m_StreamOutPass.bufferSlotArray));

        // シェーダで explicit に決めた location でフェッチシェーダを作成します。
        NW_G3D_TABLE_FIELD char* tblAttribName[] = {
            "_i0", "_w0", "_i1", "_w1", "_p0", "_n0", "_t0"
        };
        NW_G3D_TABLE_FIELD int tblAttribLoc[] = {
            0, 1, 2, 3, 4, 5, 6
        };
        u32 numValidAttrib = 0;
        u32 numVtxBuffer = 0;
        for (int idxAttrib = 0, numAttrib = sizeof(tblAttribName) / sizeof(*tblAttribName);
            idxAttrib < numAttrib; ++idxAttrib)
        {
            const char* attribName = tblAttribName[idxAttrib];
            if (const nw::g3d::ResVtxAttrib* pAttrib = pResVertex->GetVtxAttrib(attribName))
            {
                m_StreamOutPass.pVtxAttribArray[tblAttribLoc[idxAttrib]] = pAttrib;
                ++numValidAttrib;
                int bufferIndex = pAttrib->GetBufferIndex();
                NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, MAX_BUFFER);
                const nw::g3d::ResBuffer* pResBuffer = pResVertex->GetVtxBuffer(bufferIndex);
                const nw::g3d::ResBuffer** ppFound = std::find(m_StreamOutPass.pVtxBufferArray,
                    m_StreamOutPass.pVtxBufferArray + numVtxBuffer, pResBuffer);
                if (ppFound == m_StreamOutPass.pVtxBufferArray + numVtxBuffer)
                {
                    *ppFound = pResBuffer;
                    m_StreamOutPass.bufferSlotArray[numVtxBuffer++] = static_cast<s8>(bufferIndex);
                }
            }
        }
        m_StreamOutPass.vtxAttribCount = static_cast<u8>(numValidAttrib);
        m_StreamOutPass.vtxBufferCount = static_cast<u8>(numVtxBuffer);

        fetchShader.Setup();
    }
    UpdateFetchShader();
    m_StreamOutPass.validBlendCount = 0;
}

UserShape::~UserShape()
{
    nw::g3d::GfxBuffer& streamOutBuffer = m_StreamOutPass.streamOutBuffer;
    streamOutBuffer.Cleanup();
    if (streamOutBuffer.GetSize() > 0)
    {
        FreeMem2(streamOutBuffer.GetData());
    }

    m_StreamOutPass.streamOut.Cleanup();

    nw::g3d::GfxFetchShader& fetchShader = m_StreamOutPass.fetchShader;
    fetchShader.Cleanup();
    if (void* ptr = fetchShader.GetShaderPtr())
    {
        FreeMem2(ptr);
    }
}

void UserShape::Setup()
{
    nw::g3d::ResShape* pResShape = m_pShapeObj->GetResource();
    ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();

    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        ShapeAssign::Pass& assignPass = pShapeAssign->GetPass(static_cast<PassType>(idxPass));
        if (assignPass.pShadingModel == NULL)
        {
            m_pShaderSelectorArray[idxPass] = NULL;
            continue;
        }

        m_pShaderSelectorArray[idxPass] = SetupShaderSelector(assignPass.pShadingModelObj);
    }

    if (nw::g3d::ResShadingModel* pShadingModel = pShapeAssign->GetStreamOutShader())
    {
        nw::g3d::ShadingModelObj::InitArg arg(pShadingModel);
        nw::g3d::ShadingModelObj* pShadingModelObj = CreateShadingModelObj(arg);

        pShadingModelObj->UpdateShaderRange();
        pShadingModelObj->CalcOptionBlock();

        m_StreamOutPass.pShaderSelector = SetupShaderSelector(pShadingModelObj);
    }
    else
    {
        m_StreamOutPass.pShaderSelector = NULL;
    }
}

void UserShape::Cleanup()
{
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        if (m_pShaderSelectorArray[idxPass])
        {
            DestroyShaderSelector(m_pShaderSelectorArray[idxPass]);
        }
    }
    if (m_StreamOutPass.pShaderSelector)
    {
        DestroyShadingModelObj(m_StreamOutPass.pShaderSelector->GetShadingModel());
        DestroyShaderSelector(m_StreamOutPass.pShaderSelector);
    }
    m_StreamOutPass.validBlendCount = 0;
}

void UserShape::UpdateShader()
{
    for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
    {
        if (m_pShaderSelectorArray[idxPass])
        {
            bool found = m_pShaderSelectorArray[idxPass]->UpdateVariation();
            NW_G3D_UNUSED(found);
            NW_G3D_WARNING(found, "Shader variation not found.");
        }
    }
    if (m_StreamOutPass.pShaderSelector)
    {
        bool found = m_StreamOutPass.pShaderSelector->UpdateVariation();
        NW_G3D_UNUSED(found);
        NW_G3D_WARNING(found, "Shader variation not found.");
    }
}

void UserShape::UpdateFetchShader()
{
    nw::g3d::ResVertex* pResVertex = m_pShapeObj->GetResVertex();
    nw::g3d::GfxFetchShader& fetchShader = m_StreamOutPass.fetchShader;
    fetchShader.SetAttribCount(m_StreamOutPass.vtxAttribCount);
    fetchShader.CalcSize();
    NW_G3D_ASSERT_NOT_NULL(fetchShader.GetShaderPtr()); // ShaderPtr は設定済み。
    fetchShader.SetDefault();

    u32 idxValidAttrib = 0;
    for (int idxAttrib = 0; idxAttrib < MAX_ATTRIB; ++idxAttrib)
    {
        if (const nw::g3d::ResVtxAttrib* pAttrib = m_StreamOutPass.pVtxAttribArray[idxAttrib])
        {
            int bufferIndex = pAttrib->GetBufferIndex();
            const nw::g3d::ResBuffer* pResBuffer = pResVertex->GetVtxBuffer(bufferIndex);
            fetchShader.SetLocation(idxValidAttrib, idxAttrib);
            fetchShader.SetBufferSlot(idxValidAttrib, bufferIndex);
            fetchShader.SetFormat(idxValidAttrib, pAttrib->GetFormat());
            fetchShader.SetOffset(idxValidAttrib, pAttrib->GetOffset());
            fetchShader.SetVertexBuffer(idxValidAttrib, pResBuffer->GetGfxBuffer());
            ++idxValidAttrib;
        }
    }

    fetchShader.UpdateRegs();
    fetchShader.DCFlush();
}

void UserShape::UpdateBlendWeight(int /*viewIndex*/)
{
    for (int idxKeyShape = 0, idxDest = 0, numKeyShape = m_pShapeObj->GetKeyShapeCount();
        idxKeyShape < numKeyShape; ++idxKeyShape)
    {
        if (m_pShapeObj->IsBlendWeightValid(idxKeyShape))
        {
            m_pShapeObj->GetUserArea<f32>()[idxDest++] = m_pShapeObj->GetBlendWeight(idxKeyShape);
        }
    }
    // CalcShape() で DCFlush() が呼ばれるのでここでは省略します。
    //shapeBlock.DCFlush(m_CurBuffer);
}

void UserShape::UpdateShapeAnim()
{
    StreamOutPass& pass = m_StreamOutPass;
    nw::g3d::ResShape* pResShape = m_pShapeObj->GetResource();

    // 使用されているブレンド情報からシェーダバリエーションを選択します。
    ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
    StreamType streamType = pShapeAssign->GetStreamType();
    NW_G3D_TABLE_FIELD char* s_NameOption[NUM_STREAM] = { "num_pos", "num_nrm", "num_tan" };
    nw::g3d::ResKeyShape* pFirstKeyShape = pResShape->GetKeyShape(0);
    bool hasKeyAttrib[NUM_STREAM] = { pFirstKeyShape->GetPositionAttribIndex() >= 0,
        pFirstKeyShape->GetNormalAttribIndex() >= 0, pFirstKeyShape->GetTangentAttribIndex() >= 0 };
    for (int idxStream = 0; idxStream < NUM_STREAM; ++idxStream)
    {
        if (streamType >= idxStream)
        {
            // normal, tangent については、pos と同数のブレンドを行う場合は "2" を、ストリームアウトだけの場合は "1" を選びます。
            // 法線・接線を持たない場合は "0" ですが、その場合は初期値のまま変更しないはずです。
            int idxOption = pass.pShaderSelector->GetDynamicOptionIndex(s_NameOption[idxStream]);
            nw::g3d::ResShaderOption* pOption = pass.pShaderSelector->GetDynamicOption(idxOption);
            char valChoice[] = { '1', '\0' };
            if (hasKeyAttrib[idxStream])
            {
                valChoice[0] = idxStream == STREAM_P ?
                    '0' + static_cast<char>((std::max)(1, m_pShapeObj->CountValidBlendWeight())) : '2';
            }
            int idxChoice = pOption->GetChoiceIndex(valChoice);
            pass.pShaderSelector->WriteDynamicKey(idxOption, idxChoice);
        }
    }

    // 頂点属性を登録します。
    const size_t baseLocation = 4; // シェーダで定義した _p0 のロケーションです。
    pass.vtxAttribCount = static_cast<u8>(streamType) + 1;
    memset(pass.pVtxAttribArray + baseLocation + NUM_STREAM, 0,
        sizeof(nw::g3d::ResVtxAttrib*) * (MAX_ATTRIB - baseLocation - NUM_STREAM));
    const nw::g3d::ResVertex* pResVertex = pResShape->GetVertex();
    for (int idxKeyShape = 1, idxDest = 1, numKeyShape = m_pShapeObj->GetKeyShapeCount();
        idxKeyShape < numKeyShape; ++idxKeyShape)
    {
        if (m_pShapeObj->IsBlendWeightValid(idxKeyShape))
        {
            nw::g3d::ResKeyShape* pResKeyShape = pResShape->GetKeyShape(idxKeyShape);
            int vtxIndices[NUM_STREAM] = { pResKeyShape->GetPositionAttribIndex(),
                pResKeyShape->GetNormalAttribIndex(), pResKeyShape->GetTangentAttribIndex() };
            for (int idxStream = 0; idxStream < NUM_STREAM; ++idxStream)
            {
                if (vtxIndices[idxStream] >= 0)
                {
                    const nw::g3d::ResVtxAttrib* pAttrib = pResVertex->GetVtxAttrib(vtxIndices[idxStream]);
                    pass.pVtxAttribArray[baseLocation + idxDest * NUM_STREAM + idxStream] = pAttrib;
                    ++pass.vtxAttribCount;
                }
            }
            ++idxDest;
        }
    }

    // 頂点バッファを登録します。
    pass.vtxBufferCount = 0;
    for (int idxAttrib = 0; idxAttrib < MAX_ATTRIB; ++idxAttrib)
    {
        if (const nw::g3d::ResVtxAttrib* pAttrib = pass.pVtxAttribArray[idxAttrib])
        {
            int bufferIndex = pAttrib->GetBufferIndex();
            NW_G3D_ASSERT_INDEX_BOUNDS(bufferIndex, MAX_BUFFER);
            const nw::g3d::ResBuffer* pResBuffer = pResVertex->GetVtxBuffer(bufferIndex);
            const nw::g3d::ResBuffer** ppFound = std::find(pass.pVtxBufferArray,
                pass.pVtxBufferArray + pass.vtxBufferCount, pResBuffer);
            if (ppFound == pass.pVtxBufferArray + pass.vtxBufferCount)
            {
                *ppFound = pResBuffer;
                pass.bufferSlotArray[pass.vtxBufferCount++] = static_cast<s8>(bufferIndex);
            }
        }
    }
}

nw::g3d::ShaderSelector* UserShape::SetupShaderSelector(nw::g3d::ShadingModelObj* pShadingModelObj)
{
    nw::g3d::ResVertex* pResVertex = m_pShapeObj->GetResVertex();

    nw::g3d::ShaderSelector::InitArg initArg(pShadingModelObj);
    m_StreamOutPass.pShaderSelector = CreateShaderSelector(initArg);
    nw::g3d::ShaderSelector* pShaderSelector = m_StreamOutPass.pShaderSelector;

    // Vertex で静的に決まるバリエーションのキーを埋めます。
    int idxOption = pShaderSelector->GetDynamicOptionIndex("skinning");
    if (idxOption >= 0)
    {
        const nw::g3d::ResShaderOption* pOption = pShaderSelector->GetDynamicOption(idxOption);
        NW_G3D_ASSERT(pResVertex->GetVtxSkinCount() <= 8);
        // skinning="0" のバリエーションでは、Shape の UniformBlock 中の VtxSkinCount による
        // 静的分岐を使用してスキニングの種別を切り分ける実装を行っています。
        // skinning="1" 以降のバリエーションでは静的分岐を使用しない実装を行っています。
        // 開発中のシェーダコンパイル時間を短縮するために、skinning="0" のバリエーションを
        // 利用することが考えられます。
        char choice[2] = { static_cast<char>('1' + pResVertex->GetVtxSkinCount()), '\0' };
        int idxChoice = pOption->GetChoiceIndex(choice);
        pShaderSelector->WriteDynamicKey(idxOption, idxChoice);
    }

    return pShaderSelector;
}

//--------------------------------------------------------------------------------------------------

UserModel::UserModel(ModelObj* pModelObj)
#if NW_G3D_CONFIG_USE_HOSTIO
    : m_IsEditCalcSkeletalAnimationEnabled(false)
#endif
{
    NW_G3D_ASSERT_NOT_NULL(pModelObj);

    m_pModelObj = pModelObj;
    m_pView = NULL;
    m_pEnvModel = NULL;
    m_pOutlineModel = NULL;
    m_pBindModel = NULL;
    m_BindBoneIndex = -1;
    m_Scale.Set(1.0f, 1.0f, 1.0f);
    m_LayoutMtx.Identity();
    m_BaseMtx.Identity();
    m_CurBuffer = 0;
    m_pContext = NULL;
    for (int idxView = 0 ; idxView < NUM_VIEW ; ++idxView)
    {
        m_LODLevel[idxView] = 0;
    }
    memset(m_Weights, 0, sizeof(m_Weights));
    memset(m_pSkeletalAnims, 0, sizeof(m_pSkeletalAnims));
    memset(m_pVisibilityAnims, 0, sizeof(m_pVisibilityAnims));
    memset(m_pShaderParamAnims, 0, sizeof(m_pShaderParamAnims));
    memset(m_pTexPatternAnims, 0, sizeof(m_pTexPatternAnims));
    memset(m_pShapeAnims, 0, sizeof(m_pShapeAnims));
    m_StreamOutEnable = false;
    m_LockedCacheEnable = false;
    m_FrustumCullingEnable = false;
    m_DoubleBufferEnable = false;
    memset(m_pCalcLodLevel, 0, sizeof(m_pCalcLodLevel));
}

void UserModel::CreateAnimObjArg::Reserve(const void* const * pResAnim, int count)
{
    for (int index = 0; index < count; ++index)
    {
        const nw::g3d::BinaryBlockHeader* pHeader =
            static_cast<const nw::g3d::BinaryBlockHeader*>(pResAnim[index]);
        switch (pHeader->sigWord)
        {
        case nw::g3d::ResSkeletalAnim::SIGNATURE:
            initArgSkeletal.Reserve(
                static_cast<const nw::g3d::ResSkeletalAnim*>(pResAnim[index]));
            break;
        case nw::g3d::ResVisibilityAnim::SIGNATURE:
            initArgVisibility.Reserve(
                static_cast<const nw::g3d::ResVisibilityAnim*>(pResAnim[index]));
            break;
        case nw::g3d::ResShaderParamAnim::SIGNATURE:
            initArgShaderParam.Reserve(
                static_cast<const nw::g3d::ResShaderParamAnim*>(pResAnim[index]));
            break;
        case nw::g3d::ResTexPatternAnim::SIGNATURE:
            initArgTexPattern.Reserve(
                static_cast<const nw::g3d::ResTexPatternAnim*>(pResAnim[index]));
            break;
        case nw::g3d::ResShapeAnim::SIGNATURE:
            initArgShape.Reserve(
                static_cast<const nw::g3d::ResShapeAnim*>(pResAnim[index]));
            break;
        }
    }
}

void UserModel::CreateAnimObj(const CreateAnimObjArg& arg)
{
    nw::g3d::SkeletalAnimBlender::InitArg initArgBlender;
    initArgBlender.Reserve(m_pModelObj->GetResource());
    size_t bufSize = nw::g3d::SkeletalAnimBlender::CalcBufferSize(initArgBlender);
    void* ptr = AllocMem2(bufSize, nw::g3d::SkeletalAnimBlender::BUFFER_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL(ptr);
    bool success = m_SkelBlender.Init(initArgBlender, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);
    m_SkelBlender.SetBoneCount(m_pModelObj->GetSkeleton()->GetBoneCount());

    nw::g3d::SkeletalAnimObj::InitArg initArgSkeletal = arg.initArgSkeletal;
    initArgSkeletal.Reserve(m_pModelObj->GetResource());
    if (!initArgSkeletal.IsValid())
    {
        initArgSkeletal.SetMaxBoneAnimCount(0);
        initArgSkeletal.SetMaxCurveCount(0);
    }
    for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
    {
        m_pSkeletalAnims[idxAnim] = CreateSkeletalAnimObj(initArgSkeletal);
        m_Weights[idxAnim] = 0.0f;
    }

    nw::g3d::VisibilityAnimObj::InitArg initArgVisibility = arg.initArgVisibility;
    initArgVisibility.Reserve(m_pModelObj->GetResource());
    if (!initArgVisibility.IsValid())
    {
        initArgVisibility.SetMaxBoneAnimCount(0);
        initArgVisibility.SetMaxMatAnimCount(0);
        initArgVisibility.SetMaxCurveCount(0);
    }
    for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
    {
        m_pVisibilityAnims[idxAnim] = CreateVisibilityAnimObj(initArgVisibility);
    }

    nw::g3d::ShaderParamAnimObj::InitArg initArgShaderParam = arg.initArgShaderParam;
    initArgShaderParam.Reserve(m_pModelObj->GetResource());
    if (!initArgShaderParam.IsValid())
    {
        initArgShaderParam.SetMaxMatAnimCount(0);
        initArgShaderParam.SetMaxParamAnimCount(0);
        initArgShaderParam.SetMaxCurveCount(0);
    }
    for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
    {
        m_pShaderParamAnims[idxAnim] = CreateShaderParamAnimObj(initArgShaderParam);
    }

    nw::g3d::TexPatternAnimObj::InitArg initArgTexPattern = arg.initArgTexPattern;
    initArgTexPattern.Reserve(m_pModelObj->GetResource());
    if (!initArgTexPattern.IsValid())
    {
        initArgTexPattern.SetMaxMatAnimCount(0);
        initArgTexPattern.SetMaxPatAnimCount(0);
        initArgTexPattern.SetMaxCurveCount(0);
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        m_pTexPatternAnims[idxAnim] = CreateTexPatternAnimObj(initArgTexPattern);
    }

    nw::g3d::ShapeAnimObj::InitArg initArgShape = arg.initArgShape;
    initArgShape.Reserve(m_pModelObj->GetResource());
    if (!initArgShape.IsValid())
    {
        initArgShape.SetMaxVertexShapeAnimCount(0);
        initArgShape.SetMaxKeyShapeAnimCount(0);
        initArgShape.SetMaxCurveCount(0);
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        m_pShapeAnims[idxAnim] = CreateShapeAnimObj(initArgShape);
    }
}

void UserModel::DestroyAnimObj()
{
    if (void* ptr = m_SkelBlender.GetBufferPtr())
    {
        FreeMem2(ptr);
    }

    for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
    {
        nw::g3d::demo::DestroyAnimObj(m_pSkeletalAnims[idxAnim]);
        m_pSkeletalAnims[idxAnim] = NULL;
    }
    for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
    {
        nw::g3d::demo::DestroyAnimObj(m_pVisibilityAnims[idxAnim]);
        m_pVisibilityAnims[idxAnim] = NULL;
    }
    for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
    {
        nw::g3d::demo::DestroyAnimObj(m_pShaderParamAnims[idxAnim]);
        m_pShaderParamAnims[idxAnim] = NULL;
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        nw::g3d::demo::DestroyAnimObj(m_pTexPatternAnims[idxAnim]);
        m_pTexPatternAnims[idxAnim] = NULL;
    }
    for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
    {
        nw::g3d::demo::DestroyAnimObj(m_pShapeAnims[idxAnim]);
        m_pShapeAnims[idxAnim] = NULL;
    }
}

void UserModel::UpdateShader()
{
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();
        pUserShape->UpdateShader();
    }
}

void UserModel::UpdateBlendWeight()
{
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();

        // シェイプアニメ用にフェッチシェーダとブレンドウェイトを書き込みます。
        if (pShapeObj->HasValidBlendWeight())
        {
            if (pShapeObj->IsViewDependent())
            {
                for (int idxView = 0, numView = pShapeObj->GetViewCount();
                    idxView < numView; ++idxView)
                {
                    pUserShape->UpdateBlendWeight(idxView);
                }
            }
            else
            {
                pUserShape->UpdateBlendWeight(0);
            }
        }
    }
}

void UserModel::EnableStreamOut()
{
    m_StreamOutEnable = true;
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();
        for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
        {
            nw::g3d::ShaderSelector* pShaderSelector =
                pUserShape->GetShaderSelector(static_cast<PassType>(idxPass));
            if (pShaderSelector == NULL)
            {
                continue;
            }
            // ストリームアウトでワールド座標に変換済みなので、
            // ワールド座標を入力とするバリエーションを選択します。
            int idxOption = pShaderSelector->GetDynamicOptionIndex("coord");
            if (idxOption >= 0)
            {
                const nw::g3d::ResShaderOption* pOption =
                    pShaderSelector->GetDynamicOption(idxOption);
                int idxChoice = pOption->GetChoiceIndex("c_World");
                pShaderSelector->WriteDynamicKey(idxOption, idxChoice);
            }
        }
    }
}

void UserModel::DisableStreamOut()
{
    m_StreamOutEnable = false;
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();
        for (int idxPass = 0; idxPass < NUM_PASS; ++idxPass)
        {
            nw::g3d::ShaderSelector* pShaderSelector =
                pUserShape->GetShaderSelector(static_cast<PassType>(idxPass));
            if (pShaderSelector == NULL)
            {
                continue;
            }
            // ストリームアウトによるワールド座標変換が無効になるので、
            // ローカル座標を入力とするバリエーションを選択します。
            int idxOption = pShaderSelector->GetDynamicOptionIndex("coord");
            if (idxOption >= 0)
            {
                const nw::g3d::ResShaderOption* pOption =
                    pShaderSelector->GetDynamicOption(idxOption);
                int idxChoice = pOption->GetChoiceIndex("c_Local");
                pShaderSelector->WriteDynamicKey(idxOption, idxChoice);
            }
        }
    }
}

void UserModel::AttachToEdit()
{
#if NW_G3D_CONFIG_USE_HOSTIO
    nw::g3d::edit::EditServer::AttachModelArg arg;
    arg.modelObj = m_pModelObj;
    nw::g3d::edit::EditServer::Instance().AttachModel(arg);
#endif
}

void UserModel::DetachFromEdit()
{
#if NW_G3D_CONFIG_USE_HOSTIO
    nw::g3d::edit::EditServer::DetachModelArg arg;
    arg.modelObj = m_pModelObj;
    nw::g3d::edit::EditServer::Instance().DetachModel(arg);
#endif
}

bool UserModel::IsAttachedToEdit() const
{
#if NW_G3D_CONFIG_USE_HOSTIO
    return nw::g3d::edit::EditServer::Instance().HasModelObj(m_pModelObj);
#else
    return false;
#endif
}

bool UserModel::IsAnimKeyValid(bit32 key)
{
    int animType = GetAnimType(key);
    int animIndex = GetAnimIndex(key);
    switch (animType)
    {
    case ANIM_TYPE_SKELETAL: return animIndex < NUM_SKELETAL_ANIM;
    case ANIM_TYPE_VISIBILITY: return animIndex < NUM_VISIBILITY_ANIM;
    case ANIM_TYPE_SHADERPARAM: return animIndex < NUM_SHADERPARAM_ANIM;
    case ANIM_TYPE_TEXPATTERN: return animIndex < NUM_TEXPATTERN_ANIM;
    }
    return false;
}

bit32 UserModel::FindAnim(void* pResAnim) const
{
    nw::g3d::BinaryBlockHeader* pHeader = static_cast<nw::g3d::BinaryBlockHeader*>(pResAnim);
    switch (pHeader->sigWord)
    {
    case nw::g3d::ResSkeletalAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
        {
            nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
            if (pAnimObj->GetResource() == pResAnim)
            {
                return ANIM_TYPE_SKELETAL | idxAnim;
            }
        }
        break;
    case nw::g3d::ResVisibilityAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
        {
            nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[idxAnim];
            if (pAnimObj->GetResource() == pResAnim)
            {
                return ANIM_TYPE_VISIBILITY | idxAnim;
            }
        }
        break;
    case nw::g3d::ResShaderParamAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
        {
            nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[idxAnim];
            if (pAnimObj->GetResource() == pResAnim)
            {
                return ANIM_TYPE_SHADERPARAM | idxAnim;
            }
        }
        break;
    case nw::g3d::ResTexPatternAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
        {
            nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[idxAnim];
            if (pAnimObj->GetResource() == pResAnim)
            {
                return ANIM_TYPE_TEXPATTERN | idxAnim;
            }
        }
        break;
    case nw::g3d::ResShapeAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
        {
            nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[idxAnim];
            if (pAnimObj->GetResource() == pResAnim)
            {
                return ANIM_TYPE_SHAPE | idxAnim;
            }
        }
        break;
    }
    return 0;
}

bit32 UserModel::AddAnim(void* pResAnim)
{
    nw::g3d::BinaryBlockHeader* pHeader = static_cast<nw::g3d::BinaryBlockHeader*>(pResAnim);
    switch (pHeader->sigWord)
    {
    case nw::g3d::ResSkeletalAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
        {
            nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
            if (pAnimObj->GetResource() == NULL || pAnimObj->GetResource() == pResAnim)
            {
                pAnimObj->SetResource(static_cast<nw::g3d::ResSkeletalAnim*>(pResAnim));
                pAnimObj->Bind(m_pModelObj);
                m_Weights[idxAnim] = 1.0f;
                return MakeAnimKey(ANIM_TYPE_SKELETAL, idxAnim);
            }
        }
        break;
    case nw::g3d::ResVisibilityAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
        {
            nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[idxAnim];
            if (pAnimObj->GetResource() == NULL || pAnimObj->GetResource() == pResAnim)
            {
                pAnimObj->SetResource(static_cast<nw::g3d::ResVisibilityAnim*>(pResAnim));
                pAnimObj->Bind(m_pModelObj);
                return MakeAnimKey(ANIM_TYPE_VISIBILITY, idxAnim);
            }
        }
        break;
    case nw::g3d::ResShaderParamAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
        {
            nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[idxAnim];
            if (pAnimObj->GetResource() == NULL || pAnimObj->GetResource() == pResAnim)
            {
                pAnimObj->SetResource(static_cast<nw::g3d::ResShaderParamAnim*>(pResAnim));
                pAnimObj->Bind(m_pModelObj);
                return MakeAnimKey(ANIM_TYPE_SHADERPARAM, idxAnim);
            }
        }
        break;
    case nw::g3d::ResTexPatternAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
        {
            nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[idxAnim];
            if (pAnimObj->GetResource() == NULL || pAnimObj->GetResource() == pResAnim)
            {
                pAnimObj->SetResource(static_cast<nw::g3d::ResTexPatternAnim*>(pResAnim));
                pAnimObj->Bind(m_pModelObj);
                return MakeAnimKey(ANIM_TYPE_TEXPATTERN, idxAnim);
            }
        }
        break;
    case nw::g3d::ResShapeAnim::SIGNATURE:
        for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
        {
            nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[idxAnim];
            if (pAnimObj->GetResource() == NULL || pAnimObj->GetResource() == pResAnim)
            {
                pAnimObj->SetResource(static_cast<nw::g3d::ResShapeAnim*>(pResAnim));
                pAnimObj->Bind(m_pModelObj);
                return MakeAnimKey(ANIM_TYPE_SHAPE, idxAnim);
            }
        }
        break;
    }
    return 0;
}

void UserModel::RemoveAnim(bit32 key)
{
    NW_G3D_ASSERT(IsAnimKeyValid(key));
    switch (GetAnimType(key))
    {
    case ANIM_TYPE_SKELETAL:
        {
            int idxAnim = GetAnimIndex(key);
            nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
            if (pAnimObj->GetResource())
            {
                pAnimObj->ResetResource();
                m_Weights[idxAnim] = 0.0f;
            }
            break;
        }
    case ANIM_TYPE_VISIBILITY:
        {
            nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[GetAnimIndex(key)];
            if (pAnimObj->GetResource())
            {
                pAnimObj->ResetResource();
            }
            break;
        }
    case ANIM_TYPE_SHADERPARAM:
        {
            nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[GetAnimIndex(key)];
            if (pAnimObj->GetResource())
            {
                pAnimObj->ResetResource();
            }
            break;
        }
    case ANIM_TYPE_TEXPATTERN:
        {
            nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[GetAnimIndex(key)];
            if (pAnimObj->GetResource())
            {
                pAnimObj->ResetResource();
            }
            break;
        }
    case ANIM_TYPE_SHAPE:
        {
            nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[GetAnimIndex(key)];
            if (pAnimObj->GetResource())
            {
                pAnimObj->ResetResource();
            }
            break;
        }
    }
}

void UserModel::SetWeight(bit32 key, float weight)
{
    NW_G3D_ASSERT(IsAnimKeyValid(key));
    if (GetAnimType(key) == ANIM_TYPE_SKELETAL)
    {
        m_Weights[GetAnimIndex(key)] = weight;
    }
}

float UserModel::GetWeight(bit32 key) const
{
    NW_G3D_ASSERT(IsAnimKeyValid(key));
    return GetAnimType(key) == ANIM_TYPE_SKELETAL ? m_Weights[GetAnimIndex(key)] : 1.0f;
}

nw::g3d::ModelAnimObj* UserModel::GetAnimObj(bit32 key)
{
    NW_G3D_ASSERT(IsAnimKeyValid(key));
    switch (GetAnimType(key))
    {
    case ANIM_TYPE_SKELETAL:    return m_pSkeletalAnims[GetAnimIndex(key)];
    case ANIM_TYPE_VISIBILITY:  return m_pVisibilityAnims[GetAnimIndex(key)];
    case ANIM_TYPE_SHADERPARAM: return m_pShaderParamAnims[GetAnimIndex(key)];
    case ANIM_TYPE_TEXPATTERN:  return m_pTexPatternAnims[GetAnimIndex(key)];
    case ANIM_TYPE_SHAPE:       return m_pShapeAnims[GetAnimIndex(key)];
    }
    return NULL;
}

void UserModel::UpdateFrame()
{
    for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
    {
        nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
    {
        nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
    {
        nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
    {
        nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().UpdateFrame();
        }
    }
}


void UserModel::SetFrame(float frame)
{
    for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
    {
        nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
    {
        nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
    {
        nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
    {
        nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }
}

void UserModel::CalcAnim()
{
    // 各種アニメ。
    for (int idxAnim = 0; idxAnim < NUM_VISIBILITY_ANIM; ++idxAnim)
    {
        nw::g3d::VisibilityAnimObj* pAnimObj = m_pVisibilityAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->VisibilityAnimObj::Calc();
            pAnimObj->VisibilityAnimObj::ApplyTo(m_pModelObj);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHADERPARAM_ANIM; ++idxAnim)
    {
        nw::g3d::ShaderParamAnimObj* pAnimObj = m_pShaderParamAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->ShaderParamAnimObj::Calc();
            pAnimObj->ShaderParamAnimObj::ApplyTo(m_pModelObj);
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_TEXPATTERN_ANIM; ++idxAnim)
    {
        nw::g3d::TexPatternAnimObj* pAnimObj = m_pTexPatternAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->TexPatternAnimObj::Calc();
            pAnimObj->TexPatternAnimObj::ApplyTo(m_pModelObj);
        }
    }
    // シェイプアニメーション適用前にブレンドウェイトをクリアします。
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        if (pShapeObj->HasValidBlendWeight())
        {
            pShapeObj->ClearBlendWeights();
        }
    }
    for (int idxAnim = 0; idxAnim < NUM_SHAPE_ANIM; ++idxAnim)
    {
        nw::g3d::ShapeAnimObj* pAnimObj = m_pShapeAnims[idxAnim];
        if (pAnimObj->GetResource())
        {
            pAnimObj->ShapeAnimObj::Calc();
            pAnimObj->ShapeAnimObj::ApplyTo(m_pModelObj);
        }
    }
}

void UserModel::CalcWorld()
{
    // できればロックドキャッシュの確保・解放は全モデル分の処理を挟む形でまとめたい。
    size_t sizeLCAll = 0;
    size_t sizeLC0 = 0;
    size_t sizeLC1 = 0;
    void* pLC0 = NULL;
    void* pLC1 = NULL;
    bool lcdmaEnabled = false;
    if (m_LockedCacheEnable)
    {
        // 他のスレッドが別の箇所で LCDMA を使えるように必要な範囲で LCDMA を有効化します。
        lcdmaEnabled = nw::g3d::LockedCache::EnableDMA();
        if (lcdmaEnabled)
        {
            // LCDMA が使える場合のみ LockedCache を確保します。
            NW_G3D_ASSERT(nw::g3d::LockedCache::IsDMAEnabled());
            sizeLCAll = nw::g3d::LockedCache::GetAllocatableSize();
            sizeLC0 = nw::g3d::Align(sizeLCAll >> 1, LL_CACHE_FETCH_SIZE);
            sizeLC1 = sizeLCAll - sizeLC0;
            pLC0 = nw::g3d::LockedCache::Alloc(sizeLCAll);
            pLC1 = AddOffset(pLC0, sizeLC0);
        }
    }

    nw::g3d::SkeletonObj* pSkeletonObj = m_pModelObj->GetSkeleton();

    // スケルタルアニメ。
    int numSkeletalAnim = 0;
    float weightSum = 0.0f;
    for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
    {
        if (m_Weights[idxAnim] > 0.0f)
        {
            ++numSkeletalAnim;
            weightSum += m_Weights[idxAnim];
        }
    }
    if (numSkeletalAnim == 1)
    {
        for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
        {
            nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
            if (m_Weights[idxAnim] > 0.0f)
            {
                // アニメーションですべてのローカル行列が書き換えられる前提で、メモリから読みません。
                pSkeletonObj->LCMountLocalMtx(pLC0, sizeLC0, false);                // mount0
                pAnimObj->LCMount(pLC1, sizeLC1, true);                             // mount1
                pAnimObj->SkeletalAnimObj::Calc();
                pAnimObj->SkeletalAnimObj::ApplyTo(m_pModelObj);
                // フレームが停止している場合は2回目以降の Calc がスキップされるため
                // そのような場合はメモリへの書き出しが必要です。
                // ワンタイムのアニメーションが終端に達した場合も含みます。
                pAnimObj->LCUnmount(false);                                         // unmount1
            }
        }
    }
    else if (numSkeletalAnim > 1)
    {
        float invSum = nw::g3d::Math::Rcp(weightSum);
        m_SkelBlender.LCMount(pLC1, sizeLC1, false);                                // mount1
        m_SkelBlender.ClearResult();
        for (int idxAnim = 0; idxAnim < NUM_SKELETAL_ANIM; ++idxAnim)
        {
            nw::g3d::SkeletalAnimObj* pAnimObj = m_pSkeletalAnims[idxAnim];
            float weight = m_Weights[idxAnim];
            if (weight > 0.0f)
            {
                pAnimObj->LCMount(pLC0, sizeLC0, true);                             // mount0
                m_SkelBlender.Blend(pAnimObj, weight * invSum);
                // フレームが停止している場合は2回目以降の Calc がスキップされるため
                // そのような場合はメモリへの書き出しが必要です。
                // ワンタイムのアニメーションが終端に達した場合も含みます。
                pAnimObj->LCUnmount(false);                                         // unmount0
            }
        }
        // アニメーションですべてのローカル行列が書き換えられる前提で、メモリから読みません。
        pSkeletonObj->LCMountLocalMtx(pLC0, sizeLC0, false);                        // mount0
        m_SkelBlender.ApplyTo(pSkeletonObj);
        m_SkelBlender.LCUnmount(false);                                             // unmount1
    }
    else
    {
        // アニメーション無しの場合はローカル行列をメモリから読み込みます。
        pSkeletonObj->LCMountLocalMtx(pLC0, sizeLC0, true);                         // mount0
    }

    m_BaseMtx.ScaleBases(m_LayoutMtx, m_Scale);
    if (m_pBindModel && m_BindBoneIndex >= 0)
    {
        // バインド指定されている場合、指定されたボーンのワールド行列を親の行列として乗算します。
        // バインド対象のモデルが先に計算されている必要があります。
        const nw::g3d::SkeletonObj* pBindModelSkeletonObj = m_pBindModel->GetSkeleton();
        const nw::g3d::Mtx34* pWorldMtxArray = pBindModelSkeletonObj->GetWorldMtxArray();
        NW_G3D_ASSERT_INDEX_BOUNDS(m_BindBoneIndex, pBindModelSkeletonObj->GetBoneCount());
        m_BaseMtx.Mul(pWorldMtxArray[m_BindBoneIndex], m_BaseMtx);
    }

#if NW_G3D_CONFIG_USE_HOSTIO
    // LockedCache を活用するためにはスケルタルアニメ計算からワールド行列を連続して行う必要があり、
    // エディットライブラリを外部で挟むタイミングが無いため、ここで呼び出します。
    // ここで適用されたスケルタルアニメは EditServer::CalcAnimations() では適用されません。
    // アニメが登録されていない場合はすぐに処理を抜けます。
    // リアルタイム編集時もなるべくコストを抑えたい場合は、アタッチ中のモデルについてのみよびだします。
    if (m_IsEditCalcSkeletalAnimationEnabled)
    {
        nw::g3d::edit::EditServer::Instance().CalcSkeletalAnimations(m_pModelObj);
    }
#endif

    pSkeletonObj->LCMountWorldMtx(pLC1, sizeLC1, false);                            // mount1
    m_pModelObj->CalcWorld(m_BaseMtx);
    m_pModelObj->CalcBounding();
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        pShapeObj->CalcSubMeshBounding(pSkeletonObj);
    }

    // スクリーンビルボードを使用している場合は CalcView でローカル行列が参照されるので、
    // ここで結果を破棄してはいけません。
    m_pModelObj->GetSkeleton()->LCUnmountLocalMtx(false);                           // unmount0

    bool storeWorld = true;
    if (m_DoubleBufferEnable)
    {
        // ここでは Skeleton と Shape が共に複数バッファ確保されている場合しか考慮していません。

        m_CurBuffer = ++m_CurBuffer % pSkeletonObj->GetBufferingCount();
        UpdateBlendWeight();
        pSkeletonObj->LCMountMtxBlock(pLC0, sizeLC0, false, m_CurBuffer);           // mount0
        m_pModelObj->CalcMtxBlock(m_CurBuffer);
        m_pModelObj->CalcShape(m_CurBuffer);
        pSkeletonObj->LCUnmountMtxBlock(true, m_CurBuffer);                         // unmount0
        if (!m_pModelObj->IsViewDependent())
        {
            // CalcView で使用しないので破棄します。
            storeWorld = false;
        }
    }
    else
    {
        m_CurBuffer = 0;
    }
    m_pModelObj->GetSkeleton()->LCUnmountWorldMtx(storeWorld);                      // unmount1


    if (lcdmaEnabled)
    {
        // LockedCache を確保しているのは LCDMA が有効な場合のみです。
        nw::g3d::LockedCache::Free(pLC0);
        nw::g3d::LockedCache::DisableDMA();
    }
}

void UserModel::CalcGPU()
{
    // LockedCache と無関係な計算。
    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();

        // フェッチシェーダの更新が必要な場合は書き換えます。
        int validBlendCount = pShapeObj->CountValidBlendWeight();
        if (pUserShape->GetStreamOutPass().validBlendCount != validBlendCount)
        {
            pUserShape->GetStreamOutPass().validBlendCount = validBlendCount;
            pUserShape->UpdateShapeAnim();
            pUserShape->UpdateFetchShader();
        }
    }

    // ここでは Material と Skeleton と Shape が共に複数バッファ確保されている場合しか考慮していません。
    if (m_DoubleBufferEnable)
    {
        m_pModelObj->CalcMaterial(m_CurBuffer);
    }
    else
    {
        m_pModelObj->CalcMaterial();

        UpdateBlendWeight();

        // できればロックドキャッシュの確保・解放は全モデル分の処理を挟む形でまとめたい。
        size_t sizeLCAll = 0;
        size_t sizeLC0 = 0;
        size_t sizeLC1 = 0;
        void* pLC0 = NULL;
        void* pLC1 = NULL;
        bool lcdmaEnabled = false;
        if (m_LockedCacheEnable)
        {
            // 他のスレッドが別の箇所で LCDMA を使えるように必要な範囲で LCDMA を有効化します。
            lcdmaEnabled = nw::g3d::LockedCache::EnableDMA();
            if (lcdmaEnabled)
            {
                // LCDMA が使える場合のみ LockedCache を確保します。
                NW_G3D_ASSERT(nw::g3d::LockedCache::IsDMAEnabled());
                sizeLCAll = nw::g3d::LockedCache::GetAllocatableSize();
                sizeLC0 = nw::g3d::Align(sizeLCAll >> 1, LL_CACHE_FETCH_SIZE);
                sizeLC1 = sizeLCAll - sizeLC0;
                pLC0 = nw::g3d::LockedCache::Alloc(sizeLCAll);
                pLC1 = AddOffset(pLC0, sizeLC0);
            }
        }

        nw::g3d::SkeletonObj* pSkeletonObj = m_pModelObj->GetSkeleton();
        pSkeletonObj->LCMountWorldMtx(pLC0, sizeLC0, true);                         // mount0
        pSkeletonObj->LCMountMtxBlock(pLC1, sizeLC1, false);                        // mount1
        m_pModelObj->CalcMtxBlock();
        m_pModelObj->CalcShape();
        pSkeletonObj->LCUnmountWorldMtx(false);                                     // unmount0
        pSkeletonObj->LCUnmountMtxBlock(true);                                      // unmount1

        if (lcdmaEnabled)
        {
            // LockedCache を確保しているのは LCDMA が有効な場合のみです。
            nw::g3d::LockedCache::Free(pLC0);
            nw::g3d::LockedCache::DisableDMA();
        }
    }
}

void UserModel::CalcViewGPU(int viewIndex)
{
    m_pModelObj->CalcView(viewIndex, m_pView->GetViewMtx(viewIndex), m_CurBuffer);
}

void UserModel::DebugDraw(PassType passType, int viewIndex, GX2ShaderMode shaderMode)
{
    NW_G3D_ASSERT_NOT_NULL(m_pView);
    (void)shaderMode;

    const nw::g3d::SkeletonObj* pSkelObj = m_pModelObj->GetSkeleton();
    bool streamout = m_StreamOutEnable && (passType != PASS_DEBUG);

    const nw::g3d::ViewVolume& viewVolume = m_pView->GetViewVolume(viewIndex);

    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        if (!m_pModelObj->IsShapeVisible(idxShape)) // ビジビリティの制御を行います。
        {
            continue;
        }

        const nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        if (IsFrustumCullingEnabled())
        {
            // フラスタムカリング
            if (const nw::g3d::Sphere* pSphere = pShapeObj->GetBounding())
            {
                if (!viewVolume.TestIntersection(*pSphere))
                {
                    continue;
                }
            }
        }

        // マテリアル。
        const nw::g3d::MaterialObj* pMatObj =
            m_pModelObj->GetMaterial(pShapeObj->GetMaterialIndex());
        const nw::g3d::ResMaterial* pResMat = pMatObj->GetResource();
        //const UserMaterial* pUserMat = pMatObj->GetUserPtr<UserMaterial>();
        const MaterialAssign* pMatAssign =pResMat->GetUserPtr<MaterialAssign>();
        const MaterialAssign::Pass& matPass = pMatAssign->GetPass(passType);

        if (passType == PASS_TRANSLUCENT && pResMat->GetRenderState()->GetMode()
            != nw::g3d::ResRenderState::MODE_TRANSLUCENT)
        {
            continue;
        }
        else if (passType == PASS_OPAQUE && pResMat->GetRenderState()->GetMode()
            == nw::g3d::ResRenderState::MODE_TRANSLUCENT)
        {
            continue;
        }

        // シェイプ。
        const nw::g3d::ResShape* pResShape = pShapeObj->GetResource();
        const ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>();
        const ShapeAssign::Pass& shapePass = pShapeAssign->GetPass(passType);
        const UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();

        // シェーダのロード。
        const nw::g3d::ShaderSelector* pShaderSelector = pUserShape->GetShaderSelector(passType);
        if (pShaderSelector == NULL)
        {
            continue;
        }
        const nw::g3d::ShadingModelObj* pShadingModelObj = pShaderSelector->GetShadingModel();
        const nw::g3d::ResShadingModel* pShadingModel = pShadingModelObj->GetResource();
        const nw::g3d::ResShaderProgram* pShaderProgram = pShaderSelector->GetProgram();
        if (pShaderProgram == NULL)
        {
            pShaderProgram = pShaderSelector->GetDefaultProgram();
        }
        NW_G3D_ASSERT_NOT_NULL(pShaderProgram);

#ifndef _WIN32
        GX2ShaderMode newShaderMode = pShaderProgram->GetShaderMode();
        (void)newShaderMode;
        // シェーダモード切替はパイプラインのフラッシュが必要なため、内部でシェーダモードの設定は行いません。
        //nw::g3d::SetShaderMode(newShaderMode);
        NW_G3D_ASSERT(shaderMode == newShaderMode);
#endif
        pShaderProgram->Load();

        // 描画ステートのロード。
        const nw::g3d::ResRenderState* pRenderState = pMatObj->GetResRenderState();
        pRenderState->Load();

        if (passType == PASS_PREZ)
        {
            nw::g3d::GfxColorCtrl colorCtrl;
            colorCtrl.SetColorBufferEnable(GX2_FALSE);
            colorCtrl.Load();
        }
        else if (passType == PASS_SHADOW)
        {
            const UserMaterial* pUserMat = pMatObj->GetUserPtr<UserMaterial>();
            if (!pUserMat->GetRenderShadow())
            {
                continue;
            }
            nw::g3d::GfxPolygonCtrl polygonCtrl = pRenderState->GetPolygonCtrl();
            polygonCtrl.SetCullBack(GX2_FALSE);
            polygonCtrl.SetCullFront(GX2_TRUE);
            polygonCtrl.Load();

            nw::g3d::GfxColorCtrl colorCtrl;
            colorCtrl.SetColorBufferEnable(GX2_FALSE);
            colorCtrl.Load();
        }

        // サンプラーのロード。
        for (int idxSampler = 0, numSampler = pShadingModel->GetSamplerCount();
            idxSampler < numSampler; ++idxSampler)
        {
            if (const nw::g3d::ResSampler* pSampler = matPass.pSamplerArray[idxSampler])
            {
                const nw::g3d::ResTexture* pTexture = pMatObj->GetResTexture(pSampler->GetIndex());
                LoadTextureSampler(pTexture->GetGfxTexture(), pSampler->GetGfxSampler(),
                    pShaderProgram, idxSampler);
            }
            else if (m_pEnvModel)
            {
                const nw::g3d::MaterialObj* pMatEnv = m_pEnvModel->GetMaterial(0);
                if (const UserShadingModel* pUserShadingModel =
                    pShadingModel->GetUserPtr<UserShadingModel>())
                {
                    int idxEnvSampler = pUserShadingModel->GetEnvSamplerIndices()[idxSampler];
                    if (idxEnvSampler >= 0)
                    {
                        const nw::g3d::ResSampler* pEnvSampler =
                            pMatEnv->GetResSampler(idxEnvSampler);
                        const nw::g3d::ResTexture* pEnvTexture =
                            pMatEnv->GetResTexture(pEnvSampler->GetIndex());
                        LoadTextureSampler(pEnvTexture->GetGfxTexture(), pEnvSampler->GetGfxSampler(),
                            pShaderProgram, idxSampler);
                    }
                }
            }
            else
            {
                // TODO: ダミーテクスチャを割り当てる？
            }
        }

        // UniformBlock のロード。

        // オプション。
        int idxOptionBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_OPTION);
        const nw::g3d::GfxBuffer& optionBlock = pShadingModelObj->GetOptionBlock();
        if (idxOptionBlock >= 0 &&  optionBlock.GetSize() > 0)
        {
            LoadUniformBlock(&optionBlock, pShaderProgram, idxOptionBlock);
        }

        // ビュー。
        const nw::g3d::GfxBuffer& viewBlock = m_pView->GetViewBlock(viewIndex);
        if (matPass.blockIndexArray[BLOCK_VIEW] >= 0 && viewBlock.GetSize() > 0)
        {
            LoadUniformBlock(&viewBlock, pShaderProgram, matPass.blockIndexArray[BLOCK_VIEW]);
        }

        // 環境モデル。
        if (matPass.blockIndexArray[BLOCK_ENV] >= 0 && m_pEnvModel)
        {
            const nw::g3d::GfxBuffer& envBlock = m_pEnvModel->GetMaterial(0)->GetMatBlock();
            LoadUniformBlock(&envBlock, pShaderProgram, matPass.blockIndexArray[BLOCK_ENV]);
        }

        // コンテキスト
        if (matPass.blockIndexArray[BLOCK_CONTEXT] >= 0 && m_pContext)
        {
            const nw::g3d::GfxBuffer& contextBlock = m_pContext->GetContextBlock();
            LoadUniformBlock(&contextBlock, pShaderProgram, matPass.blockIndexArray[BLOCK_CONTEXT]);
        }

        // アウトラインモデル。
        if (passType == PASS_OUTLINE)
        {
            if (matPass.blockIndexArray[BLOCK_OUTLINE] >= 0 && m_pOutlineModel)
            {
                const nw::g3d::MaterialObj* pOutlineMaterial = m_pOutlineModel->GetMaterial(0);
                pOutlineMaterial->GetResRenderState()->Load();

                LoadTextureSampler(
                    pOutlineMaterial->GetResTexture(0)->GetGfxTexture(),
                    pOutlineMaterial->GetResSampler(0)->GetGfxSampler(),
                    pShaderProgram, 0);

                const nw::g3d::GfxBuffer& outlineBlock = pOutlineMaterial->GetMatBlock();
                if (outlineBlock.GetSize() > 0)
                {
                    LoadUniformBlock(
                        &outlineBlock, pShaderProgram, matPass.blockIndexArray[BLOCK_OUTLINE]);
                }
            }
        }

        // マテリアル。
        int idxMatBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_MATERIAL);
        const nw::g3d::GfxBuffer& matBlock = pMatObj->GetMatBlock();
        if (idxMatBlock >= 0 && matBlock.GetSize() > 0)
        {
            LoadUniformBlock(&matBlock, pShaderProgram, idxMatBlock);
        }

        // スケルトン。
        int idxSkelBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_SKELETON);
        const nw::g3d::GfxBuffer& skelBlock = pSkelObj->GetMtxBlock();
        if (idxSkelBlock >= 0 && skelBlock.GetSize() > 0)
        {
            LoadUniformBlock(&skelBlock, pShaderProgram, idxSkelBlock, m_CurBuffer);
        }

        // シェイプ。
        int idxShapeBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_SHAPE);
        const nw::g3d::GfxBuffer& shapeBlock = pShapeObj->GetShpBlock(viewIndex);
        if (idxShapeBlock >= 0 && shapeBlock.GetSize() > 0)
        {
            LoadUniformBlock(&shapeBlock, pShaderProgram, idxShapeBlock, m_CurBuffer);
        }

        // フェッチシェーダと頂点バッファのロード。
        if (streamout && !pShapeObj->IsViewDependent())
        {
            shapePass.fetchShaderSO.Load();
            shapePass.fetchShaderSO.ReplaceVertexBuffer(
                shapePass.vtxBufferCount, &pUserShape->GetStreamOutPass().streamOutBuffer);
            pUserShape->GetStreamOutPass().streamOutBuffer.LoadVertices(shapePass.vtxBufferCount);
        }
        else
        {
            shapePass.fetchShader.Load();
        }
        for (int idxBuffer = 0, numBuffer = shapePass.vtxBufferCount;
            idxBuffer < numBuffer; ++idxBuffer)
        {
            shapePass.pVtxBufferArray[idxBuffer]->GetGfxBuffer()->LoadVertices(idxBuffer);
        }

        int maxLODLevel = pShapeObj->GetMeshCount() - 1;
        int meshIndex = (m_LODLevel[viewIndex] < maxLODLevel) ? m_LODLevel[viewIndex] : maxLODLevel;
        const nw::g3d::ResMesh* pMesh = pShapeObj->GetResMesh(meshIndex);
        // サブメッシュカリングの判定をしながら描画。
        if (IsFrustumCullingEnabled() && pShapeObj->GetResource()->GetSubMeshCount() > 1)
        {
            nw::g3d::ICalcLodLevelFunctor* pCalcLodLevel = m_pCalcLodLevel[viewIndex];
            if (pCalcLodLevel)
            {
                nw::g3d::CullingContext ctx;
                while (pShapeObj->TestSubMeshLodIntersection(&ctx, viewVolume, *pCalcLodLevel))
                {
                    int lodLevel = (ctx.submeshLodLevel < maxLODLevel) ? ctx.submeshLodLevel : maxLODLevel;
                    pShapeObj->GetResMesh(lodLevel)->DrawSubMesh(ctx.submeshIndex, ctx.submeshCount);
                }
            }
            else
            {
                nw::g3d::CullingContext ctx;
                while (pShapeObj->TestSubMeshIntersection(&ctx, viewVolume) )
                {
                    pMesh->DrawSubMesh(ctx.submeshIndex, ctx.submeshCount);
                }
            }
        }
        else
        {
            pMesh->Draw();
        }
    }
}

void UserModel::StreamOut(GX2ShaderMode shaderMode)
{
    (void)shaderMode;
    nw::g3d::SkeletonObj* pSkelObj = m_pModelObj->GetSkeleton();

    for (int idxShape = 0, numShape = m_pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        if (!m_pModelObj->IsShapeVisible(idxShape)) // ビジビリティの制御を行います。
        {
            continue;
        }

        nw::g3d::ShapeObj* pShapeObj = m_pModelObj->GetShape(idxShape);
        if (pShapeObj->IsViewDependent()) // ビューに依存するシェイプはストリームアウトできない。
        {
            continue;
        }

        // シェイプ。
        const nw::g3d::ResVertex* pResVertex = pShapeObj->GetResVertex();
        UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>();

        // シェーダのロード。
        UserShape::StreamOutPass& pass = pUserShape->GetStreamOutPass();
        const nw::g3d::ShaderSelector* pShaderSelector = pass.pShaderSelector;
        if (pShaderSelector == NULL)
        {
            continue;
        }
        const nw::g3d::ShadingModelObj* pShadingModelObj = pShaderSelector->GetShadingModel();
        const nw::g3d::ResShadingModel* pShadingModel = pShadingModelObj->GetResource();
        const nw::g3d::ResShaderProgram* pShaderProgram = pShaderSelector->GetProgram();
        if (pShaderProgram == NULL)
        {
            pShaderProgram = pShaderSelector->GetDefaultProgram();
        }
        NW_G3D_ASSERT_NOT_NULL(pShaderProgram);

#ifndef _WIN32
        GX2ShaderMode newShaderMode = pShaderProgram->GetShaderMode();
        (void)newShaderMode;
        // シェーダモード切替はパイプラインのフラッシュが必要なため、内部でシェーダモードの設定は行いません。
        //nw::g3d::SetShaderMode(newShaderMode);
        NW_G3D_ASSERT(shaderMode == newShaderMode);
#endif
        pShaderProgram->Load();

        // UniformBlock のロード。

        // スケルトン。
        int idxSkelBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_SKELETON);
        const nw::g3d::GfxBuffer& skelBlock = pSkelObj->GetMtxBlock();
        if (idxSkelBlock >= 0 && skelBlock.GetSize() > 0)
        {
            LoadUniformBlock(&skelBlock, pShaderProgram, idxSkelBlock, m_CurBuffer);
        }

        // シェイプ。
        int idxShapeBlock = pShadingModel->GetSystemBlockIndex(ResUniformBlockVar::TYPE_SHAPE);
        const nw::g3d::GfxBuffer& shapeBlock = pShapeObj->GetShpBlock(0);
        if (idxShapeBlock >= 0 && shapeBlock.GetSize() > 0)
        {
            LoadUniformBlock(&shapeBlock, pShaderProgram, idxShapeBlock, m_CurBuffer);
        }

        // フェッチシェーダと頂点バッファのロード。
        UserShape::StreamOutPass& streamOutPass = pUserShape->GetStreamOutPass();
        streamOutPass.fetchShader.Load();
        for (int idxBuffer = 0, numBuffer = streamOutPass.vtxBufferCount;
            idxBuffer < numBuffer; ++idxBuffer)
        {
            int bufferSlot = streamOutPass.bufferSlotArray[idxBuffer];
            streamOutPass.pVtxBufferArray[idxBuffer]->GetGfxBuffer()->LoadVertices(bufferSlot);
        }

        // 描画。
        // 1頂点毎に1回計算するため、インデックスを使わない頂点を入力します。
        streamOutPass.streamOut.Load();
        streamOutPass.streamOutBuffer.LoadStreamOutBuffer(0);
        streamOutPass.streamOut.Begin();
#ifdef _WIN32
        glDrawArrays(GL_POINTS, 0, pResVertex->GetCount());
        NW_G3D_GL_ASSERT();
#else
        GX2Draw(GX2_PRIMITIVE_POINTS, pResVertex->GetCount());
#endif
        streamOutPass.streamOut.End();
    }
}

//--------------------------------------------------------------------------------------------------

UserShadingModel::UserShadingModel(ResShadingModel* pResShadingModel,
                                   const nw::g3d::ModelObj* pEnvModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResShadingModel);

    m_pResShadingModel = pResShadingModel;

    const MaterialObj* pEnvMat = pEnvModel->GetMaterial(0);
    for (int idxSampler = 0, numSampler = (std::min)(pResShadingModel->GetSamplerCount(),
        static_cast<int>(MAX_SAMPLER)); idxSampler < numSampler; ++idxSampler)
    {
        const char* pSamplerName = pResShadingModel->GetSamplerName(idxSampler);
        m_EnvSamplerIndices[idxSampler] = pEnvMat->GetSamplerIndex(pSamplerName);
    }
}

}}} // namespace nw::g3d::demo
