﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_DEMO_DEMOUTILITY_H_
#define NW_G3D_DEMO_DEMOUTILITY_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/edit/g3d_IAllocator.h>

#ifdef _WIN32
#include <cstdio>
#else
#include <cafe/demo.h>
#endif

#include <cafe/gx2/gx2Enum.h>

namespace nw { namespace g3d { namespace fnd {

class GfxColorBuffer;
class GfxContext;

}}} // namespace nw::g3d::fnd

namespace nw { namespace g3d { namespace math {

class Mtx34;
class Vec3;

}}} // namespace nw::g3d::math

namespace nw { namespace g3d { namespace demo {

void SetCurrentDir();

struct InitArg
{
    InitArg() : hRC(NULL) {}

    void* hRC;
};

void InitLauncher();
void Init(const InitArg& arg = InitArg());
void Shutdown();
bool ProcessMsg();
void PostQuitMsg();

//--------------------------------------------------------------------------------------------------
// Memory

enum
{
    DEFAULT_ALIGNMENT = 4
};

void* AllocMem1(size_t size, size_t alignment = DEFAULT_ALIGNMENT);

template <typename T>
inline
T* AllocMem1(size_t size, size_t alignment = DEFAULT_ALIGNMENT)
{
    return static_cast<T*>(AllocMem1(size, alignment));
}

void FreeMem1(void* ptr);

void* AllocMem2(size_t size, size_t alignment = DEFAULT_ALIGNMENT);

template <typename T>
inline
T* AllocMem2(size_t size, size_t alignment = DEFAULT_ALIGNMENT)
{
    return static_cast<T*>(AllocMem2(size, alignment));
}

void FreeMem2(void* ptr);

typedef void* (*FuncAlloc)(size_t size, size_t alignment);
typedef void (*FuncFree)(void* ptr);

#if NW_G3D_CONFIG_USE_HOSTIO

class EditAllocator : public nw::g3d::edit::IAllocator
{
public:
    virtual void* Alloc(size_t size, u32 alignment)
    {
        return AllocMem2(size, alignment);
    }
    virtual void Free(void* ptr)
    {
        FreeMem2(ptr);
    }
};

#endif // NW_G3D_CONFIG_USE_HOSTIO

//--------------------------------------------------------------------------------------------------
// Pad

class Pad
{
public:
    enum
    {
        BUTTON_LEFT             = 0x00000001,
        BUTTON_RIGHT            = 0x00000002,
        BUTTON_DOWN             = 0x00000004,
        BUTTON_UP               = 0x00000008,
        TRIGGER_Z               = 0x00000010,
        TRIGGER_R               = 0x00000020,
        TRIGGER_L               = 0x00000040,
        BUTTON_A                = 0x00000100,
        BUTTON_B                = 0x00000200,
        BUTTON_X                = 0x00000400,
        BUTTON_Y                = 0x00000800,
        BUTTON_START            = 0x00001000,
        STICK_R_LEFT            = 0x00010000,
        STICK_R_RIGHT           = 0x00020000,
        STICK_R_DOWN            = 0x00040000,
        STICK_R_UP              = 0x00080000,
        STICK_L_LEFT            = 0x00100000,
        STICK_L_RIGHT           = 0x00200000,
        STICK_L_DOWN            = 0x00400000,
        STICK_L_UP              = 0x00800000,
        BUTTON_STICK_L_LEFT     = BUTTON_LEFT | STICK_L_LEFT,
        BUTTON_STICK_L_RIGHT    = BUTTON_RIGHT | STICK_L_RIGHT,
        BUTTON_STICK_L_DOWN     = BUTTON_DOWN | STICK_L_DOWN,
        BUTTON_STICK_L_UP       = BUTTON_UP | STICK_L_UP
    };

    struct Stick
    {
        s8 leftX, leftY, rightX, rightY;
    };
    struct AnalogStick
    {
        float leftX, leftY, rightX, rightY;
    };

    Pad() {}

    bool Reset();
    bool Read();

    bool IsHold(bit32 mask) const { return 0 != (m_Button & mask); }
    bool IsTriggered(bit32 mask) const { return 0 != (m_Triggered & mask); }
    bool IsReleased(bit32 mask) const { return 0 != (m_Released & mask); }
    bool IsHoldAll(bit32 mask) const { return mask == (m_Button & mask); }
    bool IsHoldAllAndTriggered(bit32 mask) const { return IsHoldAll(mask) && IsTriggered(mask); }

    bit32 GetButton() const { return m_Button; }
    bit32 GetLastButton() const { return m_LastButton; }
    const Stick& GetStick() const { return m_Stick; }
    const Stick& GetLastStick() const { return m_LastStick; }
    const AnalogStick& GetAnalogStick() const { return m_AnalogStick; }

private:
    bit32 m_Button;
    bit32 m_LastButton;
    Stick m_Stick;
    Stick m_LastStick;
    AnalogStick m_AnalogStick;
    bit32 m_Triggered;
    bit32 m_Released;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(Pad);
};

//--------------------------------------------------------------------------------------------------
// CameraController

void ControllCamera(nw::g3d::Vec3* pos, nw::g3d::Vec3* up, nw::g3d::Vec3* target, const nw::g3d::Mtx34* viewMtx, const Pad* pad);

void LookAtModel(nw::g3d::Vec3* pos, nw::g3d::Vec3* target, const nw::g3d::ModelObj* model);

//--------------------------------------------------------------------------------------------------
// File

#ifdef _WIN32
typedef FILE* FileHandle;
#else
typedef DEMOFSFileInfo FileHandle;
#endif

int OpenFile(FileHandle* pHandle, const char* path, const char* mode);
int CloseFile(FileHandle handle);
int GetFileSize(FileHandle handle, size_t* pSize);
int ReadFile(FileHandle handle, void* pDst, size_t length);
int WriteFile(FileHandle handle, const void* pSrc, size_t length);

void* LoadFile(const char* path, size_t* pSize = NULL, size_t alignment = DEFAULT_ALIGNMENT);
void SaveFile(const char* path, const void* pData, size_t size);

//--------------------------------------------------------------------------------------------------
// Display

struct InitDisplayArg
{
    InitDisplayArg()
        : modeTV(GX2_TV_RENDER_720)
        , formatTV(GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB)
        , modeDRC(GX2_DRC_SINGLE)
        , formatDRC(GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB)
        , swapInterval(1)
    {
    }

    GX2TVRenderMode modeTV;
    GX2SurfaceFormat formatTV;
    GX2DRCMode modeDRC;
    GX2SurfaceFormat formatDRC;
    int swapInterval;
};

void InitDisplay(const InitDisplayArg& arg = InitDisplayArg());
void ShutdownDisplay();

void CopyOut(const nw::g3d::GfxColorBuffer* pColorBuffer, GX2ScanTarget target);
void SwapScanBuffers();

bool ActivateDisplay(GX2ScanTarget target, nw::g3d::GfxContext& ctx); // for internal use only

//--------------------------------------------------------------------------------------------------
// Container

template <typename T>
class Vector
{
public:
    Vector() : m_pBuffer(NULL), m_Size(0), m_Capacity(0) {}
    Vector(void* pBuffer, size_t size) { SetBuffer(pBuffer, size); }

    bool Empty() const { return m_Size == 0; }
    int Size() const { return static_cast<int>(m_Size); }
    int Capacity() const { return static_cast<int>(m_Capacity); }

    void SetBuffer(void* pBuffer, size_t size)
    {
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        NW_G3D_ASSERT(size > 0);
        Clear();
        m_pBuffer = static_cast<T*>(pBuffer);
        m_Size = 0;
        m_Capacity = size / sizeof(T);
    }
    void* GetBuffer() { return m_pBuffer; }
    void ResetBuffer()
    {
        Clear();
        m_pBuffer = NULL;
    }

    T& operator[](int index)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_Size);
        return m_pBuffer[index];
    }

    const T& operator[](int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_Size);
        return m_pBuffer[index];
    }

    void Clear()
    {
        for (int idx = 0, num = static_cast<int>(m_Size); idx < num; ++idx)
        {
            m_pBuffer[idx].~T();
        }
        m_Size = 0;
    }

    T& Front()
    {
        NW_G3D_ASSERT(!Empty());
        return m_pBuffer[0];
    }

    const T& Front() const
    {
        NW_G3D_ASSERT(!Empty());
        return m_pBuffer[0];
    }

    T& Back()
    {
        NW_G3D_ASSERT(!Empty());
        return m_pBuffer[m_Size - 1];
    }

    const T& Back() const
    {
        NW_G3D_ASSERT(!Empty());
        return m_pBuffer[m_Size - 1];
    }

    void PushBack(const T& x)
    {
        NW_G3D_ASSERT(m_Size < m_Capacity);
        new (&m_pBuffer[m_Size++]) T(x);
    }

    void PopBack()
    {
        NW_G3D_ASSERT(m_Size > 0);
        m_pBuffer[--m_Size].~T();
    }

    void Insert(int index, const T& x)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_Size + 1);
        NW_G3D_ASSERT(m_Size < m_Capacity);
        for (int rindex = static_cast<int>(m_Size); rindex > index; --rindex)
        {
            new(&m_pBuffer[rindex]) T(m_pBuffer[rindex - 1]);
            m_pBuffer[rindex - 1].~T();
        }
        new(&m_pBuffer[index]) T(x);
        ++m_Size;
    }

    void Erase(int index)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_Size);
        m_pBuffer[index].~T();
        for (int findex = index; findex < static_cast<int>(m_Size - 1); ++findex)
        {
            new(&m_pBuffer[findex]) T(m_pBuffer[findex + 1]);
            m_pBuffer[findex + 1].~T();
        }
        --m_Size;
    }

    void Swap(Vector<T>& other)
    {
        std::swap(m_pBuffer, other.m_pBuffer);
        std::swap(m_Size, other.m_Size);
        std::swap(m_Capacity, other.m_Capacity);
    }

private:
    T* m_pBuffer;
    size_t m_Size;
    size_t m_Capacity;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(Vector);
};

//--------------------------------------------------------------------------------------------------
// Menu

class CmdMenu
{
public:
    struct Item
    {
        const char* title;
        size_t len;
        int (*pFunc)(int argc, const char* argv[]);
    };

    CmdMenu(const Item* pItems)
        : m_NumItem(0), m_pItems(pItems)
    {
        if (pItems)
        {
            for (const Item* pItem = pItems; pItem->pFunc; ++pItem)
            {
                ++m_NumItem;
            }
        }
    }

    void PrintMenu() const;
    int Run(int item, int argc, const char* argv[]) const;
    int Select() const;

    int Loop(int argc, const char* argv[]) const;

private:
    int m_NumItem;
    const Item* m_pItems;
};

}}} // namespace nw::g3d::demo

#endif // NW_G3D_DEMO_DEMOUTILITY_H_
