﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;

namespace NintendoWare.Spy
{
    internal sealed class Loudness
    {
        public const int ChannelCount = 6; // Fr-L, Fr-R, Re-L, Re-R, Fr-C, LFE

        public const int StepTime = 100; // [msec]
        public const int GatingBlockTime = 400; // [msec]
        public const int ShortTermTime = 3000; // [msec]
        public const int VuTermTime = 300; // msec

        public const int AbsGateValue = -70; // LKFS
        public const int RelGateValue = -10; // LKFS

        public static readonly float[] ChannelWeightArray = new float[ChannelCount] { 1.0f, 1.0f, 1.41f, 1.41f, 1.0f, 0.0f };

        // 48kHz用フィルタ係数
        private static readonly KWeightFilterCoef ShelvingFilterCoef48kHz = new KWeightFilterCoef(-1.69065929318241f, 0.73248077421585f, 1.53512485958697f, -2.69169618940638f, 1.19839281085285f);
        private static readonly KWeightFilterCoef HighPassFilterCoef48kHz = new KWeightFilterCoef(-1.99004745483398f, 0.99007225036621f, 1.0f, -2.0f, 1.0f);

        // 32kHz用フィルタ係数
        private static readonly KWeightFilterCoef ShelvingFilterCoef32kHz = new KWeightFilterCoef(-1.53883606126270f, 0.62679865465060f, 1.51120520480243f, -2.46461515884083f, 1.04137254742629f);
        private static readonly KWeightFilterCoef HighPassFilterCoef32kHz = new KWeightFilterCoef(-1.98508966895223f, 0.98514532063474f, 1.0f, -2.0f, 1.0f);

        /// <summary>
        /// 48kHz 用の特性値です。
        /// </summary>
        public static readonly Parameter Parameter48kHz = new Parameter(48000, ShelvingFilterCoef48kHz, HighPassFilterCoef48kHz);

        /// <summary>
        /// 32kHz 用の特性値です。
        /// </summary>
        public static readonly Parameter Parameter32kHz = new Parameter(32000, ShelvingFilterCoef32kHz, HighPassFilterCoef32kHz);

        /// <summary>
        /// ラウドネス計算が可能なサンプルレートと、そのサンプルレートで使用する特性値の辞書です。
        /// </summary>
        public static readonly Dictionary<int, Parameter> Parameters = new Dictionary<int, Parameter>()
        {
            { 48000, Parameter48kHz },
            { 32000, Parameter32kHz },
        };

        public class KWeightFilterCoef
        {
            public float A1 { get; private set; }
            public float A2 { get; private set; }
            public float B0 { get; private set; }
            public float B1 { get; private set; }
            public float B2 { get; private set; }

            public KWeightFilterCoef(float a1, float a2, float b0, float b1, float b2)
            {
                A1 = a1;
                A2 = a2;
                B0 = b0;
                B1 = b1;
                B2 = b2;
            }
        }

        /// <summary>
        /// ラウドネスの計算に使用する特性値です。
        /// </summary>
        public class Parameter
        {
            public int SampleRate { get; }

            public int StepSampleCount { get; }

            public int GatingBlockSampleCount { get; }

            public int ShortTermSampleCount { get; }

            public int VuTermSampleCount { get; }

            public KWeightFilterCoef ShelvingFilterCoef { get; }

            public KWeightFilterCoef HighPassFilterCoef { get; }

            public Parameter(int sampleRate, KWeightFilterCoef shelvingFilterCoef, KWeightFilterCoef highPassFilterCoef)
            {
                this.SampleRate = sampleRate;
                this.StepSampleCount = sampleRate * StepTime / 1000;
                this.GatingBlockSampleCount = sampleRate * GatingBlockTime / 1000;
                this.ShortTermSampleCount = sampleRate * ShortTermTime / 1000;
                this.VuTermSampleCount = sampleRate * VuTermTime / 1000;
                this.ShelvingFilterCoef = shelvingFilterCoef;
                this.HighPassFilterCoef = highPassFilterCoef;
            }
        }

        public class LoudnessContext
        {
            public KWeightingFilter KFilter1 { get; } = new KWeightingFilter();
            public KWeightingFilter KFilter2 { get; } = new KWeightingFilter();
            public TruePeakCalculator TruePeakCalculator { get; } = new TruePeakCalculator();
        }

        public class KWeightingFilter
        {
            public float History0 { get; set; }
            public float History1 { get; set; }
        }

        /// <summary>
        /// アナログ化されたときの波形のピークをより正確に把握するため、
        /// デジタル波形を48kHzから192kHzにアップコンバートしてピーク値を計算します。
        /// </summary>
        public class TruePeakCalculator
        {
            /// <summary>
            /// 補間係数です。
            /// </summary>
            public const int Phase = 4;

            private const int Order = 48;
            private const int History = Order / Phase;

            // 48kHz -> 192kHz アップコンバージョン (order 48, 4-phase FIR filter) 係数
            // https://www.itu.int/rec/R-REC-BS.1770/
            // Annex 2: Guidelines for accurate measurement of "true-peak" level より
            private static readonly float[,] Coefficients = new float[History, Phase]
            {
                {  0.0017089843750f, -0.0291748046875f, -0.0189208984375f, -0.0083007812500f },
                {  0.0109863281250f,  0.0292968750000f,  0.0330810546875f,  0.0148925781250f },
                { -0.0196533203125f, -0.0517578125000f, -0.0582275390625f, -0.0266113281250f },
                {  0.0332031250000f,  0.0891113281250f,  0.1015625000000f,  0.0476074218750f },
                { -0.0594482421875f, -0.1665039062500f, -0.2003173828125f, -0.1022949218750f },
                {  0.1373291015625f,  0.4650878906250f,  0.7797851562500f,  0.9721679687500f },
                {  0.9721679687500f,  0.7797851562500f,  0.4650878906250f,  0.1373291015625f },
                { -0.1022949218750f, -0.2003173828125f, -0.1665039062500f, -0.0594482421875f },
                {  0.0476074218750f,  0.1015625000000f,  0.0891113281250f,  0.0332031250000f },
                { -0.0266113281250f, -0.0582275390625f, -0.0517578125000f, -0.0196533203125f },
                {  0.0148925781250f,  0.0330810546875f,  0.0292968750000f,  0.0109863281250f },
                { -0.0083007812500f, -0.0189208984375f, -0.0291748046875f,  0.0017089843750f }
            };

            private readonly float[] _history = new float[History];

            /// <summary>
            /// 入力波形の履歴を消去します。
            /// </summary>
            public void Reset()
            {
                _history.Populate(0.0f);
            }

            /// <summary>
            /// 入力波形にサンプル値を１つ追加し、出力波形の最新４サンプルのピーク値を求めます。
            /// </summary>
            /// <param name="value"></param>
            /// <returns></returns>
            public float Calculate(float value)
            {
                this.SetValue(value);
                return this.GetMaxAbsValue();
            }

            /// <summary>
            /// 入力波形にサンプル値を１つ追加します。
            /// </summary>
            /// <param name="value"></param>
            internal void SetValue(float value)
            {
                for (var i = History - 1; i > 0; i--)
                {
                    _history[i] = _history[i - 1];
                }

                _history[0] = value;
            }

            /// <summary>
            /// 出力波形のサンプル値を取得します。
            /// </summary>
            /// <param name="phase"></param>
            /// <returns></returns>
            internal float GetValue(int phase)
            {
                float value = 0.0f;
                for (var i = 0; i < History; i++)
                {
                    value += Coefficients[i, phase] * _history[i];
                }
                return value;
            }

            /// <summary>
            /// 出力波形の最新４サンプルの絶対値の最大を求めます。
            /// </summary>
            /// <returns></returns>
            private float GetMaxAbsValue()
            {
                var v0 = Math.Abs(this.GetValue(0));
                var v1 = Math.Abs(this.GetValue(1));
                var v2 = Math.Abs(this.GetValue(2));
                var v3 = Math.Abs(this.GetValue(3));
                return Math.Max(v0, Math.Max(v1, Math.Max(v2, v3)));
            }
        }

        public static float CalcLoudnessValue(float x)
        {
            return -0.691f + 10 * (float)Math.Log10(x);
        }

        public static float CalcKWeightingFilter(LoudnessContext context, float x, Parameter parameter)
        {
            float kOut = ApplyKWeightingFilter(parameter.ShelvingFilterCoef, context.KFilter1, x);
            kOut = ApplyKWeightingFilter(parameter.HighPassFilterCoef, context.KFilter2, kOut);
            return kOut;
        }

        private static float ApplyKWeightingFilter(KWeightFilterCoef coef, KWeightingFilter filter, float x)
        {
            float in1 = x - filter.History0 * coef.A1 - filter.History1 * coef.A2;
            float out1 = in1 * coef.B0 + filter.History0 * coef.B1 + filter.History1 * coef.B2;

            filter.History1 = filter.History0;
            filter.History0 = in1;

            return out1;
        }

        public static float CalcTruePeak(LoudnessContext context, float x)
        {
            return context.TruePeakCalculator.Calculate(x);
        }
    }

    internal class LoudnessCalculator
    {
        private readonly List<StepBlock> _stepBlockList = new List<StepBlock>();
        private readonly List<LoudnessGatingBlock> _gatingBlockList = new List<LoudnessGatingBlock>();
        private Loudness.Parameter _parameter;

        public float MomentaryLoudnessValue { get; private set; } = float.NegativeInfinity;
        public float ShortTermLoudnessValue { get; private set; } = float.NegativeInfinity;
        public float AbsGatedLoudnessValue { get; private set; } = float.NegativeInfinity;
        public float RelGatedLoudnessValue { get; private set; } = float.NegativeInfinity;
        public float[] RmsValue { get; } = new float[Loudness.ChannelCount].Populate(float.NegativeInfinity);
        public float[] PeakValue { get; } = new float[Loudness.ChannelCount].Populate(float.NegativeInfinity);
        public float[] TruePeakValue { get; } = new float[Loudness.ChannelCount].Populate(float.NegativeInfinity);

        private class StepBlock
        {
            public float KFilterSquare { get; private set; }
            public float[] SampleSquare { get; } = new float[Loudness.ChannelCount];
            public float[] SamplePeak { get; } = new float[Loudness.ChannelCount];
            public float[] SampleTruePeak { get; } = new float[Loudness.ChannelCount];

            public StepBlock(float kFilterSquare, float[] sampleSquare, float[] samplePeak, float[] sampleTruePeak)
            {
                this.KFilterSquare = kFilterSquare;
                sampleSquare?.CopyTo(this.SampleSquare, 0);
                samplePeak?.CopyTo(this.SamplePeak, 0);
                sampleTruePeak?.CopyTo(this.SampleTruePeak, 0);
            }
        }

        private class LoudnessGatingBlock
        {
            public float BlockMeanSquare { get; set; }
            public double AbsGatedTotalMeanSquare { get; set; }
            public int AbsGatedTotalBlockCount { get; set; }
            public float LoudnessValue { get; set; } = float.NegativeInfinity;
        }

        public void SetParameter(Loudness.Parameter parameter)
        {
            _parameter = parameter;
        }

        public void AddStep(float kfilterSquareSum, float[] sampleSquareSum, float[] samplePeak, float[] sampleTruePeak)
        {
            StepBlock stepBlock = new StepBlock(kfilterSquareSum, sampleSquareSum, samplePeak, sampleTruePeak);

            _stepBlockList.Add(stepBlock);

            Update();
        }

        public void AddStep(float kfilterSquareSum, float[] sampleSquareSum, float[] samplePeak)
        {
            this.AddStep(kfilterSquareSum, sampleSquareSum, samplePeak, sampleTruePeak: null);
        }

        private void Update()
        {
            UpdatePeak();

            if (_stepBlockList.Count >= (_parameter.VuTermSampleCount / _parameter.StepSampleCount))
            {
                UpdateVu();
            }
            if (_stepBlockList.Count >= (_parameter.GatingBlockSampleCount / _parameter.StepSampleCount))
            {
                UpdateLoudnessValue();
            }
        }

        private void UpdatePeak()
        {
            StepBlock stepBlock = _stepBlockList[_stepBlockList.Count - 1];

            for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
            {
                double peakValue = stepBlock.SamplePeak[channelIndex];
                if (peakValue == 0)
                {
                    this.PeakValue[channelIndex] = float.NegativeInfinity;
                }
                else
                {
                    this.PeakValue[channelIndex] = 20 * (float)Math.Log10(peakValue);
                }

                double truePeakValue = stepBlock.SampleTruePeak[channelIndex];
                if (truePeakValue == 0)
                {
                    this.TruePeakValue[channelIndex] = float.NegativeInfinity;
                }
                else
                {
                    this.TruePeakValue[channelIndex] = 20 * (float)Math.Log10(truePeakValue);
                }
            }
        }

        private void UpdateVu()
        {
            float[] total = new float[Loudness.ChannelCount];
            int subIndex = _stepBlockList.Count - (_parameter.VuTermSampleCount / _parameter.StepSampleCount);
            while (subIndex < _stepBlockList.Count)
            {
                StepBlock stepBlock = _stepBlockList[subIndex];

                for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                {
                    total[channelIndex] += stepBlock.SampleSquare[channelIndex];
                }
                subIndex++;
            }
            for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
            {
                double rmsValue = Math.Sqrt(total[channelIndex] / _parameter.VuTermSampleCount);
                if (rmsValue == 0)
                {
                    this.RmsValue[channelIndex] = float.NegativeInfinity;
                }
                else
                {
                    this.RmsValue[channelIndex] = 20 * (float)Math.Log10(rmsValue);
                }
            }
        }

        private void UpdateLoudnessValue()
        {
            double totalMeanSquare = 0.0;
            int subIndex = _stepBlockList.Count - (_parameter.GatingBlockSampleCount / _parameter.StepSampleCount);
            while (subIndex < _stepBlockList.Count)
            {
                StepBlock stepBlock = _stepBlockList[subIndex];

                totalMeanSquare += stepBlock.KFilterSquare;
                subIndex++;
            }

            float meanSquare = (float)(totalMeanSquare / _parameter.GatingBlockSampleCount);
            float loudnessValue = Loudness.CalcLoudnessValue(meanSquare);

            LoudnessGatingBlock gblock = new LoudnessGatingBlock();
            gblock.BlockMeanSquare = meanSquare;
            gblock.LoudnessValue = loudnessValue;
            if (_gatingBlockList.Count > 0)
            {
                LoudnessGatingBlock prevBlock = _gatingBlockList[_gatingBlockList.Count - 1];
                gblock.AbsGatedTotalMeanSquare = prevBlock.AbsGatedTotalMeanSquare;
                gblock.AbsGatedTotalBlockCount = prevBlock.AbsGatedTotalBlockCount;
            }
            if (loudnessValue > Loudness.AbsGateValue)
            {
                gblock.AbsGatedTotalMeanSquare += meanSquare;
                gblock.AbsGatedTotalBlockCount++;
            }

            _gatingBlockList.Add(gblock);

            this.MomentaryLoudnessValue = _gatingBlockList[_gatingBlockList.Count - 1].LoudnessValue;

            // Integrated
            if (gblock.AbsGatedTotalBlockCount > 0)
            {
                meanSquare = (float)(gblock.AbsGatedTotalMeanSquare / gblock.AbsGatedTotalBlockCount);
                this.AbsGatedLoudnessValue = Loudness.CalcLoudnessValue(meanSquare);

                float relGatingThreshold = Math.Max(this.AbsGatedLoudnessValue + Loudness.RelGateValue, Loudness.AbsGateValue);

                totalMeanSquare = 0.0;
                int relGatedBlockCount = 0;
                foreach (LoudnessGatingBlock block in _gatingBlockList)
                {
                    if (block.LoudnessValue > relGatingThreshold)
                    {
                        totalMeanSquare += block.BlockMeanSquare;
                        relGatedBlockCount++;
                    }
                }
                if (relGatedBlockCount > 0)
                {
                    meanSquare = (float)(totalMeanSquare / relGatedBlockCount);
                    this.RelGatedLoudnessValue = Loudness.CalcLoudnessValue(meanSquare);
                }
            }

            // Short Term
            if (_stepBlockList.Count >= _parameter.ShortTermSampleCount / _parameter.StepSampleCount)
            {
                float shortTermLoudnessValue = 0.0f;
                int sIndex = _stepBlockList.Count - _parameter.ShortTermSampleCount / _parameter.StepSampleCount;
                while (sIndex < _stepBlockList.Count)
                {
                    StepBlock stepBlock = _stepBlockList[sIndex];

                    shortTermLoudnessValue += stepBlock.KFilterSquare;
                    sIndex++;
                }

                shortTermLoudnessValue /= _parameter.ShortTermSampleCount;
                this.ShortTermLoudnessValue = Loudness.CalcLoudnessValue(shortTermLoudnessValue);
            }
        }
    }
}
