﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.Spy
{
    public static class BinarySearchUtility
    {
        /// <summary>
        /// 検索対象が見つかったと判断する条件です。
        /// </summary>
        public enum MatchConditions
        {
            /// <summary>
            /// 探索中に最初に検索条件にマッチした要素のインデックスを返します。
            /// </summary>
            FirstFound,

            /// <summary>
            /// 検索条件に適合する要素のうち、最も小さいインデックスを返します。
            /// </summary>
            SmallestIndex,

            /// <summary>
            /// 検索条件に適合する要素のうち、最も大きいインデックスを返します。
            /// </summary>
            BiggestIndex,
        }

        public class Options
        {
            /// <summary>
            /// 以下のオプションが指定されます。
            /// <list type="bullet">
            ///   <item>MatchCondition = MatchConditions.FirstFound</item>
            /// </list>
            /// </summary>
            /// </remarks>
            public static readonly Options FirstFound = new Options();

            /// <summary>
            /// 以下のオプションが指定されます。
            /// <list type="bullet">
            ///   <item>MatchCondition = MatchConditions.SmallestIndex</item>
            /// </list>
            /// </summary>
            public static readonly Options SmallestIndex = new Options()
            {
                MatchCondition = MatchConditions.SmallestIndex,
            };

            /// <summary>
            /// 以下のオプションが指定されます。
            /// <list type="bullet">
            ///   <item>MatchCondition = MatchConditions.BiggestIndex</item>
            /// </list>
            /// </summary>
            public static readonly Options BiggestIndex = new Options()
            {
                MatchCondition = MatchConditions.BiggestIndex,
            };

            /// <summary>
            /// 検索アルゴリズムを指定または取得します。
            /// デフォルト値は MatchConditions.FirstFound です。
            /// </summary>
            public MatchConditions MatchCondition { get; set; } = MatchConditions.FirstFound;
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、目標値と選択子の戻り値が比較されます。
        /// リストは選択子に対して昇順にソートされていなければなりません。
        /// 目標値と選択子の戻り値との比較には System.Collections.Generic.Comparer&lt;U&gt;.Default が使われます。
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは未定義です。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<T, U>(IList<T> list, U value, Func<T, U> selector)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            return _BinarySearch_FirstFound<T, U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、目標値と選択子の戻り値が比較されます。
        /// リストは選択子に対して昇順にソートされていなければなりません。
        /// 目標値と選択子の戻り値との比較には System.Collections.Generic.Comparer&lt;U&gt;.Default が使われます。
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは未定義です。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<U>(IList list, U value, Func<object, U> selector)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            return _BinarySearch_FirstFound<U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、比較子により目標値と選択子の戻り値が比較されます。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは未定義です。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<T, U>(IList<T> list, U value, Func<T, U> selector, IComparer<U> comparer)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(comparer);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            return _BinarySearch_FirstFound<T, U>(list, 0, list.Count - 1, value, selector, comparer);
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、比較子により目標値と選択子の戻り値が比較されます。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは未定義です。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<U>(IList list, U value, Func<object, U> selector, IComparer<U> comparer)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(comparer);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            return _BinarySearch_FirstFound<U>(list, 0, list.Count - 1, value, selector, comparer);
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、目標値と選択子の戻り値が比較されます。
        /// リストは選択子に対して昇順にソートされていなければなりません。
        /// 目標値と選択子の戻り値との比較には System.Collections.Generic.Comparer&lt;U&gt;.Default が使われます。
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは引数 options の指定によります。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="options">オプションです。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<T, U>(IList<T> list, U value, Func<T, U> selector, Options options)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(options);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            switch (options.MatchCondition)
            {
                case MatchConditions.FirstFound:
                    return _BinarySearch_FirstFound<T, U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                case MatchConditions.SmallestIndex:
                    return _BinarySearch_SmallestIndex<T, U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                case MatchConditions.BiggestIndex:
                    return _BinarySearch_BiggestIndex<T, U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                default:
                    throw new NotImplementedException("unexpected options.Algorithm");
            }
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、目標値と選択子の戻り値が比較されます。
        /// リストは選択子に対して昇順にソートされていなければなりません。
        /// 目標値と選択子の戻り値との比較には System.Collections.Generic.Comparer&lt;U&gt;.Default が使われます。
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは引数 options の指定によります。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="options">オプションです。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<U>(IList list, U value, Func<object, U> selector, Options options)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(options);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            switch (options.MatchCondition)
            {
                case MatchConditions.FirstFound:
                    return _BinarySearch_FirstFound<U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                case MatchConditions.SmallestIndex:
                    return _BinarySearch_SmallestIndex<U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                case MatchConditions.BiggestIndex:
                    return _BinarySearch_BiggestIndex<U>(list, 0, list.Count - 1, value, selector, Comparer<U>.Default);

                default:
                    throw new NotImplementedException("unexpected options.Algorithm");
            }
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、比較子により目標値と選択子の戻り値が比較されます。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは引数 options の指定によります。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <param name="options">オプションです。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<T, U>(IList<T> list, U value, Func<T, U> selector, IComparer<U> comparer, Options options)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(comparer);
            Ensure.Argument.NotNull(options);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            switch (options.MatchCondition)
            {
                case MatchConditions.FirstFound:
                    return _BinarySearch_FirstFound<T, U>(list, 0, list.Count - 1, value, selector, comparer);

                case MatchConditions.SmallestIndex:
                    return _BinarySearch_SmallestIndex<T, U>(list, 0, list.Count - 1, value, selector, comparer);

                case MatchConditions.BiggestIndex:
                    return _BinarySearch_BiggestIndex<T, U>(list, 0, list.Count - 1, value, selector, comparer);

                default:
                    throw new NotImplementedException("unexpected options.Algorithm");
            }
        }

        /// <summary>
        /// ソート済みリストから要素を検索し、要素のインデックスを返します。
        /// リストの要素に対して選択子が適用され、比較子により目標値と選択子の戻り値が比較されます。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合に、どの要素のインデックスが返されるかは未定義です。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <param name="options">オプションです。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        public static int BinarySearch<U>(IList list, U value, Func<object, U> selector, IComparer<U> comparer, Options options)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(selector);
            Ensure.Argument.NotNull(comparer);
            Ensure.Argument.NotNull(options);

            if (list.Count == 0)
            {
                return ~list.Count;
            }

            switch (options.MatchCondition)
            {
                case MatchConditions.FirstFound:
                    return _BinarySearch_FirstFound<U>(list, 0, list.Count - 1, value, selector, comparer);

                case MatchConditions.SmallestIndex:
                    return _BinarySearch_SmallestIndex<U>(list, 0, list.Count - 1, value, selector, comparer);

                case MatchConditions.BiggestIndex:
                    return _BinarySearch_BiggestIndex<U>(list, 0, list.Count - 1, value, selector, comparer);

                default:
                    throw new NotImplementedException("unexpected options.Algorithm");
            }
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最初に目標値と一致した要素のインデックスが返されます。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_FirstFound<T, U>(IList<T> list, int begin, int end, U value, Func<T, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    return middle;
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最初に目標値と一致した要素のインデックスが返されます。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_FirstFound<U>(IList list, int begin, int end, U value, Func<object, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    return middle;
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最も小さいインデックスが返されます。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_SmallestIndex<T, U>(IList<T> list, int begin, int end, U value, Func<T, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    if (begin == end)
                    {
                        return middle;
                    }
                    else
                    {
                        end = middle;
                    }
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最も小さいインデックスが返されます。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_SmallestIndex<U>(IList list, int begin, int end, U value, Func<object, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    if (begin == end)
                    {
                        return middle;
                    }
                    else
                    {
                        end = middle;
                    }
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最も大きいインデックスが返されます。
        /// </summary>
        /// <typeparam name="T">リストの要素型です。</typeparam>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_BiggestIndex<T, U>(IList<T> list, int begin, int end, U value, Func<T, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end + 1) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    if (begin == end)
                    {
                        return middle;
                    }
                    else
                    {
                        begin = middle;
                    }
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }

        /// <summary>
        /// 比較子を使用して、ソート済みリストから要素の範囲を検索し、その要素の0から始まるインデックスを返します。
        /// リストは選択子と比較子に対して昇順にソートされていなければなりません。
        /// 比較子のCompareメソッドには第一引数に目標値、第二引数にリストの要素が与えられます。比較子は以下の値を返します。
        /// <list type="bullet">
        /// <item>目標値がリストの要素と一致するときは0を返します。</item>
        /// <item>目標値がリストの要素より小さいとき、つまり見つけたい要素がより小さいインデックスにあるときは負の値を返します。</item>
        /// <item>目標値がリストの要素より大きいとき、つまり見つけたい要素がより大きいインデックスにあるときは正の値を返します。</item>
        /// </list>
        /// リストに同じ値の要素がある場合には、最も大きいインデックスが返されます。
        /// </summary>
        /// <typeparam name="U">選択子の戻り値型です。</typeparam>
        /// <param name="list">ソート済みのリストです。</param>
        /// <param name="begin">検索範囲の開始位置です。</param>
        /// <param name="end">検索範囲の終了位置です。検索対象に含まれます。</param>
        /// <param name="value">検索の目標値です。</param>
        /// <param name="selector">選択子です。</param>
        /// <param name="comparer">比較子です。</param>
        /// <returns>
        /// 目標値に一致する要素が見つかったときは、リストの0から始まるインデックスです。
        /// 見つからなかった場合は、次に大きい要素のインデックスのビットごとの補数（負の値）です。
        /// 大きい要素が存在しない場合は list.Count のビットごとの補数（負の値）です。
        /// </returns>
        private static int _BinarySearch_BiggestIndex<U>(IList list, int begin, int end, U value, Func<object, U> selector, IComparer<U> comparer)
        {
            while (begin <= end)
            {
                int middle = (begin + end + 1) / 2;
                var item = list[middle];
                var selected = selector(item);
                var comp = comparer.Compare(value, selected);
                if (comp == 0)
                {
                    // アイテムが見つかった場合。
                    if (begin == end)
                    {
                        return middle;
                    }
                    else
                    {
                        begin = middle;
                    }
                }
                else if (comp > 0)
                {
                    begin = middle + 1;
                }
                else
                {
                    end = middle - 1;
                }
            }

            return ~Math.Min(begin, list.Count);
        }
    }
}
