﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace NintendoWare.Spy
{
    public class PeakDataManager
    {
        // Minimum, Maximumの値を作る最小のサンプル数です
        private const int BaseSamplingCount = 16;

        // Minimum, Maximumの値を作る最大のサンプル数です
        private const int MaxSamplingCount = 16384;

        private readonly Dictionary<long, List<PeakData>> _peakDatas = new Dictionary<long, List<PeakData>>();
        private int _sampleRate = 0;
        private int _channelCount = 0;
        private int _peakDataIndex = 0;
        private long? _baseXIndex = null;

        private PeakData _preservationPeakData = null;
        private int _preservationSampleDataCount = 0;
        private long _currentOverallIndex = 0;

        /// <summary>
        /// 初期化です
        /// </summary>
        public void Initialize(int sampleRate, int channelCount)
        {
            _sampleRate = sampleRate;
            _channelCount = channelCount;
            _baseXIndex = null;

            // 最小サンプル数用の PeakDataを保持するリストを追加します。
            _peakDatas.Clear();
            _peakDatas.Add(BaseSamplingCount, new List<PeakData>());
        }

        /// <summary>
        /// サンプルデータを追加します
        /// </summary>
        public void Add(WaveformSpyModel.Waveform waveform)
        {
            var clearance = waveform.SampleIndex - _currentOverallIndex;
            if (clearance > 0)
            {
                var zeroWaveform = new WaveformSpyModel.Waveform(waveform.SampleRate, waveform.SampleCount);
                var addZeroCount = 0;
                while (addZeroCount < clearance)
                {
                    addZeroCount += this.Add(zeroWaveform, 0);
                }
            }
            _currentOverallIndex = waveform.SampleIndex + waveform.SampleCount;

            var addCount = 0;
            while (addCount < waveform.SampleCount)
            {
                addCount += this.Add(waveform, addCount);
            }
        }

        private int Add(WaveformSpyModel.Waveform waveform, int sampleOffset)
        {
            if (_baseXIndex == null)
            {
                _baseXIndex = (long)(waveform.Time.Seconds * _sampleRate);
            }

            if (_preservationPeakData == null)
            {
                var peakData = new PeakData(_channelCount);
                peakData.Index = _peakDataIndex++;
                peakData.Count = BaseSamplingCount;
                _peakDatas[BaseSamplingCount].Add(peakData);

                _preservationPeakData = peakData;
            }

            var sampleData = waveform.SampleData;
            var completeSampleCount = BaseSamplingCount;

            var sampleCount = waveform.SampleCount - sampleOffset;
            var spaceCount = BaseSamplingCount - _preservationSampleDataCount;
            var copyCount = 0;
            if (spaceCount >= sampleCount)
            {
                copyCount = sampleCount;
            }
            else
            {
                copyCount = spaceCount;
            }

            for (int channel = 0; channel < _channelCount; channel++)
            {
                short min = _preservationPeakData.Minimum[channel];
                short max = _preservationPeakData.Maximum[channel];

                for (var index = sampleOffset; index < sampleOffset + copyCount; index++)
                {
                    var value = sampleData.Samples[channel][index];
                    min = Math.Min(min, value);
                    max = Math.Max(max, value);
                }

                _preservationPeakData.Minimum[channel] = min;
                _preservationPeakData.Maximum[channel] = max;
            }

            _preservationSampleDataCount += copyCount;
            if (_preservationSampleDataCount == completeSampleCount)
            {
                this.UpdateUpperLevelPeakData(_preservationPeakData);

                _preservationPeakData = null;
                _preservationSampleDataCount = 0;
            }

            return copyCount;
        }

        /// <summary>
        /// 指定されたPeakDataから荒い粒度の方向に向かって、最大のサンプル数の PeakDataまで Minimum, Maximumの値を更新します
        /// </summary>
        private void UpdateUpperLevelPeakData(PeakData peakData)
        {
            var upperLevelSamplingSize = peakData.Count * 2;
            if (upperLevelSamplingSize == MaxSamplingCount)
            {
                return;
            }

            var baseIndex = (int)(peakData.Index);
            var upperLevelIndex = baseIndex / 2;
            if (_peakDatas.ContainsKey(upperLevelSamplingSize) == false)
            {
                _peakDatas.Add(upperLevelSamplingSize, new List<PeakData>());
            }

            var upperLevelPeakDatas = _peakDatas[upperLevelSamplingSize];
            if (upperLevelPeakDatas.Count <= upperLevelIndex)
            {
                for (var index = upperLevelPeakDatas.Count; index <= upperLevelIndex; index++)
                {
                    var newPeakData = new PeakData(_channelCount);
                    newPeakData.Index = index;
                    newPeakData.Count = upperLevelSamplingSize;
                    upperLevelPeakDatas.Add(newPeakData);

                    this.UpdatePeakDataMinMax(newPeakData);
                }
            }
            else
            {
                this.UpdatePeakDataMinMax(upperLevelPeakDatas[upperLevelIndex]);
            }

            var basePeakData = upperLevelPeakDatas[upperLevelIndex];
            this.UpdateUpperLevelPeakData(basePeakData);
        }

        /// <summary>
        /// 指定されたPeakDataより粒度の荒いPeakDataを(1 or 2個)使い、Minimum, Maximumの値を更新します
        /// </summary>
        private void UpdatePeakDataMinMax(PeakData peakData)
        {
            var baseIndex = (int)peakData.Index * 2;
            var lowerLevelSamplingSize = peakData.Count / 2;
            var lowerLevelPeakDatas = _peakDatas[lowerLevelSamplingSize];
            var mins = new List<short>();
            var maxs = new List<short>();

            for (int channel = 0; channel < _channelCount; channel++)
            {
                var min0 = lowerLevelPeakDatas[baseIndex + 0].Minimum[channel];
                var max0 = lowerLevelPeakDatas[baseIndex + 0].Maximum[channel];
                var min1 = min0;
                var max1 = max0;
                if (baseIndex + 1 < lowerLevelPeakDatas.Count)
                {
                    min1 = lowerLevelPeakDatas[baseIndex + 1].Minimum[channel];
                    max1 = lowerLevelPeakDatas[baseIndex + 1].Maximum[channel];
                }

                var min = Math.Min(min0, min1);
                var max = Math.Max(max0, max1);
                mins.Add(min);
                maxs.Add(max);
            }

            peakData.Minimum = mins.ToArray();
            peakData.Maximum = maxs.ToArray();
        }

        /// <summary>
        ///  Minimum, Maximumの値を取得します
        /// </summary>
        public MinMaxValue GetMinMaxValue(long position, long count)
        {
            if (count <= 0)
            {
                return null;
            }

            if (_baseXIndex == null)
            {
                return null;
            }

            int samplingSize = BaseSamplingCount;

            while (true)
            {
                if (count <= samplingSize ||
                    _peakDatas.ContainsKey(samplingSize * 2) == false)
                {
                    break;
                }
                samplingSize *= 2;
            }

            var data = _peakDatas[samplingSize];
            var index = (int)((position - _baseXIndex) / samplingSize);
            if (index < 0 || index >= data.Count)
            {
                return null;
            }

            var peakData = data[index];
            var result = new MinMaxValue(_channelCount);
            for (int channel = 0; channel < _channelCount; channel++)
            {
                result.Minimum[channel] = peakData.Minimum[channel];
                result.Maximum[channel] = peakData.Maximum[channel];
            }
            return result;
        }

        /// <summary>
        /// Minimum, Maximumの値を入れる戻り値用のクラスです
        /// </summary>
        public class MinMaxValue
        {
            public MinMaxValue(int channelCount)
            {
                this.Minimum = new short[channelCount];
                this.Maximum = new short[channelCount];
            }

            public short[] Minimum { get; set; }
            public short[] Maximum { get; set; }
        }
    }

    /// <summary>
    /// Minimum, Maximumのデータを保持するクラスです
    /// </summary>
    public class PeakData
    {
        public PeakData(int channelCount)
        {
            this.Minimum = new short[channelCount];
            this.Maximum = new short[channelCount];

            for (var channel = 0; channel < channelCount; channel++)
            {
                this.Minimum[channel] = short.MaxValue;
                this.Maximum[channel] = short.MinValue;
            }
        }

        public short[] Minimum { get; set; }
        public short[] Maximum { get; set; }
        public long Index { get; set; }
        public long Count { get; set; }
    }
}
