﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Communication;
using NintendoWare.Spy.Foundation.Communications;
using NintendoWare.Spy.Plugins;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Spy 機能を提供します。
    /// </summary>
    public class SpyService : ObservableObject
    {
        private const string WorkDirectoryName = "NintendoSDK_Spy_Temporary";
        private const string RecordingDirectoryName = ".spy_recording_";
        private const string LockFileName = ".spy_lock_";
        private const int MaxOldDataDirectoryCount = 5;
        private const int LatestDataTimestampUpdateInterval = 100; // [msec]

        private const string ConnectionMutexName = "NintendoWare.Spy.ConnectionMutex";
        private const string DirectoryMutexName = "NintendoWare.Spy.DirectoryMutex";
        private const string LaunchMutexName = "NintendoWare.Spy.LaunchMutex";

        //-----------------------------------------------------------------

        private readonly SpyModelRegistory _modelRegistory = new SpyModelRegistory();

        private readonly KeyedList<string, SpyModelPlugin> _modelPlugins =
            new KeyedList<string, SpyModelPlugin>(item => item.DataName);

        private readonly SpySession _session = new SpySession();

        private readonly SpyDataPool _currentData;

        // TODO : SpyFileManager に移す候補
        private readonly HashSet<string> _loadedFilePaths = new HashSet<string>();

        private HostIOService _hostIOService;

        private SpyFileManager _fileManager;
        private bool _isFileManagerForRecording;

        private bool _hasData;
        private bool _hasRecordedData;
        private bool _hasLoadedData;
        private bool _isLoading;
        private string _loadedDataDirectoryPath = string.Empty;
        private Task _loaderTask = Task.CompletedTask;
        private CancellationTokenSource _cancellationTokenSource;

        private List<ReceivedDataInfo> _receivedDataCache;
        private DateTime _lastDataReceivedNotificationTime = DateTime.MinValue;
        private TimeSpan _dataReceivedNotificationInterval = TimeSpan.FromMilliseconds(50);
        private SpyGlobalTime _latestDataTimestamp = SpyGlobalTime.InvalidValue;

        private readonly MutexLock _connectionMutex = new MutexLock(ConnectionMutexName, TimeSpan.Zero);
        private readonly MutexLock _directoryMutex = new MutexLock(DirectoryMutexName, TimeSpan.FromSeconds(5));
        private Mutex _launchMutex;

        //-----------------------------------------------------------------

        /// <summary>
        /// サービスの状態を示します。
        /// </summary>
        public enum ServiceState
        {
            NotStarted = SpySession.SessionState.NotStarted,
            Starting = SpySession.SessionState.Starting,
            Stopping = SpySession.SessionState.Stopping,
            Running = SpySession.SessionState.Running,
        }

        private class ReceivedDataInfo
        {
            public uint DataID { get; set; }
            public string DataName { get; set; }
            public Version DataVersion { get; set; }
            public long Timestamp { get; set; }
            public byte[] RawData { get; set; }
            public SpyDataBlock DataBlock { get; set; }
        }

        /// <summary>
        /// LoadFiles() でデータファイルを並行して読み込むのに使用する一時データです。
        /// </summary>
        private class DataFileContent
        {
            public Binary.SpyDataBinaryHeader Header { get; set; }
            public IEnumerator<SpyDataBlock> DataBlocks { get; set; }
            public long Size { get; set; }
            public long LastReadSize { get; set; }
        }

        /// <summary>
        /// 複数の Spy.exe が起動された場合にインスタンス間の相互排除を行います。
        /// </summary>
        private class MutexLock
        {
            private string _mutexName;
            private TimeSpan _timeout;
            private Mutex _mutex;

            public bool IsLocked => _mutex != null;

            public MutexLock(string mutexName, TimeSpan timeout)
            {
                _mutexName = mutexName;
                _timeout = timeout;
            }

            public bool TryLock()
            {
                if (_mutex == null)
                {
                    _mutex = new Mutex(false, _mutexName);
                    if (_mutex.WaitOne(_timeout, false) == false)
                    {
                        _mutex.Close();
                        _mutex = null;
                        return false;
                    }
                }
                return true;
            }

            public void Release()
            {
                if (_mutex != null)
                {
                    _mutex.ReleaseMutex();
                    _mutex.Close();
                    _mutex = null;
                }
            }
        }

        //-----------------------------------------------------------------

        public SpyService()
        {
            _currentData = new SpyDataPool();

            _session.StateChanged += (sender, e) =>
            {
                if (e.NewState == SpySession.SessionState.NotStarted)
                {
                    this.FlushFiles();
                }

                this.NotifyStateChanged();
                this.NotifyPropertyChanged(() => this.State);
            };

            _session.DataReceiveStarted += (sender, e) => UpdateDataVersion();

            _session.DataReceived += (sender, e) => this.HandleNewData(e);

            Directory.CreateDirectory(this.WorkDirectoryPath);

            // 通信が終了しても、データファイルを閉じるまでは受信ディレクトリの名前を変更できないので、
            // 複数の Spy.exe が起動している場合に同時に複数の受信ディレクトリを作れるように、
            // 受信ディレクトリの名前は Spy.exe のインスタンス毎に異なるようにします。
            this.RecordingDirectoryPath = Path.Combine(this.WorkDirectoryPath, RecordingDirectoryName + Guid.NewGuid().ToString());
        }

        private void UpdateDataVersion()
        {
            var dataVersions = this.GetCurrentData().DataVersions;
            dataVersions.ClearData();

            foreach (var item in _session.DataInfosList)
            {
                dataVersions.Add(item.DataName, item.DataVersion);
            }
        }

        //-----------------------------------------------------------------

        public event EventHandler StateChanged = (sender, e) => { };

        public event EventHandler<SpyDataChangedEventArgs> DataChanged = (sender, e) => { };

        /// <summary>
        /// データファイルのロードを開始するときに通知します。
        /// </summary>
        public event EventHandler BeginLoadFiles;

        /// <summary>
        /// データファイルのロードが終了したら通知します。
        /// </summary>
        public event EventHandler EndLoadFiles;

        //-----------------------------------------------------------------

        /// <summary>
        /// サービスの状態を取得します。
        /// </summary>
        public ServiceState State
        {
            get { return (ServiceState)_session.State; }
        }

        /// <summary>
        /// データ受信通知間隔を取得または設定します。
        /// </summary>
        public TimeSpan DataReceivedNotificationInterval
        {
            get { return _dataReceivedNotificationInterval; }
            set { this.SetPropertyValue(ref _dataReceivedNotificationInterval, value); }
        }

        /// <summary>
        /// データに記録された最新の時刻です。
        /// </summary>
        public SpyGlobalTime LatestDataTimestamp
        {
            get { return _latestDataTimestamp; }
            private set { this.SetPropertyValue(ref _latestDataTimestamp, value); }
        }

        /// <summary>
        /// Spy.exe のインスタンスを区別する必要があるときに使います。
        /// </summary>
        private Guid Guid { get; } = Guid.NewGuid();

        /// <summary>
        /// 作業ディレクトリです。
        /// </summary>
        public string WorkDirectoryPath { get; } = Path.GetFullPath(Path.Combine(Path.GetTempPath(), WorkDirectoryName));

        /// <summary>
        /// 受信ディレクトリです。
        /// </summary>
        public string RecordingDirectoryPath { get; }

        /// <summary>
        /// 現在ロードされているデータファイルのディレクトリです。
        /// </summary>
        public string LoadedDataDirectoryPath
        {
            get { return _loadedDataDirectoryPath; }
            private set
            {
                if (this.SetPropertyValue(ref _loadedDataDirectoryPath, value))
                {
                    this.UpdateHasRecordedData();
                    this.UpdateHasLoadedData();
                }
            }
        }

        /// <summary>
        /// 現在、データファイルのロード中であるかを示します。
        /// </summary>
        public bool IsLoading
        {
            get { return _isLoading; }
            private set { this.SetPropertyValue(ref _isLoading, value); }
        }

        /// <summary>
        /// データがある場合 true になります。
        /// </summary>
        public bool HasData
        {
            get { return _hasData; }
            set
            {
                if (this.SetPropertyValue(ref _hasData, value))
                {
                    this.UpdateHasRecordedData();
                    this.UpdateHasLoadedData();
                }
            }
        }

        /// <summary>
        /// 受信したデータがある場合に true になります。
        /// </summary>
        /// <remarks>
        /// 設定は <see cref="UpdateHasRecordedData"/> でします。
        /// </remarks>
        public bool HasRecordedData
        {
            get { return _hasRecordedData; }
        }

        /// <summary>
        /// ロードされたデータがある場合に true になります。
        /// </summary>
        /// <remarks>
        /// 設定は <see cref="UpdateHasLoadedData"/> でします。
        /// </remarks>
        public bool HasLoadedData
        {
            get { return _hasLoadedData; }
        }

        /// <summary>
        /// 複数の Spy.exe が起動された場合に、ターゲットへの排他的な接続権を持つかを取得します。
        /// </summary>
        public bool IsConnectionMutexLocked => _connectionMutex.IsLocked;

        /// <summary>
        /// 複数の Spy.exe が起動された場合に、作業ディレクトリへの排他的なアクセス権を持つかを取得します。
        /// </summary>
        public bool IsWorkDirectoryMutexLocked => _directoryMutex.IsLocked;

        private SynchronizationContext FrameworkSyncContext { get; set; }

        //-----------------------------------------------------------------

        public void Initialize(IEnumerable<SpyModelPlugin> modelPlugins, HostIOService hostIOService)
        {
            Ensure.Argument.NotNull(modelPlugins);
            Ensure.Argument.NotNull(hostIOService);

            try
            {
                DoFirstLaunchJob();

                foreach (var modelPlugin in modelPlugins)
                {
                    _modelPlugins.Add(modelPlugin);
                    _modelRegistory.Register(modelPlugin.DataName);
                }

                _hostIOService = hostIOService;
                this.FrameworkSyncContext = SynchronizationContext.Current;
            }
            catch
            {
                this.Uninitialize();
                throw;
            }
        }

        private void DoFirstLaunchJob()
        {
            bool isFirstLaunch = false;
            _launchMutex = new Mutex(false, LaunchMutexName, out isFirstLaunch);

            if (!isFirstLaunch)
            {
                return;
            }

            if (_directoryMutex.TryLock())
            {
                try
                {
                    // ロックファイルを削除します。
                    DirectoryUtility.EnumerateDateTimePath(this.WorkDirectoryPath)
                        .SelectMany(dir => this.EnumerateLockFile(dir))
                        .ForEach(lockFile =>
                        {
                            try
                            {
                                File.Delete(lockFile);
                            }
                            catch
                            {
                                // 無視
                            }
                        });

                    // 受信ディレクトリを削除します。
                    Directory.EnumerateDirectories(this.WorkDirectoryPath, RecordingDirectoryName + "*")
                        .ForEach(dir =>
                        {
                            try
                            {
                                Directory.Delete(dir, recursive: true);
                            }
                            catch
                            {
                                // 無視
                            }
                        });
                }
                finally
                {
                    _directoryMutex.Release();
                }
            }
        }

        public void Uninitialize()
        {
            this.ClearAll();
            _modelPlugins.Clear();

            _hostIOService = null;
            this.FrameworkSyncContext = null;
        }

        public void ClearAll()
        {
            this.Stop();
            this.StopLoadFiles();

            this.GetCurrentData().Uninitialize();
            this.HasData = false;

            Disposer.SafeDispose(ref _fileManager);

            // 作業ディレクトリ内のデータを開いていたときはロックファイルを削除します。
            if (this.HasLoadedData)
            {
                var parentDir = Path.GetDirectoryName(this.LoadedDataDirectoryPath);
                if (parentDir == this.WorkDirectoryPath)
                {
                    DeleteLockFile(this.LoadedDataDirectoryPath);
                }
            }

            this.LoadedDataDirectoryPath = string.Empty;

            _loadedFilePaths.Clear();

            this.LatestDataTimestamp = SpyGlobalTime.InvalidValue;
        }

        public void ClearAllAndDeleteOldDataDirectories()
        {
            var isDataRecorded = _fileManager != null && _isFileManagerForRecording;

            this.ClearAll();

            // 受信ディレクトリが日時の名前に変更されたはずなので、古いデータディレクトリを整理します。
            if (isDataRecorded)
            {
                this.DeleteOldDataDirectories();
            }
        }

        public void Start(string targetPlatformName, object syncPort, object dataPort)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(targetPlatformName);
            Ensure.Operation.NotNull(_hostIOService);

            var hostIO = _hostIOService.GetHostIO(targetPlatformName);
            Ensure.Operation.NotNull(hostIO);

            Ensure.Operation.AreEqual(this.State, ServiceState.NotStarted);

            this.ClearAllAndDeleteOldDataDirectories();

            _session.Start(hostIO, this.FrameworkSyncContext, syncPort, dataPort, this.RecordingDirectoryPath);
        }

        public void Stop()
        {
            _session.Stop();

            _receivedDataCache = null;
            _lastDataReceivedNotificationTime = DateTime.MinValue;
        }

        /// <summary>
        /// 要求する Spy データ名の一覧を更新します。
        /// </summary>
        public void RequestSpyDatas(IEnumerable<string> dataNames)
        {
            Ensure.Argument.NotNull(dataNames);

            var hashSet = new HashSet<string>(dataNames);

            // FrameSync は必須。
            hashSet.Add(FrameSyncSpyModelPlugin.SpyDataName);

            _session.RequestSpyDatas(hashSet);
        }

        public void CopyAllRecordedFiles(string outputDirectory)
        {
            Ensure.Operation.NotNull(_fileManager);

            Directory.CreateDirectory(outputDirectory);

            _fileManager.CopyDataFileTo(
                _modelRegistory.AllDataNames,
                (_isFileManagerForRecording) ? this.RecordingDirectoryPath : this.LoadedDataDirectoryPath,
                outputDirectory);
        }

        public void StopLoadFiles()
        {
            if (_cancellationTokenSource != null)
            {
                _cancellationTokenSource.Cancel();
                try
                {
                    Task.WaitAll(_loaderTask);
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// 通信データのセーブファイルを読み込みます。
        ///
        /// パケットの送信順とモデルへの到着順を同じにするため
        /// これまでに読み込まれたデータは一旦破棄し
        /// すべて読み直します。
        /// </summary>
        /// <param name="filePaths"></param>
        /// <param name="progress"></param>
        public Task LoadFiles(IEnumerable<string> filePaths, IProgress<int> progress = null)
        {
            Ensure.Argument.NotNull(filePaths);
            Ensure.Argument.True(filePaths.Count() > 0);
            foreach (var filePath in filePaths)
            {
                Ensure.Argument.StringIsNotNullOrEmpty(filePath);
                Ensure.Argument.True(File.Exists(filePath));
                Ensure.Argument.True(Path.GetExtension(filePath) == ConstConfig.SpyDataFileExtention);
            }

            // NOTE: 読込済ファイルは ClearAll() で消えてしまうので、ここで含めておきます。
            var allFilePaths = filePaths
                .Select(it => Path.GetFullPath(it))
                .Concat(_loadedFilePaths)
                .Distinct()
                .ToList();

            var directoryPath = allFilePaths
                .Select(it => Path.GetDirectoryName(it))
                .Distinct()
                .SingleOrDefault();

            // 複数のディレクトリが指定されていないこと、および
            // 読み込み済みのファイルと同じディレクトリであることを確認します。
            Ensure.Argument.True(directoryPath != null);

            Ensure.Operation.True(!this.HasRecordedData);

            this.ClearAll();

            this.IsLoading = true;
            this.BeginLoadFiles?.Invoke(this, EventArgs.Empty);

            // 作業ディレクトリ内のデータを開くときは、
            // 古いデータの削除に含まれないようにするため
            // ロックファイルを作成します。
            {
                var parentPath = Path.GetDirectoryName(directoryPath);
                if (parentPath == this.WorkDirectoryPath)
                {
                    CreateLockFile(directoryPath);
                }
            }

            this.LoadedDataDirectoryPath = directoryPath;
            this.OpenFileManager(directoryPath);

            _cancellationTokenSource = new CancellationTokenSource();

            _loaderTask = Task.Run(() =>
            {
                try
                {
                    this.LoadFiles(allFilePaths, _cancellationTokenSource.Token, progress);
                }
                finally
                {
                    this.IsLoading = false;
                    this.EndLoadFiles?.Invoke(this, EventArgs.Empty);
                }
            });

            return _loaderTask;
        }

        private void LoadFiles(IEnumerable<string> filePaths, CancellationToken cancel, IProgress<int> progress)
        {
            // すべてのファイルを開きなおします。
            foreach (var filePath in filePaths)
            {
                try
                {
                    var header = _fileManager.ReadDataFileHeader(filePath);

                    Ensure.Operation.True(_modelRegistory.ModelExists(header.DataName));

                    {
                        var dataVersions = this.GetCurrentData().DataVersions;

                        // HACK : とりあえず同じデータ ID は読めないようにしている（読めるように対処してもよい）
                        Ensure.Operation.False(dataVersions.ContainsKey(header.DataName));
                        dataVersions.Add(header.DataName, header.DataVersion);
                    }

                    _loadedFilePaths.Add(filePath);
                }
                catch
                {
                    _fileManager.CloseDataFile(filePath);
                }
            }

            if (_loadedFilePaths.Count == 0)
            {
                return;
            }

            // パケットの送信順にモデルにデータを送ります。
            var sortedList = new SortedList<long, DataFileContent>();

            foreach (var filePath in _loadedFilePaths)
            {
                var enumerator = _fileManager.ReadDataFileBlocks(filePath).GetEnumerator();
                if (enumerator.MoveNext())
                {
                    var header = _fileManager.ReadDataFileHeader(filePath);
                    var dataBlock = enumerator.Current;

                    var dataFileContent = new DataFileContent()
                    {
                        Header = header,
                        DataBlocks = enumerator,
                        Size = dataBlock.Position + header.TotalDataBlockLength,
                    };

                    sortedList.Add(enumerator.Current.ID, dataFileContent);
                }
            }

            if (sortedList.Count == 0)
            {
                return;
            }

            var latestTimestamp = sortedList.Values[0].DataBlocks.Current.Timestamp;
            this.SetLatestDataTimestamp(latestTimestamp);

            long totalSize = sortedList.Values
                .Select(it => it.Size)
                .Sum();

            long totalReadSize = 0;
            int percent = 0;

            progress.Report(percent);

            var stopwatch = Stopwatch.StartNew();

            while (sortedList.Count > 0 && !cancel.IsCancellationRequested)
            {
                var first = sortedList.Values[0];

                sortedList.RemoveAt(0);

                var dataBlock = first.DataBlocks.Current;

                this.PushData(first.Header.DataName, first.Header.DataVersion, dataBlock);

                var timestamp = dataBlock.Timestamp;
                if (latestTimestamp < timestamp)
                {
                    latestTimestamp = timestamp;
                }

                var readSize = dataBlock.Position + dataBlock.RawData.Length;
                totalReadSize += readSize - first.LastReadSize;
                first.LastReadSize = readSize;

                if (first.DataBlocks.MoveNext())
                {
                    sortedList.Add(first.DataBlocks.Current.ID, first);
                }

                if (stopwatch.ElapsedMilliseconds > LatestDataTimestampUpdateInterval)
                {
                    // Sync 情報が無いときの時間更新用
                    this.SetLatestDataTimestamp(latestTimestamp);
                    stopwatch.Restart();
                }

                if (progress != null)
                {
                    var newPercent = (int)(totalReadSize * 100 / totalSize);
                    if (percent != newPercent)
                    {
                        percent = newPercent;
                        progress.Report(percent);
                    }
                }
            }

            // Sync 情報が無いときの時間更新用
            this.SetLatestDataTimestamp(latestTimestamp);

            progress.Report(percent);
        }

        /// <summary>
        /// 指定したディレクトリに作られるロックファイルのパスを得ます。
        /// ロックファイルの名前は Spy.exe のインスタンス毎に変わります。
        /// </summary>
        /// <param name="basePath"></param>
        /// <returns></returns>
        public string GetLockFilePath(string basePath)
        {
            return Path.Combine(basePath, LockFileName + this.Guid.ToString());
        }

        /// <summary>
        /// 指定したディレクトリ内にあるロックファイルの一覧を得ます。
        /// </summary>
        /// <param name="basePath"></param>
        /// <returns></returns>
        public IEnumerable<string> EnumerateLockFile(string basePath)
        {
            return Directory.EnumerateFiles(basePath, LockFileName + "*");
        }

        /// <summary>
        /// 指定したディレクトリ内にロックファイルを作成します。
        /// </summary>
        /// <param name="dirPath"></param>
        private void CreateLockFile(string dirPath)
        {
            var lockFilePath = this.GetLockFilePath(dirPath);
            Ensure.Operation.True(this.LockWorkDirectoryMutex());
            try
            {
                if (!File.Exists(lockFilePath))
                {
                    File.Create(lockFilePath).Close();
                }
            }
            finally
            {
                this.ReleaseWorkDirectoryMutex();
            }
        }

        /// <summary>
        /// 指定したディレクトリ内にあるロックファイルを削除します。
        /// </summary>
        /// <param name="dirPath"></param>
        private void DeleteLockFile(string dirPath)
        {
            var lockFilePath = this.GetLockFilePath(dirPath);
            Ensure.Operation.True(this.LockWorkDirectoryMutex());
            try
            {
                if (File.Exists(lockFilePath))
                {
                    File.Delete(lockFilePath);
                }
            }
            finally
            {
                this.ReleaseWorkDirectoryMutex();
            }
        }

        private void UpdateHasRecordedData()
        {
            bool value = string.IsNullOrEmpty(this.LoadedDataDirectoryPath) && this.HasData;

            this.SetPropertyValue(ref _hasRecordedData, value, nameof(HasRecordedData));
        }

        private void UpdateHasLoadedData()
        {
            bool value = !string.IsNullOrEmpty(this.LoadedDataDirectoryPath) && this.HasData;

            this.SetPropertyValue(ref _hasLoadedData, value, nameof(HasLoadedData));
        }

        /// <summary>
        /// 指定したディレクトリが受信ディレクトリであるかを判定します。
        /// (Spy.exe の他のインスタンスの受信ディレクトリも含みます。)
        /// </summary>
        /// <param name="dirPath"></param>
        /// <returns></returns>
        public bool IsRecordingDirectory(string dirPath)
        {
            return dirPath.StartsWith(Path.Combine(this.WorkDirectoryPath, RecordingDirectoryName));
        }

        public SpyDataPool GetCurrentData()
        {
            return _currentData;
        }

        /// <summary>
        /// ターゲットとの排他的な通信権を得ます。
        /// </summary>
        /// <returns>権利が得られたら true を返します。</returns>
        public bool LockConnectionMutex()
        {
            return _connectionMutex.TryLock();
        }

        /// <summary>
        /// ターゲットとの排他的な通信権を解放します。
        /// </summary>
        public void ReleaseConnectionMutex()
        {
            _connectionMutex.Release();
        }

        /// <summary>
        /// 作業ディレクトリへの排他的なアクセス権を得ます。
        /// </summary>
        /// <returns>権利が得られたら true を返します。</returns>
        public bool LockWorkDirectoryMutex()
        {
            return _directoryMutex.TryLock();
        }

        /// <summary>
        /// 作業ディレクトリへの排他的なアクセス権を解放します。
        /// </summary>
        public void ReleaseWorkDirectoryMutex()
        {
            _directoryMutex.Release();
        }

        protected override void Dispose(bool disposing)
        {
            this.Uninitialize();
            base.Dispose(disposing);
        }

        private void HandleNewData(SpyRawDataEventArgs args)
        {
            this.CreateFileManager();

            if (_session.StateWithEvaluation < SpySession.SessionState.Running)
            {
                return;
            }

            // 対応していないデータはスキップする。
            if (!_modelRegistory.ModelExists(args.DataName))
            {
                return;
            }

            // DataVersion が読まれていること前提
            var dataVersions = this.GetCurrentData().DataVersions;
            Ensure.Operation.NotNull(dataVersions.Count > 0);

            // バージョンが設定されていないデータは無視する。
            Version dataVersion = dataVersions[args.DataName];
            if (dataVersion.Major == 0 && dataVersion.Minor == 0 && dataVersion.Build == 0 && dataVersion.Revision == 0)
            {
                return;
            }

            // バージョンの一致しないデータは無視する。
            if (dataVersion != args.DataVersion)
            {
                return;
            }

            // Stop() のタイミングですでに閉じられている場合は、データを破棄する
            if (!_fileManager.IsOpened)
            {
                return;
            }

            var dataInfo = new ReceivedDataInfo()
            {
                DataID = args.DataID,
                DataName = args.DataName,
                DataVersion = args.DataVersion,
                Timestamp = args.Timestamp,
                RawData = args.RawData,
            };

            this.PushNewData(dataInfo);
        }

        private void OpenFileManager(string directoryPath)
        {
            if (_fileManager != null)
            {
                return;
            }

            _fileManager = new SpyFileManager();
            _fileManager.Open(directoryPath);
            _isFileManagerForRecording = false;

            this.GetCurrentData().Initialize(_fileManager);
        }

        private void CreateFileManager()
        {
            if (_fileManager != null)
            {
                return;
            }

            _fileManager = new SpyFileManager();
            _fileManager.Create(this.RecordingDirectoryPath, _session.IsLittleEndian);
            _isFileManagerForRecording = true;

            this.GetCurrentData().Initialize(_fileManager);
        }

        /// <summary>
        /// 古いデータディレクトリを削除します。
        /// </summary>
        private void DeleteOldDataDirectories()
        {
            this.LockWorkDirectoryMutex();
            try
            {
                DirectoryUtility.EnumerateDateTimePath(this.WorkDirectoryPath)
                    .OrderByDescending(dir => dir)
                    .Skip(MaxOldDataDirectoryCount)
                    .Where(dir => this.EnumerateLockFile(dir).IsEmpty())
                    .ToArray()
                    .ForEach(dir =>
                    {
                        try
                        {
                            Directory.Delete(dir, recursive: true);
                        }
                        catch
                        {
                            // 無視
                        }
                    });
            }
            finally
            {
                this.ReleaseWorkDirectoryMutex();
            }
        }

        private void PushNewData(ReceivedDataInfo dataInfo)
        {
            // ファイル書き込みはそのままワーカースレッドで処理。
            this.WriteData(dataInfo);

            // Post() を頻繁に実行すると処理負荷に影響するため、
            // 受信データをキャッシュしてまとめて Post() する。
            if (_receivedDataCache == null)
            {
                _receivedDataCache = new List<ReceivedDataInfo>();
            }

            _receivedDataCache.Add(dataInfo);

            var now = DateTime.Now;

            if (_lastDataReceivedNotificationTime + this.DataReceivedNotificationInterval < now)
            {
                _lastDataReceivedNotificationTime = now;

                var temp = _receivedDataCache;
                _receivedDataCache = null;

                // データ受信通知はメインスレッドで処理。
                this.FrameworkSyncContext.Post(
                    state => this.NotifyDataReceived((IList<ReceivedDataInfo>)state),
                    temp);
            }
        }

        private void WriteData(ReceivedDataInfo dataInfo)
        {
            Assertion.Operation.NotNull(_fileManager);

            // 作業ファイルに書き込む
            if (!_fileManager.IsDataFileMade(dataInfo.DataName))
            {
                _fileManager.MakeDataFile(dataInfo.DataID, dataInfo.DataName, dataInfo.DataVersion);
            }

            var dataBlock = _fileManager.WriteDataBlock(dataInfo.DataName, dataInfo.RawData, dataInfo.Timestamp);
            dataInfo.DataBlock = dataBlock;
        }

        private void FlushFiles()
        {
            if (_fileManager != null)
            {
                _fileManager.FlushFiles();
            }
        }

        private void NotifyDataReceived(IList<ReceivedDataInfo> dataInfos)
        {
            foreach (var dataInfo in dataInfos)
            {
                this.NotifyDataReceived(dataInfo);
            }

            if (!dataInfos.IsEmpty())
            {
                var timestamp = new SpyGlobalTime(dataInfos.Last().Timestamp);
                this.SetLatestDataTimestamp(timestamp);
            }
        }

        private void NotifyDataReceived(ReceivedDataInfo dataInfo)
        {
            this.PushData(dataInfo.DataName, dataInfo.DataVersion, dataInfo.DataBlock);
            this.DataChanged(this, new SpyDataChangedEventArgs(dataInfo.DataName));
        }

        private SpyModel CreateModel(ISpyDataReader reader, string dataName, Version dataVersion)
        {
            Assertion.Argument.NotNull(reader);

            SpyModelPlugin plugin;
            if (!_modelPlugins.TryGetItem(dataName, out plugin))
            {
                return null;
            }

            var result = plugin.CreateSpyModel();
            Ensure.Operation.NotNull(result);

            result.Initialize(GetCurrentData(), reader, dataName, dataVersion);
            return result;
        }

        private void PushData(string dataName, Version dataVersion, SpyDataBlock dataBlock)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);
            Assertion.Argument.NotNull(dataBlock);

            Assertion.Operation.NotNull(_fileManager);

            SpyModel model = null;

            this.GetCurrentData().Models.TryGetValue(dataName, out model);

            if (model == null)
            {
                model = this.CreateModel(_fileManager, dataName, dataVersion);

                if (model == null)
                {
                    // TODO : LogService が実装されたら、ログに流す
                    Debug.WriteLine(string.Format("Warning : unknown data id. : {0}", dataName));
                    return;
                }

                this.GetCurrentData().AddModel(dataName, model);
                this.HasData = true;
            }

            model.PushData(dataBlock);
        }

        private void NotifyStateChanged()
        {
            this.StateChanged(this, EventArgs.Empty);
        }

        private void SetLatestDataTimestamp(SpyGlobalTime timestamp)
        {
            if (!this.LatestDataTimestamp.IsValid || this.LatestDataTimestamp < timestamp)
            {
                this.LatestDataTimestamp = timestamp;
            }
        }
    }
}
