﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using System;
using System.IO;
using System.Text;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Spy モデルです。
    /// </summary>
    public abstract class SpyModel : ObservableObject
    {
        private ISpyMandatoryModelAccessor _mandatoryModelAccessor;
        private ISpyDataReader _dataManager;
        private bool? _isLittleEndian;

        //-----------------------------------------------------------------

        /// <summary>
        /// モデルのデータ名を取得します。
        /// </summary>
        public string DataName { get; private set; }

        /// <summary>
        /// モデルのデータバージョンを取得します。
        /// </summary>
        public Version DataVersion { get; private set; }

        //-----------------------------------------------------------------

        internal void Initialize(
            ISpyMandatoryModelAccessor mandatoryModelAccessor,
            ISpyDataReader dataManager,
            string dataName,
            Version dataVersion)
        {
            Ensure.Argument.NotNull(mandatoryModelAccessor);
            Ensure.Argument.NotNull(dataManager);

            _mandatoryModelAccessor = mandatoryModelAccessor;
            _dataManager = dataManager;
            _isLittleEndian = _dataManager.IsLittleEndian;
            this.DataName = dataName;
            this.DataVersion = dataVersion;
        }

        /// <summary>
        /// モデルにデータを追加します。
        /// </summary>
        /// <param name="data">データを指定します。</param>
        internal void PushData(SpyDataBlock data)
        {
            Assertion.Argument.NotNull(data);
            Assertion.Argument.NotNull(data.RawData);

            if (data.RawData.Length == 0)
            {
                return;
            }

            this.OnPushData(data);
        }

        /// <summary>
        /// 指定データブロック ID のデータを読み込みます。
        /// </summary>
        /// <param name="dataBlockID">データブロック ID を指定します。</param>
        /// <returns>読み込んだデータを返します。データが無いときはnullを返します。</returns>
        protected SpyDataBlock ReadData(long dataBlockID)
        {
            Ensure.Operation.NotNull(_dataManager);

            var dataBlock = _dataManager.ReadDataBlock(this.DataName, dataBlockID);

            return dataBlock;
        }

        /// <summary>
        /// データ読み込み用のBinaryReaderを取得します。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        protected BinaryReader CreateDataReader(SpyDataBlock data)
        {
            Ensure.Operation.True(_isLittleEndian.HasValue);

            if (data == null)
            {
                return null;
            }

            var dataStream = new MemoryStream(data.RawData);

            return _isLittleEndian.Value ?
                LittleEndianBinaryReader.Create(dataStream, Encoding.ASCII) :
                BigEndianBinaryReader.Create(dataStream, Encoding.ASCII);
        }

        /// <summary>
        /// モデルにデータを追加します。
        /// </summary>
        /// <param name="dataBlock">データブロックが渡されます。</param>
        protected abstract void OnPushData(SpyDataBlock dataBlock);

        /// <summary>
        /// 指定したタイムスタンプを含むフレームのフレーム同期情報を得ます。
        /// </summary>
        /// <param name="timestamp"></param>
        /// <returns>
        /// 見つからなかった場合は SpyTime.InvalidValue を返します。
        /// </returns>
        protected SpyTime GetBelongingFrame(SpyGlobalTime timestamp)
        {
            var frameSyncModel = _mandatoryModelAccessor.FrameSyncModel;
            if (frameSyncModel == null || frameSyncModel.MaximumFrame == null)
            {
                return SpyTime.InvalidValue;
            }
            else if (frameSyncModel.MaximumFrame.Timestamp <= timestamp)
            {
                // 受信中のパケットのフレームは末尾に決まっているので高速にアクセス。
                return frameSyncModel.MaximumFrame;
            }
            else
            {
                SpyTime value;
                if (frameSyncModel.AllFrames.TryFindValueOf(timestamp, out value))
                {
                    return value;
                }
                else
                {
                    // 最初の FrameSync 情報より前のとき。
                    return SpyTime.InvalidValue;
                }
            }
        }

        /// <summary>
        /// 指定した AudioFrame を含むフレームのフレーム同期情報を得ます。
        /// </summary>
        /// <param name="audioFrame"></param>
        /// <returns>
        /// 見つからなかった場合は SpyTime.InvalidValue を返します。
        /// </returns>
        protected SpyTime GetBelongingFrameFromAudioFrame(Frame audioFrame)
        {
            var frameSyncModel = _mandatoryModelAccessor.FrameSyncModel;
            if (frameSyncModel == null || frameSyncModel.MaximumFrame == null)
            {
                return SpyTime.InvalidValue;
            }
            else
            {
                // 存在しない AudioFrame を指定すると二分探索が必要になるので、
                // 最終フレーム以降の AudioFrame が指定されたときは最終フレームで検索します。
                // MaximumFrame と AudioFrames.Last() は必ずしも一致しないので、
                // MaximumFrame をそのまま返してはいけません。
                if (frameSyncModel.MaximumFrame.AudioFrame < audioFrame)
                {
                    audioFrame = frameSyncModel.MaximumFrame.AudioFrame;
                }

                SpyTime value;
                if (frameSyncModel.AudioFrames.TryGetValueOf(audioFrame, out value))
                {
                    return value;
                }
                else
                {
                    Assertion.Operation.Fail();
                    return SpyTime.InvalidValue;
                }
            }
        }
    }
}
