﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Globalization;
using System.Windows.Data;
using System.Windows.Markup;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// 辞書の登録に従って値を変換するValueConverterです。
    /// </summary>
    [ContentProperty("Items")]
    public class DictionaryConverter : IValueConverter
    {
        private readonly ObservableCollection<DictionaryConverterItem> _items = new ObservableCollection<DictionaryConverterItem>();
        private readonly Dictionary<object, object> _dictionary = new Dictionary<object, object>();
        private TypeConverter _keyTypeConverter;
        private TypeConverter _valueTypeConverter;
        private bool _isStringKey = true;
        private object _defaultValue = null;
        private object _convertedDefaultValue = null;

        /// <summary>
        /// 辞書に登録される要素です。
        /// </summary>
        public Collection<DictionaryConverterItem> Items { get { return _items; } }

        /// <summary>
        /// KeyTypeConverter が指定された場合は Items の各要素は Key が型変換されたのちに辞書に登録されます。
        /// XAMLの読み込み時にまとめて変換が行われるため、オーバーヘッドが削減されます。
        /// </summary>
        public TypeConverter KeyTypeConverter
        {
            get
            {
                return _keyTypeConverter;
            }

            set
            {
                _keyTypeConverter = value;
                this.RefreshDictionary();
            }
        }

        /// <summary>
        /// ValueTypeConverter が指定された場合は Items の各要素は Value が型変換されたのちに辞書に登録されます。
        /// XAMLの読み込み時にまとめて変換が行われるため、オーバーヘッドが削減されます。
        /// </summary>
        public TypeConverter ValueTypeConverter
        {
            get
            {
                return _valueTypeConverter;
            }

            set
            {
                _valueTypeConverter = value;
                this.RefreshDictionary();
                _convertedDefaultValue = this.ConvertValue(_defaultValue);
            }
        }

        public object DefaultValue
        {
            get
            {
                return _defaultValue;
            }

            set
            {
                _defaultValue = value;
                _convertedDefaultValue = this.ConvertValue(_defaultValue);
            }
        }

        public DictionaryConverter()
        {
            _items.CollectionChanged += OnItemsCollectionChanged;
        }

        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            object result;
            bool found;

            Assertion.Argument.NotNull(value);
            if (value == null)
            {
                return _convertedDefaultValue;
            }

            if (this.KeyTypeConverter != null)
            {
                found = _dictionary.TryGetValue(this.KeyTypeConverter.ConvertFrom(value), out result);
            }
            else if (_isStringKey)
            {
                found = _dictionary.TryGetValue(value.ToString(), out result);
            }
            else
            {
                found = _dictionary.TryGetValue(value, out result);
            }

            if (!found)
            {
                result = _convertedDefaultValue;
            }

            return result;
        }

        [Obsolete("実装しません")]
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }

        private void OnItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (e.Action == NotifyCollectionChangedAction.Reset)
            {
                this.RefreshDictionary();
            }
            else
            {
                if (e.OldItems != null)
                {
                    foreach (var item in e.OldItems)
                    {
                        this.RemoveItem((DictionaryConverterItem)item);
                    }
                }

                if (e.NewItems != null)
                {
                    foreach (var item in e.NewItems)
                    {
                        Assertion.Operation.NotNull(item, "アイテムに不正な値(null)が指定されました。オブジェクトの生成に失敗したか参照が解決できなかった可能性があります。");
                        this.AddItem((DictionaryConverterItem)item);
                    }
                }

                this.UpdateIsStringKey();
            }
        }

        private object ConvertKey(object key)
        {
            if (_keyTypeConverter == null || key == null)
            {
                return key;
            }
            else
            {
                return _keyTypeConverter.ConvertFrom(key);
            }
        }

        private object ConvertValue(object value)
        {
            if (_valueTypeConverter == null || value == null)
            {
                return value;
            }
            else
            {
                return _valueTypeConverter.ConvertFrom(value);
            }
        }

        private void RefreshDictionary()
        {
            _dictionary.Clear();

            foreach (var item in _items)
            {
                Assertion.Operation.NotNull(item, "アイテムに不正な値(null)が指定されました。オブジェクトの生成に失敗したか参照が解決できなかった可能性があります。");
                this.AddItem(item);
            }

            this.UpdateIsStringKey();
        }

        private void UpdateIsStringKey()
        {
            _isStringKey = true;
            foreach (var item in _dictionary)
            {
                if (!(item.Key is string))
                {
                    _isStringKey = false;
                }
            }
        }

        private void AddItem(DictionaryConverterItem item)
        {
            object key = this.ConvertKey(item.Key);
            object value = this.ConvertValue(item.Value);
            _dictionary[key] = value;
        }

        private void RemoveItem(DictionaryConverterItem item)
        {
            object key = this.ConvertKey(item.Key);
            _dictionary.Remove(key);
        }
    }

    /// <summary>
    /// DictionaryConverter の辞書の要素です。
    /// </summary>
    [ContentProperty("Value")]
    public class DictionaryConverterItem
    {
        /// <summary>
        /// 辞書のキーです。
        /// DictionaryConverter.KeyTypeConverter を指定すると、
        /// 登録時に型変換を行うことができます。
        /// </summary>
        public object Key { get; set; }

        /// <summary>
        /// 辞書の値です。
        /// DictionaryConverter.ValueTypeConverter を指定すると、
        /// 登録時に型変換を行うことができます。
        /// </summary>
        public object Value { get; set; }
    }
}
