﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation;
using NintendoWare.Spy.Foundation.Commands;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace NintendoWare.Spy.Framework
{
    /// <summary>
    /// プレゼンターです。
    /// </summary>
    public abstract class Presenter : ObservableObject
    {
        private readonly Lazy<ServiceProvider> _serviceProvider = new Lazy<ServiceProvider>();

        private bool _isInitialized = false;

        //-----------------------------------------------------------------

        public bool IsInitialized
        {
            get { return _isInitialized; }
            private set { this.SetPropertyValue(ref _isInitialized, value); }
        }

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public abstract IEnumerable<Type> DependentServiceTypes { get; }

        protected ServiceProvider ServiceProvider
        {
            get { return _serviceProvider.Value; }
        }

        //-----------------------------------------------------------------

        public void Initialize()
        {
            this.Initialize(null, null);
        }

        public void Initialize(
            IServiceProvider dependentServiceProvider,
            CommandService inheritableCommandService)
        {
            Ensure.Argument.True(
                this.DependentServiceTypes.FirstOrDefault() == null ||
                dependentServiceProvider != null);
            Ensure.Argument.False(_isInitialized);

            try
            {
                this.InstallRequiredServices(dependentServiceProvider);
                this.InitializeCommandService(inheritableCommandService);

                this.OnInitialize();
            }
            catch
            {
                this.Uninitialize();
                throw;
            }

            this.IsInitialized = true;
        }

        public void Uninitialize()
        {
            try
            {
                if (this.IsInitialized)
                {
                    this.OnUninitialize();
                }

                _serviceProvider.Value.UninstallAllServices();
            }
            finally
            {
                this.IsInitialized = false;
            }
        }

        protected override void OnDisposed(EventArgs e)
        {
            base.OnDisposed(e);

            this.Uninitialize();
            _serviceProvider.Value.UninstallAllServices();
        }

        protected virtual void OnInitialize() { }
        protected virtual void OnUninitialize() { }

        protected virtual void OnBindCommands(CommandService commandService) { }

        protected virtual Presenter GetActiveInnerPresenter()
        {
            return null;
        }

        protected ServiceProvider CreateDeepClonedServiceProvider()
        {
            return _serviceProvider.Value.DeepClone();
        }

        [DebuggerStepThrough]
        protected CommandService GetCommandService()
        {
            return this.ServiceProvider.GetService<CommandService>();
        }

        private void InstallRequiredServices(IServiceProvider dependentServiceProvider)
        {
            foreach (var serviceType in this.DependentServiceTypes)
            {
                // CommandService は Presenter 毎に生成するので、ここではスキップします。
                if (serviceType == typeof(CommandService))
                {
                    continue;
                }

                Assertion.Argument.NotNull(dependentServiceProvider);

                var service = dependentServiceProvider.GetService(serviceType);
                Ensure.NotNull<KeyNotFoundException>(service);

                _serviceProvider.Value.InstallService(serviceType, service);
            }
        }

        private void InitializeCommandService(CommandService inheritableCommandService)
        {
            var commandService = new CommandService();

            commandService.Initialize(
                () =>
                {
                    var activePresenter = this.GetActiveInnerPresenter();
                    return activePresenter == null ?
                        null : activePresenter.ServiceProvider.GetService<CommandService>();
                });

            if (inheritableCommandService != null)
            {
                commandService.Inherit(inheritableCommandService);
            }

            this.OnBindCommands(commandService);

            _serviceProvider.Value.InstallService(typeof(CommandService), commandService);
        }
    }
}
