﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;

namespace NintendoWare.Spy.Foundation
{
    public class ServiceProvider : DisposableObject, IServiceProvider, IDeepClonable
    {
        private readonly Dictionary<Type, object> _installedServices = new Dictionary<Type, object>();

        //-----------------------------------------------------------------

        public static ServiceProvider Create(IEnumerable<Tuple<Type, object>> items)
        {
            return CreateImpl(items);
        }

        public static ServiceProvider Create(params Tuple<Type, object>[] items)
        {
            return CreateImpl(items);
        }

        /// <summary>
        /// 指定した型のサービス オブジェクトを取得します。
        /// </summary>
        /// <param name="serviceType">サービスの型を指定します。</param>
        /// <returns>
        /// 指定した方のサービスを取得します。
        /// 存在しない場合は null を返します。
        /// </returns>
        public object GetService(Type serviceType)
        {
            Ensure.Argument.NotNull(serviceType);

            object result;
            if (!_installedServices.TryGetValue(serviceType, out result))
            {
                return null;
            }

            Assertion.Operation.True(result.GetType() == serviceType);
            return result;
        }

        /// <summary>
        /// 指定した型のサービス オブジェクトを取得します。
        /// </summary>
        /// <typeparam name="TService">サービスの型を指定します。</typeparam>
        /// <returns>
        /// 指定した方のサービスを取得します。
        /// 存在しない場合は null を返します。
        /// </returns>
        public TService GetService<TService>()
            where TService : class
        {
            object result;
            if (!_installedServices.TryGetValue(typeof(TService), out result))
            {
                return null;
            }

            Assertion.Operation.True(result is TService);
            return result as TService;
        }

        public void InstallService(Type type, object service)
        {
            Ensure.Argument.NotNull(type);
            Ensure.Argument.NotNull(service);

            _installedServices.Add(type, service);
        }

        public void UninstallService(Type type)
        {
            Ensure.Argument.NotNull(type);

            _installedServices.Remove(type);
        }

        public void UninstallAllServices()
        {
            _installedServices.Clear();
        }

        /// <summary>
        /// オブジェクトのディープクローンを生成します。
        /// </summary>
        /// <returns>オブジェクトのディープクローンを返します。</returns>
        object IDeepClonable.DeepClone()
        {
            var result = new ServiceProvider();

            foreach (var item in _installedServices)
            {
                result._installedServices.Add(item.Key, item.Value);
            }

            return result;
        }

        protected override void OnDisposed(EventArgs e)
        {
            this.UninstallAllServices();

            base.OnDisposed(e);
        }

        public static ServiceProvider CreateImpl(IEnumerable<Tuple<Type, object>> items)
        {
            var result = new ServiceProvider();

            foreach (var item in items)
            {
                result.InstallService(item.Item1, item.Item2);
            }

            return result;
        }
    }
}
