﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.IO;
using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading;

namespace NintendoWare.Spy.Foundation.Communications.Htcs
{
    /// <summary>
    /// Siglo HostIO のチャンネルを制御するクラスです。
    /// </summary>
    public class HostIOHtcsChannel : IComChannel
    {
        private readonly IPEndPoint _ipEndPoint;
        private readonly string _portName;

        private TcpClient _tcpClient;
        private int _sendTimeout = 1000;
        private int _receiveTimeout = 1000;
        private int _keepAliveTimeout = 1000;

        private Stream _stream;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="ipEndPoint">接続先を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        private HostIOHtcsChannel(IPEndPoint ipEndPoint, string portName)
        {
            Assertion.Argument.NotNull(ipEndPoint);
            _ipEndPoint = ipEndPoint;
            _portName = portName;
        }

        /// <summary>
        /// デストラクタです。
        /// </summary>
        ~HostIOHtcsChannel()
        {
            this.Close();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ポート番号を取得します。
        /// </summary>
        public int Port
        {
            get { throw new NotImplementedException(); }
        }

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string PortName
        {
            get { return _portName; }
        }

        /// <summary>
        /// 開閉状態の有無を取得します。
        /// </summary>
        public bool IsOpened
        {
            get
            {
                return _tcpClient == null ? false : _tcpClient.Connected;
            }
        }

        /// <summary>
        /// 送信タイムアウトを取得または設定します。
        /// </summary>
        public int SendTimeout
        {
            get { return _sendTimeout; }
            set
            {
                if (_sendTimeout == value)
                {
                    return;
                }

                _sendTimeout = value;

                if (_tcpClient != null)
                {
                    if (value == Timeout.Infinite)
                    {
                        // TcpClient では 0 がタイムアウト無しを表します。
                        _tcpClient.SendTimeout = 0;
                    }
                    else
                    {
                        _tcpClient.SendTimeout = value;
                    }
                }
            }
        }

        /// <summary>
        /// 受信タイムアウトを取得または設定します。
        /// </summary>
        public int ReceiveTimeout
        {
            get { return _receiveTimeout; }
            set
            {
                if (_receiveTimeout == value)
                {
                    return;
                }

                _receiveTimeout = value;

                if (_tcpClient != null)
                {
                    if (value == Timeout.Infinite)
                    {
                        // TcpClient では 0 がタイムアウト無しを表します。
                        _tcpClient.ReceiveTimeout = 0;
                    }
                    else
                    {
                        _tcpClient.ReceiveTimeout = value;
                    }
                }
            }
        }

        /// <summary>
        /// キープアライブタイムアウトを取得または設定します。
        /// </summary>
        public int KeepAliveTimeout
        {
            get { return _keepAliveTimeout; }
            set
            {
                if (_keepAliveTimeout == value)
                {
                    return;
                }

                _keepAliveTimeout = value;
                this.ApplyKeepAliveTimeout();
            }
        }

        /// <summary>
        /// TCP クライアントを取得します。
        /// </summary>
        protected TcpClient TcpClient
        {
            get { return _tcpClient; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 新しいチャンネルを開きます。
        /// </summary>
        /// <param name="ipEndPoint">接続先を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        public static IComChannel Open(IPEndPoint ipEndPoint, string portName)
        {
            var newChannel = new HostIOHtcsChannel(ipEndPoint, portName);

            try
            {
                newChannel.Open();
            }
            catch
            {
                newChannel.Dispose();
                throw;
            }

            return newChannel;
        }

        /// <summary>
        /// チャンネルを閉じます。
        /// </summary>
        public void Close()
        {
            this.OnClosing();

            if (_stream != null)
            {
                _stream.Close();
            }

            if (_tcpClient != null)
            {
                _tcpClient.Close();
            }

            this.OnClosed();
        }

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        public void Dispose()
        {
            this.Close();
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ストリームを取得します。
        /// </summary>
        public Stream GetStream()
        {
            return _tcpClient != null ? _stream : null;
        }

        protected virtual void OnOpened() { }
        protected virtual void OnClosing() { }
        protected virtual void OnClosed() { }

        /// <summary>
        /// チャンネルを開きます。
        /// </summary>
        protected void Open()
        {
            var tcpClient = new TcpClient();
            tcpClient.Connect(_ipEndPoint);
            _tcpClient = tcpClient;

            this.ApplyKeepAliveTimeout();

            // 送受信タイムアウトは TcpClient のデフォルト値を利用。
            this.SendTimeout = _tcpClient.SendTimeout;
            this.ReceiveTimeout = _tcpClient.ReceiveTimeout;

            _stream = new CachedStream(_tcpClient.GetStream());
        }

        /// <summary>
        /// キープアライブタイムアウトを反映します。
        /// </summary>
        private void ApplyKeepAliveTimeout()
        {
            if (_tcpClient == null)
            {
                return;
            }

            byte[] buffer = new byte[12];
            BitConverter.GetBytes(_keepAliveTimeout == 0 ? 0 : 1).CopyTo(buffer, 0);    // スイッチ
            BitConverter.GetBytes(_keepAliveTimeout).CopyTo(buffer, 4);                 // msec

            _tcpClient.Client.IOControl(IOControlCode.KeepAliveValues, buffer, null);
        }
    }
}
