﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Reflection;

namespace NintendoWare.Spy.Foundation.Binary
{
    internal class ObjectBinaryReader
    {
        private const string PropertySizePostfix = "SizeInternal";
        private const string PropertyItemTypePostfix = "TypeInternal";

        private delegate void ReadObjectHandler(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj);

        private static Dictionary<Type, ReadObjectHandler> _handlers =
            new Dictionary<Type, ReadObjectHandler>();

        private readonly BinaryReader _reader;

        static ObjectBinaryReader()
        {
            Initialize();
        }

        public ObjectBinaryReader(BinaryReader reader)
        {
            Assertion.Argument.NotNull(reader);
            _reader = reader;
        }

        public BinaryReader Reader
        {
            get { return _reader; }
        }

        public long Position
        {
            get { return _reader.BaseStream.Position; }
            set { _reader.BaseStream.Position = value; }
        }

        public static bool IsTypeSupported(Type type)
        {
            return ObjectBinaryReader.GetTargetType(type) != null;
        }

        public void Read(IBinarizable obj)
        {
            Assertion.Argument.NotNull(obj);

            foreach (var property in obj.GetProperties())
            {
                Type type = ObjectBinaryReader.GetTargetType(property.PropertyType);

                // プロパティの型に対応していない場合は、
                // プロパティ値の型から検索します。
                if (type == null)
                {
                    var value = property.GetValue(obj, null);

                    if (value == null)
                    {
                        throw new ArgumentException("unsupported type.");
                    }

                    type = ObjectBinaryReader.GetTargetType(value.GetType());

                    if (type == null)
                    {
                        throw new ArgumentException("unsupported type.");
                    }
                }

                _handlers[type](this, property, obj);
            }
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        private static void Initialize()
        {
            ObjectBinaryReader._handlers.Add(
                typeof(byte),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadByte(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(Int16),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadInt16(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(Int32),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadInt32(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(Int64),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadInt64(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(UInt16),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadUInt16(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(UInt32),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadUInt32(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(UInt64),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadUInt64(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(float),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadSingle(), null));

            ObjectBinaryReader._handlers.Add(
                typeof(bool),
                (self, property, obj) => property.SetValue(obj, self.Reader.ReadBoolean(), null));

            ObjectBinaryReader._handlers.Add(typeof(char[]), ReadChars);
            ObjectBinaryReader._handlers.Add(typeof(byte[]), ReadBytes);
            ObjectBinaryReader._handlers.Add(typeof(IBinarizable), ReadProperties);
            ObjectBinaryReader._handlers.Add(typeof(IList), ReadList);
            ObjectBinaryReader._handlers.Add(typeof(Version), ReadVersion);
        }

        private static Type GetTargetType(Type type)
        {
            foreach (Type supportedType in _handlers.Keys)
            {
                if (type.IsSupported(supportedType))
                {
                    return supportedType;
                }
            }

            return null;
        }

        private static void ReadChars(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Reader);

            PropertyInfo sizeProperty = obj.GetProperty(property.Name + PropertySizePostfix);

            if (sizeProperty == null || sizeProperty.PropertyType != typeof(int))
            {
                throw new InvalidDataException("unsupported type.");
            }

            int size = (int)sizeProperty.GetValue(obj, null);

            if (size == 0)
            {
                return;
            }

            var readValue = self.Reader.ReadChars(size);

            if (readValue == null)
            {
                throw new IOException($"self.Reader.ReadChars() returns null, expected char[{size}].");
            }
            else if (readValue.Length < size)
            {
                throw new IOException($"self.Reader.ReadChars() returns char[{readValue.Length}], expected char[{size}].");
            }

            property.SetValue(obj, readValue, null);
        }

        private static void ReadBytes(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Reader);

            PropertyInfo sizeProperty = obj.GetProperty(property.Name + PropertySizePostfix);

            if (sizeProperty == null || sizeProperty.PropertyType != typeof(int))
            {
                throw new InvalidDataException("unsupported type.");
            }

            int size = (int)sizeProperty.GetValue(obj, null);

            if (size == 0)
            {
                return;
            }

            var readValue = self.Reader.ReadBytes(size);

            if (readValue == null || readValue.Length < size)
            {
                throw new IOException();
            }

            property.SetValue(obj, readValue, null);
        }

        private static void ReadProperties(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Reader);

            IBinarizable proeprtyValue = property.GetValue(obj, null) as IBinarizable;

            if (proeprtyValue == null)
            {
                throw new InvalidDataException("unsupported type.");
            }

            self.Read(proeprtyValue);
        }

        private static void ReadList(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Reader);

            var list = property.GetValue(obj, null) as IList;

            if (list == null)
            {
                // 今の実装ではコレクションインスタンスが必要です。
                throw new InvalidDataException("unsupported type.");
            }

            PropertyInfo sizeProperty = obj.GetProperty(property.Name + PropertySizePostfix);

            if (sizeProperty == null || sizeProperty.PropertyType != typeof(int))
            {
                throw new InvalidDataException("unsupported type.");
            }

            PropertyInfo itemTypeProperty = obj.GetProperty(property.Name + PropertyItemTypePostfix);

            if (itemTypeProperty == null || itemTypeProperty.PropertyType != typeof(Type))
            {
                throw new InvalidDataException("unsupported type.");
            }

            int count = (int)sizeProperty.GetValue(obj, null);

            if (count == 0)
            {
                return;
            }

            Type itemType = (Type)itemTypeProperty.GetValue(obj, null);

            for (int index = 0; index < count; ++index)
            {
                var item = Activator.CreateInstance(itemType) as IBinarizable;

                if (item == null)
                {
                    throw new InvalidDataException("unsupported type.");
                }

                self.Read(item);
                list.Add(item);
            }
        }

        private static void ReadVersion(ObjectBinaryReader self, PropertyInfo property, IBinarizable obj)
        {
            Assertion.Argument.NotNull(self);
            Assertion.Argument.NotNull(self.Reader);

            var readValue = self.Reader.ReadUInt32();

            var version = new Version(
                (int)((readValue >> 0) & 0xff), // major
                (int)((readValue >> 8) & 0xff), // minor
                (int)((readValue >> 16) & 0xff), // build
                (int)((readValue >> 24) & 0xff)); // revision

            property.SetValue(obj, version, null);
        }
    }
}
