﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;

namespace NintendoWare.Spy.Extensions
{
    /// <summary>
    /// Type の機能拡張を提供します。
    /// </summary>
    public static class TypeExtension
    {
        /// <summary>
        /// インターフェースをサポートしているかどうかを調査します。
        /// </summary>
        /// <param name="target">調査する型を指定します。</param>
        /// <param name="type">インターフェースの型を指定します。</param>
        /// <returns>対象の型が指定インターフェースをサポートしていれば true を返します。</returns>
        public static bool HasInterface(this Type target, Type type)
        {
            Ensure.Argument.NotNull(target);
            Ensure.Argument.NotNull(type);
            Ensure.Argument.True(type.IsInterface);

            if (target == type)
            {
                return true;
            }

            foreach (Type interfaceType in target.GetInterfaces())
            {
                if (interfaceType == type)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 型をサポートしているかどうかを調査します。
        /// </summary>
        /// <param name="target">調査する型を指定します。。</param>
        /// <param name="type">サポートしているかどうかを調べる型を指定します。。</param>
        /// <returns>対象の型がサポートしていれば true を返します。</returns>
        public static bool IsSupported(this Type target, Type type)
        {
            Ensure.Argument.NotNull(target);
            Ensure.Argument.NotNull(type);

            if (type == target)
            {
                return true;
            }

            if (type.IsInterface)
            {
                return HasInterface(target, type);
            }
            else
            {
                return target.IsSubclassOf(type);
            }
        }

        /// <summary>
        /// 指定した型が構造体であるかどうかを調べます。
        /// </summary>
        public static bool IsStruct(this Type target)
        {
            Ensure.Argument.NotNull(target);
            return !target.IsPrimitive && !target.IsClass && !target.IsInterface;
        }
    }
}
