﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;

namespace NintendoWare.Spy.Extensions
{
    /// <summary>
    /// <see cref="IEnumerable"/> の拡張メソッドを提供します。
    /// </summary>
    public static class EnumerableExtension
    {
        /// <summary>
        /// 入力シーケンスの連続する２つの要素を、出力シーケンスに射影します。
        /// </summary>
        /// <typeparam name="TSource">入力シーケンスの要素の型です。</typeparam>
        /// <param name="source">入力シーケンスです。</param>
        /// <returns>入力シーケンスの連続する２つの要素の <see cref="Tuple"/> を要素とするシーケンスです。</returns>
        /// <remarks>
        /// 入力シーケンスの要素の数が２に満たないときは、出力シーケンスは空になります。
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// 引数 source が null です。
        /// </exception>
        public static IEnumerable<Tuple<TSource, TSource>> SuccessingPair<TSource>(this IEnumerable<TSource> source)
        {
            Ensure.Argument.NotNull(source, nameof(source));

            return SuccessingPairImpl(source, (i, j) => Tuple.Create(i, j));
        }

        /// <summary>
        /// 入力シーケンスの連続する２つの要素を、出力シーケンスに射影します。
        /// </summary>
        /// <typeparam name="TSource">入力シーケンスの要素の型です。</typeparam>
        /// <typeparam name="TResult">出力シーケンスの要素の型です。</typeparam>
        /// <param name="source">入力シーケンスです。</param>
        /// <param name="selector">入力シーケンスの連続する２つの要素に対して適用される射影関数です。</param>
        /// <returns>射影関数の結果を要素とするシーケンスです。</returns>
        /// <remarks>
        /// 入力シーケンスの要素の数が２に満たないときは、出力シーケンスは空になります。
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// 引数 source または selector が null です。
        /// </exception>
        public static IEnumerable<TResult> SuccessingPair<TSource, TResult>(this IEnumerable<TSource> source, Func<TSource, TSource, TResult> selector)
        {
            Ensure.Argument.NotNull(source, nameof(source));
            Ensure.Argument.NotNull(selector, nameof(selector));

            return SuccessingPairImpl(source, selector);
        }

        private static IEnumerable<TResult> SuccessingPairImpl<TSource, TResult>(IEnumerable<TSource> source, Func<TSource, TSource, TResult> selector)
        {
            using (var enumerator = source.GetEnumerator())
            {
                if (enumerator.MoveNext())
                {
                    var item1 = enumerator.Current;

                    while (enumerator.MoveNext())
                    {
                        var item2 = enumerator.Current;
                        yield return selector(item1, item2);
                        item1 = item2;
                    }
                }
            }
        }

        /// <summary>
        /// 入力シーケンスの連続する３つの要素を、出力シーケンスに射影します。
        /// </summary>
        /// <typeparam name="TSource">入力シーケンスの要素の型です。</typeparam>
        /// <param name="source">入力シーケンスです。</param>
        /// <returns>入力シーケンスの連続する３つの要素の <see cref="Tuple"/> を要素とするシーケンスです。</returns>
        /// <remarks>
        /// 入力シーケンスの要素の数が３に満たないときは、出力シーケンスは空になります。
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// 引数 source が null です。
        /// </exception>
        public static IEnumerable<Tuple<TSource, TSource, TSource>> SuccessingTriple<TSource>(this IEnumerable<TSource> source)
        {
            Ensure.Argument.NotNull(source, nameof(source));

            return SuccessingTripleImpl(source, (i, j, k) => Tuple.Create(i, j, k));
        }

        /// <summary>
        /// 入力シーケンスの連続する３つの要素を、出力シーケンスに射影します。
        /// </summary>
        /// <typeparam name="TSource">入力シーケンスの要素の型です。</typeparam>
        /// <typeparam name="TResult">出力シーケンスの要素の型です。</typeparam>
        /// <param name="source">入力シーケンスです。</param>
        /// <param name="selector">入力シーケンスの連続する３つの要素に対して適用される射影関数です。</param>
        /// <returns>射影関数の結果を要素とするシーケンスです。</returns>
        /// <remarks>
        /// 入力シーケンスの要素の数が３に満たないときは、出力シーケンスは空になります。
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// 引数 source または selector が null です。
        /// </exception>
        public static IEnumerable<TResult> SuccessingTriple<TSource, TResult>(this IEnumerable<TSource> source, Func<TSource, TSource, TSource, TResult> selector)
        {
            Ensure.Argument.NotNull(source, nameof(source));
            Ensure.Argument.NotNull(selector, nameof(selector));

            return SuccessingTripleImpl(source, selector);
        }

        private static IEnumerable<TResult> SuccessingTripleImpl<TSource, TResult>(IEnumerable<TSource> source, Func<TSource, TSource, TSource, TResult> selector)
        {
            using (var enumerator = source.GetEnumerator())
            {
                if (!enumerator.MoveNext())
                {
                    yield break;
                }

                var item1 = enumerator.Current;

                if (!enumerator.MoveNext())
                {
                    yield break;
                }

                var item2 = enumerator.Current;

                while (enumerator.MoveNext())
                {
                    var item3 = enumerator.Current;
                    yield return selector(item1, item2, item3);
                    item1 = item2;
                    item2 = item3;
                }
            }
        }

        /// <summary>
        /// <see cref="System.Linq.Enumerable.Skip"/> のリスト専用版です。
        /// </summary>
        /// <typeparam name="TSource">リストの要素の型です。</typeparam>
        /// <param name="source">入力のリストです。</param>
        /// <returns>リストの先頭から指定した数の要素を除いた残りのシーケンスです。</returns>
        /// <exception cref="ArgumentNullException">
        /// 引数 source が null です。
        /// </exception>
        /// <exception cref="InvalidOperationException">
        /// シーケンスの走査中に list が変更されました。
        /// </exception>
        public static IEnumerable<TSource> ListSkip<TSource>(this IList<TSource> list, int count)
        {
            Ensure.Argument.NotNull(list, nameof(list));

            return ListSkipImpl(list, count);
        }

        private static IEnumerable<TSource> ListSkipImpl<TSource>(IList<TSource> list, int count)
        {
            // NOTE: list が変更されたときは InvalidOperationException がスローされます。
            using (var e = list.GetEnumerator())
            {
                for (int i = Math.Max(0, count); i < list.Count && e.MoveNext(); i++)
                {
                    yield return list[i];
                }
            }
        }
    }
}
