﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Text;

namespace NintendoWare.Spy.Binary
{
    internal class SpyDataBinaryWriter : ObservableObject
    {
        private static readonly Version ContainerVersion = SpyDataBinary.ContainerVersion_3_2_0_0;
        private static readonly ushort Bom = 0xfeff;

        //-----------------------------------------------------------------

        private BinaryWriter _writer;
        private long _totalDataBlockLengthPosition;
        private long _headerLength;
        private long _totalDataBlockLength;

        //-----------------------------------------------------------------

        public bool IsOpened
        {
            get { return _writer != null; }
        }

        //-----------------------------------------------------------------

        public void Open(Stream stream, bool isLittleEndian)
        {
            Ensure.Argument.NotNull(stream);
            Ensure.Operation.Null(_writer);

            _writer = isLittleEndian ?
                LittleEndianBinaryWriter.Create(stream, Encoding.ASCII) :
                BigEndianBinaryWriter.Create(stream, Encoding.ASCII);

            _totalDataBlockLengthPosition = 0;
            _totalDataBlockLength = 0;
        }

        public void Close()
        {
            this.Dispose();
        }

        public long GetNextBlockPosition()
        {
            Ensure.Operation.True(this.IsOpened);

            // パフォーマンス向上のため、BinaryWriter.BaseStream.Position 呼び出しを行わずに、次のブロック書き込み位置を計算する
            return _headerLength + _totalDataBlockLength;
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        public void WriteHeader(string dataName, Version dataVersion)
        {
            Ensure.Operation.True(this.IsOpened);
            Ensure.Argument.NotNull(dataName);
            Ensure.Argument.NotNull(dataVersion);

            var nameBytes = Encoding.ASCII.GetBytes(dataName);
            Ensure.Operation.True(nameBytes.Length > 0);

            _writer.Write(SpyDataBinary.Signature);
            _writer.Write((Byte)ContainerVersion.Major);
            _writer.Write((Byte)ContainerVersion.Minor);
            _writer.Write((Byte)ContainerVersion.Build);
            _writer.Write((Byte)ContainerVersion.Revision);
            _writer.Write((UInt16)Bom);

            _writer.Write((UInt16)nameBytes.Length);
            _writer.Write((Byte)dataVersion.Major);
            _writer.Write((Byte)dataVersion.Minor);
            _writer.Write((Byte)dataVersion.Build);
            _writer.Write((Byte)dataVersion.Revision);

            // ブロック開始位置へのオフセット
            var nameLengthPadded = (nameBytes.Length + 3) & ~3;
            _writer.Write((UInt32)(sizeof(UInt32) /* dataBlockOffset */ + sizeof(Int64) /* totalDataBlockLength */ + nameLengthPadded));

            // 全ブロックの合計サイズ
            // 最後に書き戻す
            _totalDataBlockLengthPosition = _writer.BaseStream.Position;
            _writer.Write((Int64)0);

            // データ名
            _writer.Write(nameBytes);
            for (int i = nameBytes.Length; i < nameLengthPadded; ++i)
            {
                _writer.Write((byte)0);
            }

            _headerLength = _writer.BaseStream.Position;
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        public long WriteBlock(long dataBlockID, byte[] rawData, long timestamp)
        {
            Ensure.Operation.True(this.IsOpened);
            Assertion.Argument.NotNull(rawData);

            var result = this.GetNextBlockPosition();

            _writer.Write((Int64)timestamp);
            _writer.Write((Int64)dataBlockID);
            _writer.Write((UInt32)rawData.Length);
            _writer.Write(rawData);

            _totalDataBlockLength += sizeof(Int64) + sizeof(Int64) + sizeof(UInt32) + (uint)rawData.Length;

            return result;
        }

        public void WriteTotalDataBlockLength()
        {
            Ensure.Operation.True(this.IsOpened);

            // 全ブロックサイズを書き戻す
            Ensure.Operation.True(_totalDataBlockLengthPosition > 0);

            long current = _writer.BaseStream.Position;
            _writer.Seek((int)_totalDataBlockLengthPosition, SeekOrigin.Begin);
            _writer.Write(_totalDataBlockLength);

            _writer.Flush();
            _writer.BaseStream.Seek(current, SeekOrigin.Begin);
        }

        protected override void Dispose(bool disposing)
        {
            if (_writer != null)
            {
                this.WriteTotalDataBlockLength();

                _writer.Dispose();
                _writer = null;
            }

            base.Dispose(disposing);
        }
    }
}
