﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Commands;
using NintendoWare.Spy.Foundation;
using NintendoWare.Spy.Foundation.Commands;
using NintendoWare.Spy.Framework;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using NintendoWare.Spy.Windows.Input;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Windows.Input;
using System.Windows.Threading;

namespace NintendoWare.Spy
{
    internal sealed class RootPresenter : ApplicationPresenter
    {
        private readonly Lazy<MainWindowPresenter> _mainWindowPresenter = new Lazy<MainWindowPresenter>();

        private DispatcherTimer _timer;

        private bool _isConnecting = false;
        private string _targetPlatformName = string.Empty;
        private object _syncPort;
        private object _dataPort;

        //-----------------------------------------------------------------

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public override IEnumerable<Type> DependentServiceTypes
        {
            get
            {
                yield return typeof(ApplicationInfoService);
                yield return typeof(PluginsService);
                yield return typeof(SpyService);
                yield return typeof(SpyPlaybackService);
                yield return typeof(SpyTemporaryDataService);
                yield return typeof(SettingsService);
                yield return typeof(TeamSettingsService);
            }
        }

        /// <summary>
        /// 現在、接続しようとしているかどうかを取得します。
        /// </summary>
        private bool CurrentIsConnecting
        {
            get { return this.GetSettingsService().GetConnectionSettings().IsConnecting; }
        }

        /// <summary>
        /// 現在の対象プラットフォーム名を取得します。
        /// </summary>
        private string CurrentTargetPlatformName
        {
            get { return this.GetSettingsService().GetConnectionSettings().TargetPlatformName; }
        }

        /// <summary>
        /// 現在の対象プラットフォームの SpyConnectionPlugin を取得します。
        /// </summary>
        /// <returns></returns>
        private SpyConnectionPlugin CurrentSpyConnectionPlugin
        {
            get
            {
                var connectionPlugin = this.GetPluginsService().ConnectionPlugins
                    .Where(it => string.Equals(it.TargetPlatformName, this.CurrentTargetPlatformName))
                    .FirstOrDefault();

                if (connectionPlugin == null)
                {
                    Assertion.Fail("unknown target platform.");
                    throw new Exception("unknown target platform.");
                }

                return connectionPlugin;
            }
        }

        /// <summary>
        /// 現在の Sync ポートを取得します。
        /// </summary>
        private object CurrentSyncPort
        {
            get
            {
                return this.CurrentSpyConnectionPlugin.GetSyncPort(this.GetSettingsService().GetConnectionSettings());
            }
        }

        /// <summary>
        /// 現在の Data ポートを取得します。
        /// </summary>
        private object CurrentDataPort
        {
            get
            {
                return this.CurrentSpyConnectionPlugin.GetDataPort(this.GetSettingsService().GetConnectionSettings());
            }
        }

        /// <summary>
        /// 現在の出力先フォルダを取得します。
        /// </summary>
        private string CurrentOutputDirectoryPath { get; set; }

        //-----------------------------------------------------------------

        protected override void OnInitialize()
        {
            base.OnInitialize();

            this.SetupSettings();

            if (UISettingsAccessor.GetWindowsSettings(GetSettingsService()).IsMouseTiltEnabled)
            {
                MouseTilt.Initialize();
            }

            _mainWindowPresenter.Value.Initialize(this.ServiceProvider, this.GetCommandService());

            this.GetSpyService().StateChanged += this.OnSessionStateChanged;
            this.GetSettingsService().Applied += this.OnSettingsApplied;

            this.UpdateConnectionBehavior();
        }

        protected override void OnUninitialize()
        {
            if (_timer != null)
            {
                _timer.Stop();
                _timer = null;
            }

            this.StopSession();
            this.GetSpyService().StateChanged -= this.OnSessionStateChanged;
            this.GetSettingsService().Applied -= this.OnSettingsApplied;

            // 作業ディレクトリを開いていた場合にはロックを解放します。
            // アプリケーションの終了時なので、古いデータディレクトリの整理はしません。
            this.GetSpyService().ClearAll();

            _mainWindowPresenter.Value.Uninitialize();

            this.UnsetupSettings();

            base.OnUninitialize();
        }

        protected override void OnBindCommands(CommandService commandService)
        {
            base.OnBindCommands(commandService);

            this.BindSpyCommands(commandService);
        }

        protected override object OnRun()
        {
            return _mainWindowPresenter.Value.Run();
        }

        [DebuggerStepThrough]
        private ApplicationInfoService GetApplicationInfoService()
        {
            return this.ServiceProvider.GetService<ApplicationInfoService>();
        }

        [DebuggerStepThrough]
        private SettingsService GetSettingsService()
        {
            return this.ServiceProvider.GetService<SettingsService>();
        }

        [DebuggerStepThrough]
        private SpyService GetSpyService()
        {
            return this.ServiceProvider.GetService<SpyService>();
        }

        [DebuggerStepThrough]
        private PluginsService GetPluginsService()
        {
            return this.ServiceProvider.GetService<PluginsService>();
        }

        private void BindSpyCommands(CommandService commandService)
        {
            commandService.BindInheritableCommand<ConnectCommandArgs>(
                SpyCommands.Connect,
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    args.TargetPlatformName = this.CurrentTargetPlatformName;
                    args.SyncPort = this.CurrentSyncPort;
                    args.DataPort = this.CurrentDataPort;
                    return SpyConnectHandler.CanExecute(args);
                },
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    args.TargetPlatformName = this.CurrentTargetPlatformName;
                    args.SyncPort = this.CurrentSyncPort;
                    args.DataPort = this.CurrentDataPort;
                    return SpyConnectHandler.Execute(args);
                });

            commandService.BindInheritableCommand<SpyServiceCommandArgs>(
                SpyCommands.Disconnect,
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    return SpyDisconnectHandler.CanExecute(args);
                },
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    return SpyDisconnectHandler.Execute(args);
                });

            commandService.BindInheritableCommand<RecordAllFilesCommandArgs>(
                SpyCommands.RecordAllFiles,
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    args.OutputDirectoryPath = this.CurrentOutputDirectoryPath;
                    return SpyRecordAllFilesHandler.CanExecute(args);
                },
                (command, args) =>
                {
                    args.SpyService = this.GetSpyService();
                    args.OutputDirectoryPath = this.CurrentOutputDirectoryPath;
                    var result = SpyRecordAllFilesHandler.Execute(args);
                    if (result.IsSucceeded && result.ReturnValue is string)
                    {
                        this.CurrentOutputDirectoryPath = (string)result.ReturnValue;
                    }
                    return result;
                });
        }

        private string GetAppDataPath()
        {
            string appData = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
            string companyName = ConstConfig.AppDataCompanyName;
            string appName = Assembly.GetEntryAssembly().GetName().Name;

            return Path.Combine(appData, companyName, appName);
        }

        private string GetSettingFilePath()
        {
            // ユーザ毎の設定ファイルは %APPDATA% に置かれます。
            string dirName = GetAppDataPath();
            string fileName = Path.ChangeExtension(Path.GetFileName(Assembly.GetEntryAssembly().Location), ConstConfig.SettingsFileExtention);
            return Path.Combine(dirName, fileName);
        }

        private string GetDefaultSettingFilePath()
        {
            // デフォルト設定ファイルは実行ファイルと同じパスに置かれます。
            return Path.ChangeExtension(Assembly.GetEntryAssembly().Location, ConstConfig.SettingsFileExtention);
        }

        private IEnumerable<string> GetSettingFiles()
        {
            yield return GetSettingFilePath();
            yield return GetDefaultSettingFilePath();
        }

        private void SetupSettings()
        {
            foreach (var filePath in GetSettingFiles())
            {
                if (File.Exists(filePath))
                {
                    try
                    {
                        using (var file = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                        {
                            this.GetSettingsService().Load(file);
                        }

                        break;
                    }
                    catch
                    {
                    }
                }
            }

            this.GetSettingsService().Applied += this.SettingsChanged;

            this.GetSettingsService().Apply();
        }

        private void UnsetupSettings()
        {
            this.GetSettingsService().Applied -= this.SettingsChanged;

            var filePath = GetSettingFilePath();

            try
            {
                string directory = Path.GetDirectoryName(filePath);
                if (Directory.Exists(directory) == false)
                {
                    Directory.CreateDirectory(directory);
                }

                using (var file = File.Open(filePath, FileMode.Create, FileAccess.ReadWrite, FileShare.None))
                {
                    this.GetSettingsService().Save(file);
                }
            }
            catch
            {
                if (File.Exists(filePath))
                {
                    File.Delete(filePath);
                }
            }
        }

        private void SettingsChanged(object sender, EventArgs e)
        {
            this.UpdateConnectionBehavior();
        }

        private void UpdateConnectionBehavior()
        {
            if (_isConnecting == this.CurrentIsConnecting &&
                _targetPlatformName == this.CurrentTargetPlatformName &&
                _syncPort != null && _syncPort.Equals(this.CurrentSyncPort) &&
                _dataPort != null && _dataPort.Equals(this.CurrentDataPort))
            {
                return;
            }

            this.StopSession();

            if (_timer != null)
            {
                _timer.Stop();
                _timer = null;
            }

            var connectionSetting = this.GetSettingsService().GetConnectionSettings();

            if (connectionSetting.IsConnecting == true)
            {
                if (this.GetSpyService().LockConnectionMutex() == false)
                {
                    _isConnecting = false;
                    connectionSetting.IsConnecting = false;
                    return;
                }
            }
            else
            {
                this.GetSpyService().ReleaseConnectionMutex();
            }

            if (!connectionSetting.IsConnecting || string.IsNullOrEmpty(connectionSetting.TargetPlatformName))
            {
                return;
            }

            // 選択されていたプラットフォームのプラグインが読み込まれていない場合。
            if (!this.GetPluginsService()
                .ConnectionPlugins
                .Any(it => it.TargetPlatformName == connectionSetting.TargetPlatformName))
            {
                return;
            }

            this.StartSession();

            // HACK : 定期的に接続を試みる（暫定）
            _timer = new DispatcherTimer(
                TimeSpan.FromMilliseconds(3000),
                DispatcherPriority.Input,
                (sender, e) =>
                {
                    if (_timer == null) { return; }

                    // 接続中、または、すでにデータを受信済みなら再接続しない
                    if (this.GetSpyService().State == SpyService.ServiceState.Running &&
                        this.GetSpyService().GetCurrentData().Models.Count > 0)
                    {
                        _timer.Stop();
                        _timer = null;
                        return;
                    }

                    this.StartSession();
                },
                Dispatcher.CurrentDispatcher);
        }

        private void StartSession()
        {
            Ensure.Operation.NotNull(this.GetSpyService());

            if (this.GetSpyService().State > SpyService.ServiceState.NotStarted)
            {
                return;
            }

            try
            {
                this.GetSpyService().Start(
                    this.CurrentTargetPlatformName,
                    this.CurrentSyncPort,
                    this.CurrentDataPort);

                _isConnecting = true;
                _targetPlatformName = this.CurrentTargetPlatformName;
                _syncPort = this.CurrentSyncPort;
                _dataPort = this.CurrentDataPort;
            }
            catch
            {
                return;
            }
        }

        private void StopSession()
        {
            _isConnecting = false;
            _targetPlatformName = string.Empty;
            _syncPort = null;
            _dataPort = null;

            this.GetSpyService().Stop();
        }

        private void OnSessionStateChanged(object sender, EventArgs e)
        {
            // 接続中、または、すでにデータを受信済みなら再接続しない
            if (this.GetSpyService().State == SpyService.ServiceState.NotStarted &&
                this.GetSpyService().GetCurrentData().Models.Count > 0)
            {
                this.StopSession();
                this.GetSettingsService().GetConnectionSettings().IsConnecting = false;
                this.GetSettingsService().Apply();
            }

            // HACK: いらんかも
            CommandManager.InvalidateRequerySuggested();
        }

        private void OnSettingsApplied(object sender, EventArgs e)
        {
            this.UpdateConnectionBehavior();

            // HACK: いらんかも
            CommandManager.InvalidateRequerySuggested();
        }
    }
}
