﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Windows.Markup;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// プロットアイテムのビューモデルです。
    /// </summary>
    [ContentProperty("Values")]
    public class TimelineChartItemViewModel : ObservableObject, ILineChartItemData
    {
        private readonly object _observerOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private readonly MergedRequestDispatcher _requestNotifySamplesChanged = new MergedRequestDispatcher();

        private readonly ObservableList<TimelineChartValue> _values = new ObservableList<TimelineChartValue>();
        private bool _isVisible = true;
        private double _currentValue;

        protected TimelineChartItemViewModel()
        {
            // ファイルからの読み出し時にイベントが大量に発生するのを防ぎます。
            CollectionChangedObservation.GetObserver(_observerOwner, _values).AddHandler(
                (sender, e) => _requestNotifySamplesChanged.Request(this.NotifySamplesChanged));
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="spyService">SpyService を指定します。</param>
        /// <param name="playbackService">SpyPlaybackService を指定します。</param>
        /// <param name="model">プロットアイテムモデルを指定します。</param>
        /// <param name="timeUnit">時間の表示単位を指定します。</param>
        public TimelineChartItemViewModel(
            SpyService spyService,
            SpyPlaybackService playbackService,
            PlotSpyModel.PlotFloat model,
            string name,
            Color color,
            LineChartDrawType drawType,
            TimelineChartValue.ValueMode valueMode)
            : this()
        {
            Assertion.Argument.NotNull(spyService);
            Assertion.Argument.NotNull(playbackService);
            Assertion.Argument.NotNull(model);

            _spyService = spyService;
            _playbackService = playbackService;

            this.Name = name;
            this.Color = color;
            this.DrawType = drawType;
            this.ValueMode = valueMode;

            this.MinimumY = model.Minimum;
            this.MaximumY = model.Maximum;
            this.Interpolation = ConvertToLineChartInterpolation(model.InterpolationMode);

            PropertyChangedObservation.GetObserver(_observerOwner, _playbackService).AddHandler(
                target => target.Begin,
                (sender, e) => this.NotifyPropertyChanged(() => this.MinimumX));

            PropertyChangedObservation.GetObserver(_observerOwner, _playbackService).AddHandler(
                target => target.End,
                (sender, e) => this.NotifyPropertyChanged(() => this.MaximumX));

            PropertyChangedObservation.GetObserver(_observerOwner, _playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateCurrentFrame());

            this.UpdateCurrentFrame();
        }

        public event EventHandler SamplesChanged;

        public string Name { get; protected set; }

        public double CurrentValue
        {
            get { return _currentValue; }
            private set { this.SetPropertyValue(ref _currentValue, value); }
        }

        public ObservableCollection<TimelineChartValue> Values { get { return _values; } }

        public bool IsVisible
        {
            get { return _isVisible; }
            set { this.SetPropertyValue(ref _isVisible, value); }
        }

        public LineChartDrawType DrawType { get; private set; }

        public Color Color
        {
            get;
            protected set;
        }

        public LineChartInterpolation Interpolation
        {
            get;
            protected set;
        }

        public double MinimumX
        {
            get
            {
                if (_playbackService == null)
                {
                    return 0;
                }

                return _playbackService.Begin.GetMicroSeconds();
            }
        }

        public double MaximumX
        {
            get
            {
                if (_playbackService == null)
                {
                    return 0;
                }

                return _playbackService.End.GetMicroSeconds();
            }
        }

        public double MinimumY { get; set; }

        public double MaximumY { get; protected set; }

        public double CenterY
        {
            get { return this.MinimumY; }
        }

        public TimelineChartValue.ValueMode ValueMode { get; private set; }

        public string ToolTip { get; set; }

        public IEnumerable<LineChartSample> EnumerateSamples(double beginX, double endX, double scale)
        {
            switch (this.ValueMode)
            {
                case TimelineChartValue.ValueMode.Scalar:
                    return EnumerateSamplesValue(beginX, endX, scale);

                case TimelineChartValue.ValueMode.MinMax:
                    return EnumerateSamplesMinMax(beginX, endX, scale);

                default:
                    throw new NotImplementedException();
            }
        }

        private IEnumerable<LineChartSample> EnumerateSamplesValue(double beginX, double endX, double scale)
        {
            var currentSampleIndex = this.FindSampleSmallestIndex(beginX);

            if (currentSampleIndex >= _values.Count)
            {
                yield break;
            }

            if (currentSampleIndex < 0)
            {
                // beginX がサンプル前方に範囲外の場合、先頭サンプルから処理します
                currentSampleIndex = 0;
            }

            var currentXBegin = (double)_values[currentSampleIndex].Time.GetMicroSeconds();
            var currentXEnd = scale > 1.0
                ? currentXBegin / scale * scale + scale
                : currentXBegin + 1;

            Range1 rangeX = new Range1(currentXBegin, currentXBegin);
            Range1 rangeY = new Range1(
                (double)_values[currentSampleIndex].Value,
                (double)_values[currentSampleIndex].Value);

            while (true)
            {
                // 指定範囲を超えたら、サンプルを返して、ループ終了
                // スクリーン領域外（右側）のサンプルも描画対象にする
                if (endX < currentXBegin)
                {
                    yield return new LineChartSample(rangeX, rangeY);
                    break;
                }

                currentSampleIndex++;

                // すべての plotValue を確認したら、サンプルを返してループ終了
                if (_values.Count <= currentSampleIndex)
                {
                    yield return new LineChartSample(rangeX, rangeY);
                    break;
                }

                var nextXBegin = (double)_values[currentSampleIndex].Time.GetMicroSeconds();

                // 次のサンプルが、同一ピクセルに含めるべきなら、Range に含める
                // 次のピクセルに含めるべきサンプルなら、今のサンプルを yield return して、次のサンプル取得を待つ
                if (nextXBegin < currentXEnd)
                {
                    rangeX.Max = currentXBegin;
                    rangeY.Min = Math.Min(rangeY.Min, (double)_values[currentSampleIndex].Value);
                    rangeY.Max = Math.Max(rangeY.Max, (double)_values[currentSampleIndex].Value);
                }
                else
                {
                    yield return new LineChartSample(rangeX, rangeY);

                    currentXBegin = nextXBegin;
                    currentXEnd = scale > 1.0
                        ? currentXBegin / scale * scale + scale
                        : currentXBegin + 1;

                    rangeX = new Range1(currentXBegin, currentXBegin);
                    rangeY = new Range1(
                        (double)_values[currentSampleIndex].Value,
                        (double)_values[currentSampleIndex].Value);
                }
            }
        }

        private IEnumerable<LineChartSample> EnumerateSamplesMinMax(double beginX, double endX, double scale)
        {
            beginX = Math.Floor(beginX / scale) * scale;

            var currentSampleIndex = this.FindSampleSmallestIndex(beginX);

            if (currentSampleIndex >= _values.Count)
            {
                yield break;
            }

            if (currentSampleIndex < 0)
            {
                // beginX がサンプル前方に範囲外の場合、先頭サンプルから処理します
                currentSampleIndex = 0;
            }

            // beginX～endX までを scale の間隔で表示データを作ると LineChart の表示で隙間が入る問題があります。
            // scale の間隔を微妙に減らして、表示データの個数をピクセル数より多く詰め込むようにすると回避できます。
            var interval = scale * 0.9999999;

            var value = _values[currentSampleIndex];
            var firstX = Math.Floor(Math.Max(beginX, (double)value.Time.GetMicroSeconds()) / interval) * interval;
            var rangeY = new Range1(value.MinValue, value.MaxValue);

            for (var x = firstX; x <= endX; x += interval)
            {
                if (currentSampleIndex >= _values.Count)
                {
                    break;
                }

                value = _values[currentSampleIndex];
                var valueX = (double)value.Time.GetMicroSeconds();

                if (valueX < x + interval)
                {
                    var minY = value.MinValue;
                    var maxY = value.MaxValue;
                    currentSampleIndex++;

                    while (currentSampleIndex < _values.Count)
                    {
                        value = _values[currentSampleIndex];
                        valueX = (double)value.Time.GetMicroSeconds();
                        if (valueX >= x + interval)
                        {
                            break;
                        }

                        minY = Math.Min(minY, value.MinValue);
                        maxY = Math.Max(maxY, value.MaxValue);
                        currentSampleIndex++;
                    }

                    rangeY = new Range1(minY, maxY);
                }

                // rangeX.Length >= 1 にすることで LineChart を MinMax モードで描画させます。
                // rangeX = new Range1(x, x + 1) だと、計算誤差により rangeX.Length < 1 になってしまう場合があります。
                var rangeX = new Range1(x, x + 2);
                yield return new LineChartSample(rangeX, rangeY);
            }
        }

        public double Evaluate(double x)
        {
            var valueData = this.GetValueData(x);
            return valueData != null ? valueData.Value : double.NaN;
        }

        public TimelineChartValue GetChartValue(double x)
        {
            var valueData = this.GetValueData(x);

            if (valueData == null && _spyService.State == SpyService.ServiceState.Running)
            {
                if (_values.Count > 0 && _values[_values.Count - 1].Time.GetMicroSeconds() < x)
                {
                    valueData = _values[_values.Count - 1];
                }
            }

            return valueData;
        }

        protected override void DisposeManagedInstance()
        {
            base.DisposeManagedInstance();

            PropertyChangedObservation.RemoveObservers(_observerOwner);
            CollectionChangedObservation.RemoveObservers(_observerOwner);
        }

        private TimelineChartValue GetValueData(double x)
        {
            if (!this.IsVisible)
            {
                return null;
            }

            if (_values.Count == 0)
            {
                return null;
            }

            if ((double)_values[_values.Count - 1].Time.GetMicroSeconds() < x)
            {
                return null;
            }

            var currentSampleIndex = this.FindSampleBiggestIndex(x);

            if (currentSampleIndex < 0)
            {
                return null;
            }

            if (currentSampleIndex >= _values.Count)
            {
                return null;
            }

            return _values[currentSampleIndex];
        }

        private void UpdateCurrentFrame()
        {
            var x = _playbackService.Current.GetMicroSeconds();
            var valueData = this.GetChartValue(x);
            this.CurrentValue = valueData != null ? valueData.Value : double.NaN;
        }

        /// <summary>
        /// 指定フレームに近いサンプルインデックスを検索します
        /// 指定フレームに複数のサンプルがあるときは最初のサンプルを検索します。
        /// </summary>
        /// <param name="x">現在の時間単位におけるフレームを指定します。</param>
        /// <returns>
        /// <list type="bullet">
        /// <item>指定フレームに最も近いサンプルのインデックスを返します。</item>
        /// <item>指定フレーム &lt; サンプル先頭の場合、-1 を返します。</item>
        /// <item>指定フレーム &gt; サンプル末尾の場合、サンプル数（リストの要素数）を返します。</item>
        /// </list>
        /// </returns>
        private int FindSampleSmallestIndex(double x)
        {
            if (_values.Count == 0)
            {
                return 0;
            }

            // 二分探索して一致した場合
            // ・・・そのインデックスを返す
            // 一致しなかった場合
            // ・・・範囲内なら、BinarySearch() が返した補数を解決して、指定フレーム以下の最も近いインデックスを返す
            // ・・・範囲外なら、そのままサンプル数（リストの要素数）を返す
            var result = BinarySearchUtility.BinarySearch(_values, x, value => (double)value.Time.GetMicroSeconds(), BinarySearchUtility.Options.SmallestIndex);

            if (result >= 0)
            {
                return result;
            }

            result = ~result;

            return result < _values.Count ? result - 1 : result;
        }

        /// <summary>
        /// 指定フレームに近いサンプルインデックスを検索します。
        /// 指定フレームに複数のサンプルがあるときは最後のサンプルを検索します。
        /// </summary>
        /// <param name="x">現在の時間単位におけるフレームを指定します。</param>
        /// <returns>
        /// <list type="bullet">
        /// <item>指定フレームに最も近いサンプルのインデックスを返します。</item>
        /// <item>指定フレーム &lt; サンプル先頭の場合、-1 を返します。</item>
        /// <item>指定フレーム &gt; サンプル末尾の場合、サンプル数（リストの要素数）を返します。</item>
        /// </list>
        /// </returns>
        private int FindSampleBiggestIndex(double x)
        {
            if (_values.Count == 0)
            {
                return 0;
            }

            // 二分探索して一致した場合
            // ・・・そのインデックスを返す
            // 一致しなかった場合
            // ・・・範囲内なら、BinarySearch() が返した補数を解決して、指定フレーム以下の最も近いインデックスを返す
            // ・・・範囲外なら、そのままサンプル数（リストの要素数）を返す
            var result = BinarySearchUtility.BinarySearch(_values, x, value => (double)value.Time.GetMicroSeconds(), BinarySearchUtility.Options.BiggestIndex);

            if (result >= 0)
            {
                return result;
            }

            result = ~result;

            return result < _values.Count ? result - 1 : result;
        }

        private void NotifySamplesChanged()
        {
            if (this.SamplesChanged != null)
            {
                this.SamplesChanged(this, EventArgs.Empty);
            }
        }

        private static LineChartInterpolation ConvertToLineChartInterpolation(PlotSpyModel.PlotFloatInterpolationMode interpolation)
        {
            switch (interpolation)
            {
                case PlotSpyModel.PlotFloatInterpolationMode.Linear:
                    return LineChartInterpolation.Linear;

                default:
                    return LineChartInterpolation.None;
            }
        }
    }
}
