﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Media;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// プロットステートのビューモデルです。
    /// </summary>
    public class PlotStateData : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private readonly ObservableList<PlotStateValueData> _values = new ObservableList<PlotStateValueData>();
        private IList<PlotSpyModel.PlotStateValue> _valuesSource;
        private Color _currentColor = Colors.Black;
        private string _currentValue = string.Empty;
        private bool _isVisible = true;
        private double _currentFrame;
        private int _valuesCount;
        private readonly MergedRequestDispatcher _addNewValueRequest = new MergedRequestDispatcher();

        public PlotStateData(PlotSpyModel.PlotState model)
        {
            Assertion.Argument.NotNull(model);

            this.Name = model.Name;
            this.Color = model.Color;
            this.ValuesSource = model.Values;
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            base.DisposeManagedInstance();
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set
            {
                if (this.SetPropertyValue(ref _currentFrame, value))
                {
                    this.UpdateCurrent();
                }
            }
        }

        public string Name { get; set; }

        public Color Color { get; set; }

        public ObservableCollection<PlotStateValueData> Values { get { return _values; } }

        /// <summary>
        /// Values のソースです。
        /// ValuesSource の内容は Values に追加されます。
        /// </summary>
        public IList<PlotSpyModel.PlotStateValue> ValuesSource
        {
            get { return _valuesSource; }
            set
            {
                if (this.SetPropertyValue(ref _valuesSource, value))
                {
                    CollectionChangedObservation.RemoveObservers(_observerOwner);
                    _valuesCount = 0;

                    CollectionChangedObservation.GetObserver(_observerOwner, value).AddHandlerForAddItems(
                        (sender, e) =>
                        {
                            this.RequestAddNewValue();
                        });

                    this.AddNewValue();
                }
            }
        }

        public Color CurrentColor
        {
            get { return _currentColor; }
            private set { this.SetPropertyValue(ref _currentColor, value); }
        }

        public string CurrentValue
        {
            get { return _currentValue; }
            private set { this.SetPropertyValue(ref _currentValue, value); }
        }

        public bool IsVisible
        {
            get { return _isVisible; }
            set { this.SetPropertyValue(ref _isVisible, value); }
        }

        private void RequestAddNewValue()
        {
            _addNewValueRequest.Request(AddNewValue, DispatcherPriority.Background);
        }

        private void AddNewValue()
        {
            if (_valuesSource != null)
            {
                _valuesSource.ListSkip(_valuesCount).ForEach(it => this.AddNewValue(it));
                _valuesCount = _valuesSource.Count;
            }
        }

        private void AddNewValue(PlotSpyModel.PlotStateValue model)
        {
            if (!this.Values.IsEmpty())
            {
                var last = this.Values.Last();

                if (model.Time.Timestamp < last.BeginTime.Timestamp)
                {
                    Assertion.Operation.Fail();
                    return;
                }

                last.SetEndTime(model.Time);
            }

            this.Values.Add(new PlotStateValueData(model));
        }

        private void UpdateCurrent()
        {
            var currentFrame = this.CurrentFrame;
            var index = BinarySearchUtility.BinarySearch(_values, currentFrame, it => it.LeftX);
            if (index < 0)
            {
                index = ~index - 1;
            }

            if (index >= 0)
            {
                var currentData = _values[index];

                if (currentData.LeftX <= currentFrame && currentFrame < currentData.RightX)
                {
                    this.CurrentColor = currentData.Color;
                    this.CurrentValue = currentData.Value;
                    return;
                }
            }

            this.CurrentColor = Colors.Black;
            this.CurrentValue = string.Empty;
        }
    }
}
