﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Input;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows
{
    public class PlotPanelViewModel : ObservableObject
    {
        private const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。

        private readonly object _plotObserverOwner = new object();
        private readonly object _markerObserverOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;
        private MarkerSpyModel _markerModel;
        private FrameSyncSpyModel _frameSyncModel;

        private readonly ObservableKeyedList<string, PlotNodeData> _plotNodes = new ObservableKeyedList<string, PlotNodeData>(node => node.Name);
        private readonly ObservableList<MarkerViewModel> _markers = new ObservableList<MarkerViewModel>();

        private IList<PlotNodeData> _filteredNodes;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private double _currentFrame;
        private double _minimumFrame = double.NaN;
        private double _maximumFrame = double.NaN;
        private string _itemNameFilterText;
        private string _nodeNameFilterText;
        private double _scaleX = 0.001;
        private double _viewportLeft;
        private string _error = null;

        private readonly MergedRequestDispatcher _addNewPlotItemsRequest = new MergedRequestDispatcher();
        private int _floatsCount;
        private int _statesCount;

        public PlotPanelViewModel()
        {
            _filteredNodes = _plotNodes;
        }

        public PlotPanelViewModel(SpyService spyService, SpyPlaybackService playbackService)
            : this()
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(playbackService);

            _spyService = spyService;
            _playbackService = playbackService;

            this.CheckAllCommand = new DelegateCommand(
                param =>
                {
                    _plotNodes.ForEach(node => node.CheckAllFloats());
                },
                param => _plotModel != null);

            this.UncheckAllCommand = new DelegateCommand(
                param =>
                {
                    _plotNodes.ForEach(node => node.UncheckAllFloats());
                },
                param => _plotModel != null);

            this.ClearFilterCommand = new DelegateCommand(
                param => this.ItemNameFilterText = string.Empty,
                param => !string.IsNullOrEmpty(this.ItemNameFilterText));

            this.CheckNodeAllCommand = new DelegateCommand(
                param =>
                {
                    _plotNodes.ForEach(node => node.IsChecked = true);
                },
                param => _plotModel != null);

            this.UncheckNodeAllCommand = new DelegateCommand(
                param =>
                {
                    _plotNodes.ForEach(node => node.IsChecked = false);
                },
                param => _plotModel != null);

            this.ClearNodeFilterCommand = new DelegateCommand(
                param => this.NodeNameFilterText = string.Empty,
                param => !string.IsNullOrEmpty(this.NodeNameFilterText));

            this.DismissErrorCommand = new DelegateCommand(
                param => this.Error = null);
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);
            CollectionChangedObservation.RemoveObservers(_markerObserverOwner);
            PropertyChangedObservation.RemoveObservers(_markerObserverOwner);

            _plotNodes.ForEach(it => it.Dispose());
            _markers.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel; }
            set { this.SetPropertyValue(ref _frameSyncModel, value); }
        }

        public ObservableCollection<PlotNodeData> Nodes
        {
            get { return _plotNodes; }
        }

        public IList<PlotNodeData> FilteredNodes
        {
            get { return _filteredNodes; }
            private set { this.SetPropertyValue(ref _filteredNodes, value); }
        }

        public ObservableCollection<MarkerViewModel> Markers
        {
            get { return _markers; }
        }

        public ICommand CheckAllCommand { get; private set; }

        public ICommand UncheckAllCommand { get; private set; }

        public ICommand ClearFilterCommand { get; private set; }

        public ICommand CheckNodeAllCommand { get; private set; }

        public ICommand UncheckNodeAllCommand { get; private set; }

        public ICommand ClearNodeFilterCommand { get; private set; }

        public ICommand DismissErrorCommand { get; private set; }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set { this.SetCurrentFrameImpl(value, from: this); }
        }

        public double MinimumFrame
        {
            get { return _minimumFrame; }
            set
            {
                if (this.SetPropertyValue(ref _minimumFrame, value))
                {
                    foreach (var node in _plotNodes)
                    {
                        node.MinimumFrame = value;
                    }
                }
            }
        }

        public double MaximumFrame
        {
            get { return _maximumFrame; }
            set
            {
                if (this.SetPropertyValue(ref _maximumFrame, value))
                {
                    foreach (var node in _plotNodes)
                    {
                        node.MaximumFrame = value;
                    }
                }
            }
        }

        public string ItemNameFilterText
        {
            get { return _itemNameFilterText; }
            set
            {
                if (this.SetPropertyValue(ref _itemNameFilterText, value))
                {
                    foreach (var node in _plotNodes)
                    {
                        node.ItemNameFilterText = value;
                    }
                }
            }
        }

        public string NodeNameFilterText
        {
            get { return _nodeNameFilterText; }
            set
            {
                if (this.SetPropertyValue(ref _nodeNameFilterText, value))
                {
                    if (string.IsNullOrEmpty(value))
                    {
                        this.FilteredNodes = _plotNodes;
                    }
                    else
                    {
                        var filterText = value;
                        this.FilteredNodes = new WhereListDecorator<PlotNodeData>(
                            node => node.IsGlobal || node.Name.IndexOf(filterText, StringComparison.OrdinalIgnoreCase) >= 0,
                            _plotNodes as IReadOnlyList<PlotNodeData>,
                            dependentPropertyNames: null);
                    }
                }
            }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set
            {
                value = Math.Min(value, MaxScaleX);
                this.SetPropertyValue(ref _scaleX, value);
            }
        }

        public double ViewportLeft
        {
            get { return _viewportLeft; }
            set { this.SetPropertyValue(ref _viewportLeft, value); }
        }

        public string Error
        {
            get { return _error; }
            set { this.SetPropertyValue(ref _error, value); }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);

            _floatsCount = 0;
            _statesCount = 0;
            _plotModel = model;
            this.SyncPlotItems();

            this.ResetErrorIfNecessary();
        }

        public void SetMarkerModel(MarkerSpyModel model)
        {
            if (_markerModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_markerObserverOwner);
            PropertyChangedObservation.RemoveObservers(_markerObserverOwner);

            _markerModel = model;
            this.SyncMarkers();

            this.ResetErrorIfNecessary();
        }

        public void SetFrameSyncModel(FrameSyncSpyModel model)
        {
            this.FrameSyncModel = model;
        }

        public void SetCurrentFrameFromPresenter(double value)
        {
            this.SetCurrentFrameImpl(value, from: null);
        }

        private void SetCurrentFrameImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.MinimumFrame, this.MaximumFrame);
            }

            if (this.SetPropertyValue(ref _currentFrame, value, nameof(this.CurrentFrame)))
            {
                foreach (var node in _plotNodes)
                {
                    node.CurrentFrame = value;
                }

                if (from == this)
                {
                    if (!double.IsNaN(value))
                    {
                        _playbackService?.SetCurrentTime(SpyGlobalTime.FromMicroSeconds((long)value));
                    }
                }
            }
        }

        private void SyncPlotItems()
        {
            _plotNodes.ForEach(it => it.Dispose());
            _plotNodes.Clear();

            var globalNode = this.CreatePlotNodeData(string.Empty);
            globalNode.IsGlobal = true;

            _plotNodes.Add(globalNode);

            if (_plotModel == null)
            {
                return;
            }

            CollectionChangedObservation.GetObserver(_plotObserverOwner, _plotModel.Floats).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.RequestAddNewPlotItems();
                });

            CollectionChangedObservation.GetObserver(_plotObserverOwner, _plotModel.States).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.RequestAddNewPlotItems();
                });

            this.AddNewPlotItems();

            PropertyChangedObservation.GetObserver(_plotObserverOwner, _plotModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckPlotModelErrorUnexpectedDataVersion();
                });

            this.CheckPlotModelError();
        }

        private void RequestAddNewPlotItems()
        {
            _addNewPlotItemsRequest.Request(AddNewPlotItems, DispatcherPriority.Background);
        }

        private void AddNewPlotItems()
        {
            if (_plotModel == null)
            {
                return;
            }

            this.NewPlotFloats(_plotModel.Floats.ListSkip(_floatsCount));
            _floatsCount = _plotModel.Floats.Count;

            this.NewPlotStates(_plotModel.States.ListSkip(_statesCount));
            _statesCount = _plotModel.States.Count;
        }

        private void SyncMarkers()
        {
            _markers.ForEach(it => it.Dispose());
            _markers.Clear();

            if (_markerModel == null)
            {
                return;
            }

            CollectionChangedObservation.GetObserver(_markerObserverOwner, _markerModel.Markers).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.NewMarkers(
                        e.Items.Cast<KeyValuePair<uint, MarkerSpyModel.Marker>>().Select(item => item.Value));
                });

            this.NewMarkers(_markerModel.Markers.Values);

            PropertyChangedObservation.GetObserver(_markerObserverOwner, _markerModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckMarkerModelErrorUnexpectedDataVersion();
                });

            this.CheckMarkerModelError();
        }

        private void NewPlotFloats(IEnumerable<PlotSpyModel.PlotFloat> floats)
        {
            floats.ForEach(item => this.NewPlotFloat(item));
        }

        private void NewMarkers(IEnumerable<MarkerSpyModel.Marker> markers)
        {
            foreach (var marker in markers)
            {
                // MarkerSpyModel.Marker.Values に要素が追加されたら、_markers に追加
                CollectionChangedObservation.GetObserver(_markerObserverOwner, marker.Values).AddHandlerForAddItems(
                    (sender, e) =>
                    {
                        e.Items.Cast<MarkerSpyModel.MarkerValue>()
                            .ForEach(value => this.NewMarker(value));
                    });

                // 新規 Marker を _markers に追加
                marker.Values.ForEach(value => this.NewMarker(value));
            }
        }

        private void NewPlotFloat(PlotSpyModel.PlotFloat item)
        {
#if !DEBUG
            // リリースビルドでは Spy が内部で使用するアイテムを表示しません。
            if (item.FullName.StartsWith(PlotSpyModel.SpyInternalPrefix))
            {
                return;
            }
#endif

            PlotNodeData nodeData = null;
            if (item.Parent == null)
            {
                nodeData = _plotNodes[0]; // globalNode
            }
            else
            {
                var node = item.Parent;
                if (!_plotNodes.TryGetItem(node.FullName, out nodeData))
                {
                    nodeData = this.CreatePlotNodeData(node.FullName);
                    _plotNodes.Add(nodeData);
                }
            }

            if (nodeData != null)
            {
                nodeData.NewPlotFloat(item);
            }
        }

        private void NewPlotStates(IEnumerable<PlotSpyModel.PlotState> states)
        {
            states.ForEach(it => this.NewPlotState(it));
        }

        private void NewPlotState(PlotSpyModel.PlotState state)
        {
#if !DEBUG
            // リリースビルドでは Spy が内部で使用するアイテムを表示しません。
            if (state.FullName.StartsWith(PlotSpyModel.SpyInternalPrefix))
            {
                return;
            }
#endif

            PlotNodeData nodeData = null;
            if (state.Parent == null)
            {
                nodeData = _plotNodes[0]; // globalNode
            }
            else
            {
                var node = state.Parent;
                if (!_plotNodes.TryGetItem(node.FullName, out nodeData))
                {
                    nodeData = this.CreatePlotNodeData(node.FullName);
                    _plotNodes.Add(nodeData);
                }
            }

            if (nodeData != null)
            {
                nodeData.NewPlotState(state);
            }
        }

        private PlotNodeData CreatePlotNodeData(string name)
        {
            return new PlotNodeData(name)
            {
                ItemNameFilterText = this.ItemNameFilterText,
                MinimumFrame = this.MinimumFrame,
                MaximumFrame = this.MaximumFrame,
                CurrentFrame = this.CurrentFrame,
            };
        }

        private void NewMarker(MarkerSpyModel.MarkerValue value)
        {
            _markers.Add(new MarkerViewModel(value));
        }

        private void AddErrorMessage(string errorMessage)
        {
            if (this.Error == null)
            {
                this.Error = errorMessage;
            }
            else
            {
                this.Error = errorMessage + "\n" + this.Error;
            }
        }

        private void CheckPlotModelErrorUnexpectedDataVersion()
        {
            if (_plotModel != null && _plotModel.ErrorUnexpectedDataVersion)
            {
                this.AddErrorMessage(Resources.Messages.UnexpectedPlotDataVersion);
            }
        }

        private void CheckPlotModelError()
        {
            this.CheckPlotModelErrorUnexpectedDataVersion();
        }

        private void CheckMarkerModelErrorUnexpectedDataVersion()
        {
            if (_markerModel != null && _markerModel.ErrorUnexpectedDataVersion)
            {
                this.AddErrorMessage(Resources.Messages.UnexpectedMarkerDataVersion);
            }
        }

        private void CheckMarkerModelError()
        {
            this.CheckMarkerModelErrorUnexpectedDataVersion();
        }

        private void ResetErrorIfNecessary()
        {
            if (_plotModel == null && _markerModel == null)
            {
                this.Error = null;
            }
        }
    }
}
