﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Windows;
using System.Windows.Interactivity;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// リードオンリーな依存プロパティへのバインディングを行う添付ビヘビアーです。
    /// リードオンリーな依存プロパティはバインディングのターゲットには指定できないため、
    /// 添付ビヘビアのプロパティを中継することでビューモデルとのバインディングを行います。
    /// <code>
    /// &lt;Grid Name="grid"&gt;
    ///     &lt;i:Interaction.Behaviors&gt;
    ///         &lt;local:PropertyProxyBehavior
    ///             Source="{Binding ActualWidth, ElementName=grid}"
    ///             Target="{Binding Width}" /&gt;
    ///     &lt;/i:Interaction.Behaviors&gt;
    /// &lt;/Grid&gt;
    /// </code>
    /// この例では grid 要素の ActualWidth 依存プロパティが DataContext の Width プロパティにバインドします。
    /// </summary>
    public class PropertyProxyBehavior : Behavior<FrameworkElement>
    {
        public static readonly DependencyProperty SourceProperty = DependencyProperty.Register(
            nameof(Source),
            typeof(object),
            typeof(PropertyProxyBehavior),
            new FrameworkPropertyMetadata(
                null,
                (d, e) => Self(d).SourceChanged(e)));

        public static readonly DependencyProperty TargetProperty = DependencyProperty.Register(
            nameof(Target),
            typeof(object),
            typeof(PropertyProxyBehavior),
            new FrameworkPropertyMetadata(
                null,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault));

        public object Source
        {
            get { return GetValue(SourceProperty); }
            set { SetValue(SourceProperty, value); }
        }

        public object Target
        {
            get { return GetValue(TargetProperty); }
            set { SetValue(TargetProperty, value); }
        }

        private void SourceChanged(DependencyPropertyChangedEventArgs e)
        {
            SetCurrentValue(TargetProperty, e.NewValue);
        }

        private static PropertyProxyBehavior Self(object d)
        {
            return (PropertyProxyBehavior)d;
        }
    }
}
