﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Commands;
using NintendoWare.Spy.Foundation.Commands;
using NintendoWare.Spy.Plugins;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Input;

namespace NintendoWare.Spy.Windows
{
    public sealed class MainWindowViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private readonly ObservableCollection<SpyPanelViewModel> _panelViewModels =
            new ObservableCollection<SpyPanelViewModel>();

        private readonly WindowPositionViewModel _position = new WindowPositionViewModel();

        private readonly MainWindowStatusConnectionViewModel _statusConnectionViewModel;
        private readonly MainWindowStatusLoadingViewModel _statusLoadingViewModel;

        private CommandService _commandService;

        // TODO: CommandService 経由で MainWindowPresenter に実行させるようにする
        private ICommand _connectCommand;
        private ICommand _openFileCommand;
        private ICommand _stopOpenFileCommand;
        private ICommand _closeAllFilesCommand;
        private ICommand _playCommand;
        private ICommand _stopCommand;
        private ICommand _playOrStopCommand;
        private ICommand _seekToBeginCommand;
        private ICommand _seekToEndCommand;
        private ICommand _optionCommand;
        private ICommand _saveWaveDataCommand;
        private ICommand _recordCommand;

        private string _title = string.Empty;

        private ICollectionView _connectionInfos;
        private ConnectionInfoViewModel _selectedConnectionInfo;

        private bool _isConnecting = false;
        private bool _isLoading = false;
        private bool _isAutoScrollEnabled = false;
        private bool _isPlaying = false;

        private UIElement _globalTimelineElement;

        private Action<object> _dragOverAction;
        private Action<object> _dropAction;
        private string _currentSaveWaveDataTargetName;
        private string _currentPlaybackTargetName;
        private IEnumerable<string> _waveSourceOutputNames;

        //-----------------------------------------------------------------
        public MainWindowViewModel()
            : this(null)
        {
        }

        public MainWindowViewModel(CommandService commandService)
        {
            _commandService = commandService;

            this.PanelViewModelsView.SortDescriptions.Add(
                new SortDescription(nameof(SpyPanelViewModel.PanelTitle), ListSortDirection.Ascending));

            _statusConnectionViewModel = new MainWindowStatusConnectionViewModel(this);
            _statusLoadingViewModel = new MainWindowStatusLoadingViewModel(this);
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            _panelViewModels.ForEach(it => it.Dispose());
            _position.Dispose();
            _statusConnectionViewModel.Dispose();
            _statusLoadingViewModel.Dispose();

            base.DisposeManagedInstance();
        }

        //-----------------------------------------------------------------

        public WindowPositionViewModel Position
        {
            get { return _position; }
        }

        public string Title
        {
            get { return _title; }
            set { this.SetPropertyValue(ref _title, value); }
        }

        public IList<SpyPanelViewModel> PanelViewModels
        {
            get { return _panelViewModels; }
        }

        public ICollectionView PanelViewModelsView
        {
            get { return CollectionViewSource.GetDefaultView(_panelViewModels); }
        }

        public ICollectionView ConnectionInfos
        {
            get { return _connectionInfos; }
            set { this.SetPropertyValue(ref _connectionInfos, value); }
        }

        public ConnectionInfoViewModel SelectedConnectionInfo
        {
            get { return _selectedConnectionInfo; }

            set
            {
                if (_selectedConnectionInfo == value)
                {
                    return;
                }

                if (_selectedConnectionInfo != null)
                {
                    _selectedConnectionInfo.IsConnected = false;
                }

                _selectedConnectionInfo = value;
                this.NotifyPropertyChanged();

                this.PanelViewModelsView.Filter = item =>
                {
                    var panelViewModel = (SpyPanelViewModel)item;

                    // すべてのプラットフォームで有効
                    if (panelViewModel.SupportedPlatforms.Contains(SpyPanelPlugin.PlatformNames.All))
                    {
                        return true;
                    }

                    // すべての NintendoSDK 対応プラットフォームで有効
                    if (_selectedConnectionInfo.IsForNintendoSDK)
                    {
                        return panelViewModel.SupportedPlatforms.Contains(SpyPanelPlugin.PlatformNames.AllForNintendoSdk);
                    }

                    // 指定プラットフォームで有効
                    if (panelViewModel.SupportedPlatforms.Contains(_selectedConnectionInfo.TargetPlatformName))
                    {
                        return true;
                    }

                    return false;
                };
                this.PanelViewModelsView.Refresh();
            }
        }

        public ICommand ConnectCommand
        {
            get { return _connectCommand; }
            set { this.SetPropertyValue(ref _connectCommand, value); }
        }

        public ICommand OpenFileCommand
        {
            get { return _openFileCommand; }
            set { this.SetPropertyValue(ref _openFileCommand, value); }
        }

        public ICommand StopOpenFileCommand
        {
            get { return _stopOpenFileCommand; }
            set { this.SetPropertyValue(ref _stopOpenFileCommand, value); }
        }

        public ICommand CloseAllFilesCommand
        {
            get { return _closeAllFilesCommand; }
            set { this.SetPropertyValue(ref _closeAllFilesCommand, value); }
        }

        public ICommand RecordCommand
        {
            get { return _recordCommand; }
            set { this.SetPropertyValue(ref _recordCommand, value); }
        }

        public ICommand PlayCommand
        {
            get { return _playCommand; }
            set { this.SetPropertyValue(ref _playCommand, value); }
        }

        public ICommand StopCommand
        {
            get { return _stopCommand; }
            set { this.SetPropertyValue(ref _stopCommand, value); }
        }

        public ICommand PlayOrStopCommand
        {
            get { return _playOrStopCommand; }
            set { this.SetPropertyValue(ref _playOrStopCommand, value); }
        }

        public ICommand SeekToBeginCommand
        {
            get { return _seekToBeginCommand; }
            set { this.SetPropertyValue(ref _seekToBeginCommand, value); }
        }

        public ICommand SeekToEndCommand
        {
            get { return _seekToEndCommand; }
            set { this.SetPropertyValue(ref _seekToEndCommand, value); }
        }

        public ICommand OptionCommand
        {
            get { return _optionCommand; }
            set { this.SetPropertyValue(ref _optionCommand, value); }
        }

        public ICommand SaveWaveDataCommand
        {
            get { return _saveWaveDataCommand; }
            set { this.SetPropertyValue(ref _saveWaveDataCommand, value); }
        }

        public bool IsConnecting
        {
            get { return _isConnecting; }
            set { this.SetPropertyValue(ref _isConnecting, value); }
        }

        public bool IsLoading
        {
            get { return _isLoading; }
            set
            {
                if (this.SetPropertyValue(ref _isLoading, value))
                {
                    this.NotifyPropertyChanged(nameof(IsNotLoading));
                }
            }
        }

        public bool IsNotLoading
        {
            get { return !_isLoading; }
        }

        public bool IsAutoScrollEnabled
        {
            get { return _isAutoScrollEnabled; }
            set { this.SetPropertyValue(ref _isAutoScrollEnabled, value); }
        }

        public bool IsPlaying
        {
            get { return _isPlaying; }
            set { this.SetPropertyValue(ref _isPlaying, value); }
        }

        public UIElement GlobalTimelineElement
        {
            get { return _globalTimelineElement; }
            set { this.SetPropertyValue(ref _globalTimelineElement, value); }
        }

        public Action<object> DragOverAction
        {
            get { return _dragOverAction; }
            set { this.SetPropertyValue(ref _dragOverAction, value); }
        }

        public Action<object> DropAction
        {
            get { return _dropAction; }
            set { this.SetPropertyValue(ref _dropAction, value); }
        }

        public string CurrentSaveWaveDataTargetName
        {
            get { return _currentSaveWaveDataTargetName; }
            set { this.SetPropertyValue(ref _currentSaveWaveDataTargetName, value); }
        }

        public string CurrentPlaybackTargetName
        {
            get { return _currentPlaybackTargetName; }
            set { this.SetPropertyValue(ref _currentPlaybackTargetName, value); }
        }

        public IEnumerable<string> WaveSourceOutputNames
        {
            get { return _waveSourceOutputNames; }
            set { this.SetPropertyValue(ref _waveSourceOutputNames, value); }
        }

        public MainWindowStatusConnectionViewModel StatusConnection => _statusConnectionViewModel;

        public MainWindowStatusLoadingViewModel StatusLoading => _statusLoadingViewModel;

        //-----------------------------------------------------------------

        public void RegisterPanelPlugins(
            IEnumerable<SpyPanelPlugin> panelPlugins,
            Func<SpyPanelPlugin, Control> createPanelContent)
        {
            Ensure.Argument.NotNull(panelPlugins);

            foreach (var panelPlugin in panelPlugins)
            {
                var panelViewModel = new SpyPanelViewModel(panelPlugin, createPanelContent);

                PropertyChangedObservation.GetObserver(_observerOwner, panelViewModel)
                    .AddHandler(
                        target => target.IsVisible,
                        (target, args) => _commandService.Execute(SpyCommands.UpdateSpyDataIDFlags));

                _panelViewModels.Add(panelViewModel);
            }
        }

        public SpyPanelViewModel GetPanelViewModel(string panelID)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(panelID);

            return _panelViewModels
                .Where(panelViewModel => panelViewModel.PanelID == panelID)
                .FirstOrDefault();
        }

        public void SetStatusConnection()
        {
            _statusConnectionViewModel.Visible = true;
            _statusLoadingViewModel.Visible = false;
            this.IsLoading = false;
        }

        public void SetStatusLoading()
        {
            _statusConnectionViewModel.Visible = false;
            _statusLoadingViewModel.Visible = true;
            this.IsLoading = true;
        }
    }
}
