﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Settings;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Controls;

namespace NintendoWare.Spy.Windows
{
    public sealed class ApiCallSpyPanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private ApiCallSpyViewModel _viewModel;
        private ApiCallSpyControl _view;

        //-----------------------------------------------------------------

        protected override Control CreateContent()
        {
            Assertion.Operation.Null(_viewModel);
            Assertion.Operation.Null(_view);

            _viewModel = new ApiCallSpyViewModel()
            {
                SyncContext = SynchronizationContext.Current,
                CopyAllApiCallLogsCommand = new DelegateCommand(param => this.CopyAllLogLines()),
            };

            _view = new ApiCallSpyControl()
            {
                DataContext = _viewModel,
            };

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.SelectedItem,
                    (target, args) =>
                    {
                        if (target.SelectedItem != null)
                        {
                            if (!this.GetPlaybackService().ShouldAutoScroll)
                            {
                                target.CurrentTime = target.SelectedItem.Time;
                                this.GetPlaybackService().SetCurrentTime(target.SelectedItem.Time.Timestamp);
                            }
                        }
                    });

            _viewModel.VisibleItemsAdded += (sender, args) => this.LogListBoxScrollToEnd();

            return _view;
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            switch (dataName)
            {
                case ApiCallSpyModelPlugin.SpyDataName:
                    _viewModel.SetModel((ApiCallSpyModel)model);
                    break;

#if false // TODO: NW-104
                case SoundDataInfoSpyModelPlugin.SpyDataName:
                    _viewModel.SetSoundDataInfoSpyModel((SoundDataInfoSpyModel)model);
                    break;
#endif
            }
        }

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => this.SyncGlobalTimelineCurrent());

            this.SyncGlobalTimelineCurrent();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            _view = null;

            base.OnUninitialize();
        }

        private void CopyAllLogLines()
        {
            if (_viewModel == null)
            {
                return;
            }

            StringBuilder sb = new StringBuilder();

            foreach (var logLine in _viewModel.Items
                .Where(it => _viewModel.FilterLogLine(it)))
            {
                sb.Append(logLine.Text);
                sb.AppendLine();
            }

            ClipboardUtility.SetText(sb.ToString());
        }

        private void LogListBoxScrollToEnd()
        {
            if (!this.GetPlaybackService().ShouldAutoScroll)
            {
                return;
            }

            _view.LogListBoxScrollToEnd();
        }

        private void SyncGlobalTimelineCurrent()
        {
            // カレント時間のアイテムが表示されるようにスクロールします。
            // スクロールの方向に応じて、なるべく多くのアイテムが表示範囲に入るようにします。

            if (_viewModel == null)
            {
                return;
            }

            var prev = _viewModel.CurrentTime;
            var curr = this.GetPlaybackService().Current;
            if (prev.Timestamp == curr.Timestamp)
            {
                return;
            }

            _viewModel.CurrentTime = curr;

            if (!_viewModel.CurrentItems.IsEmpty())
            {
                if (curr.Timestamp < prev.Timestamp)
                {
                    _view.LogListBoxScroll(_viewModel.CurrentItems.First());
                }
                else
                {
                    _view.LogListBoxScroll(_viewModel.CurrentItems.Last());
                }
            }
            else
            {
                var item = _viewModel.ItemLinedBelow ?? _viewModel.ItemLinedAbove;
                if (item != null)
                {
                    _view.LogListBoxScroll(item);
                }
            }
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (_viewModel != null)
            {
                _viewModel.TimeUnit = settings.TimeUnit;
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => this.ApplySettings(target));

            base.OnSettingsApplied(service);
        }
    }
}
