﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Resources;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

namespace NintendoWare.Spy.Settings
{
    /// <summary>
    /// チーム設定を提供するサービスです。
    /// </summary>
    public class TeamSettingsService
    {
        private TeamSettingsRoot _settingsRoot;
        private string[] _pluginDirectoryPaths = { };

        private string _warningMessage = null;

        /// <summary>
        /// プラグインを読み込むディレクトリを取得します。
        /// </summary>
        public IEnumerable<string> PluginDirectoryPaths
        {
            get
            {
                return _pluginDirectoryPaths;
            }
        }

        /// <summary>
        /// チーム設定ファイルを読み込みます。
        /// </summary>
        public void Load(string filePath)
        {
            Assertion.Argument.NotNull(filePath);

            _warningMessage = null;

            // チーム設定ファイルが存在しない場合は警告メッセージを設定します。
            if (File.Exists(filePath) == false)
            {
                _warningMessage = string.Format(Messages.WarningNotExistsTeamSettingsFile, filePath);
                return;
            }

            // チーム設定ファイルの読み込み
            try
            {
                using (var file = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                {
                    _settingsRoot = new XmlSerializer(typeof(TeamSettingsRoot)).Deserialize(file) as TeamSettingsRoot;
                }
            }
            catch
            {
                _warningMessage = string.Format(Messages.WarningCannotOpenTeamSettingsFile, filePath);
                return;
            }

            // 読み込んだデータからプラグインディレクトリ文字列を取得
            string value = _settingsRoot.Items
                .Where(i => i.Key == "PluginDirectories")
                .Select(i => i.Value).FirstOrDefault();

            if (value == null)
            {
                return;
            }

            // プラグインディレクトリ文字列から個々のプラグインディレクトリを取得
            var paths = value
                .Split(new char[] { '\n' }, StringSplitOptions.RemoveEmptyEntries)
                .Where(p => string.IsNullOrWhiteSpace(p) == false)
                .Select(p => p.Trim())
                .ToArray();

            // 個々のプラグインディレクトリの環境変数処理、相対パス処理
            string filePathDirectory = Path.GetDirectoryName(Path.GetFullPath(filePath));
            var newPaths = paths
                .Select(p => Environment.ExpandEnvironmentVariables(p).Trim())
                .Select(p => Path.IsPathRooted(p) ? p : Path.Combine(filePathDirectory, p))
                .ToArray();

            // プラグインディレクトリの取得と絶対パス化、不正パスの警告メッセージ作成
            List<string> pluginPathList = new List<string>();
            StringBuilder builder = new StringBuilder();

            for (int i = 0; i < newPaths.Length; i++)
            {
                if (this.IsValidPath(newPaths[i]) == true)
                {
                    string fullPath = Path.GetFullPath(newPaths[i]);
                    if (Directory.Exists(fullPath) == true)
                    {
                        pluginPathList.Add(fullPath); // 適正なパス
                    }
                    else // 存在しないパス
                    {
                        if (paths[i] == fullPath)
                        {
                            builder.AppendFormat("* {0} : \"{1}\"\n", Messages.WarningNotExists, paths[i]);
                        }
                        else
                        {
                            builder.AppendFormat("* {0} : \"{1}\"  [{2}]\n", Messages.WarningNotExists, paths[i], fullPath);
                        }
                    }
                }
                else // 不正なパス
                {
                    builder.AppendFormat("* {0} : \"{1}\"\n", Messages.WarningIniquityPath, paths[i]);
                }
            }

            if (builder.Length > 0)
            {
                _warningMessage = string.Format(Messages.WarningIniquityPluginDirectory, filePath, builder.ToString());
            }

            _pluginDirectoryPaths = pluginPathList.ToArray();
        }

        /// <summary>
        /// ディレクトリパスが正しいかどうか調べます。
        /// </summary>
        private bool IsValidPath(string path)
        {
            try
            {
                Path.GetFullPath(path);
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// チーム設定ファイルの読み込み時の警告文字列を取得します。
        /// </summary>
        public string GetWarningMessage()
        {
            return _warningMessage;
        }
    }
}
