﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.Spy;
using System;
using System.Drawing;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public partial class ProfileControl : UserControl
    {
        public ProfileSpyModel Model { get; set; } = null;

        public event EventHandler<DragMoveEventArgs> CurrentDraggedEvent
        {
            add { _controlHandler.CurrentDraggedEvent += value; }
            remove { _controlHandler.CurrentDraggedEvent -= value; }
        }

        private readonly LineGraphControlHandler _controlHandler;

        private LineGraphDrawer.BitmapStock _bitmapStock = new LineGraphDrawer.BitmapStock();

        /// <summary>
        /// 凡例の描画に使用するフォントです。
        /// </summary>
        private Font _legendFont;

        /// <summary>
        /// 棒グラフの凡例の表示位置(右端)です。
        /// </summary>
        private int _barGraphLegendPosRight;

        /// <summary>
        /// 折れ線グラフの凡例の表示位置(右端)です。
        /// </summary>
        private int _lineGraphLegendPosRight;

        /// <summary>
        /// 棒グラフの表示幅です。
        /// </summary>
        private int _barGraphWidth;

        /// <summary>
        /// 棒グラフの表示位置(左端)です。
        /// </summary>
        private int _barGraphPosLeft;

        /// <summary>
        /// 凡例の値を描画するためのSringFormatです。右寄せで描画します。
        /// </summary>
        private static StringFormat legendValueStringFormat = new StringFormat()
        {
            Alignment = StringAlignment.Far,
        };

        /// <summary>
        /// モデルデータが無い場合のダミーです。
        /// (TimeSpan.Span<0)のときに無効なTimeSpanと判定します。
        /// </summary>
        private static readonly ProfileSpyModel.Profile FieldNullProfile = new ProfileSpyModel.Profile();

        /// <summary>
        /// 棒グラフに表示する凡例のタイプです。
        /// </summary>
        private enum BarGraphLegends
        {
            SyncVoiceParam = 0,
            PpcVoiceRendering,
            AuxProcess,
            NwVoiceParamUpdate,
            MainMixProcess,
            FinalMixProcess,
            OutputFormatProcess,
            NwFrameProcess,
            DspFrameProcess,
        }

        /// <summary>
        /// 棒グラフに表示する凡例のラベルです。
        /// </summary>
        private static readonly string[] BarGraphLegendLabels = new string[]
        {
            "SyncParam",
            "PPCRendering",
            "AUX",
            "NWParamUpdate",
            "MainMix",
            "FinalMix",
            "OutputFormat",
            "NWFrame",
            "DSPFrame",
        };

        /// <summary>
        /// 折れ線グラフに表示する凡例のタイプです。
        /// </summary>
        private enum LineGraphLegends
        {
            Dsp = 0,
            Cpu,
            Ax,
            Nw,
        }

        /// <summary>
        /// 折れ線グラフに表示する凡例のラベルです。
        /// </summary>
        private static readonly string[] LineGraphLegendLabels = new string[]
        {
            "DSP",
            "CPU",
            " AX",
            " NW",
        };

        public bool PauseFlag
        {
            get { return _controlHandler.PauseFlag; }
            set { _controlHandler.PauseFlag = value; }
        }

        public ProfileControl()
        {
            // this.InitializeComponent();
            DoubleBuffered = true;
            BackColor = Color.Black;

            _controlHandler = new LineGraphControlHandler(this, -4);
        }

        public void Reset()
        {
            this.Model = null;
            _bitmapStock = new LineGraphDrawer.BitmapStock();
            _controlHandler.Reset();
            Invalidate();
        }

        private const int VoiceCountMax = 96;
        private const int AudioFrameNSec = 3 * 1000 * 1000;
        private const int BarHeight = 7;
        private const int BarMargin = 1;
        private const int TimeRange = AudioFrameNSec * 2;
        private const int LegendPosLeft = 0;
        private const int NoPaintSize = 30;
        private const float LegendFontScale = 0.8f;

        public long GetCurrentFrame()
        {
            if (this.Model == null || this.Model.AudioFrameBegin < 0)
            {
                return -1;
            }
            return this.Model.AudioFrameBegin + _controlHandler.CurrentSampleIndex;
        }

        public void ScrollToAudioFramePos(long audioFrame)
        {
            if (this.Model == null || _controlHandler.CaptureFlag)
            {
                // キャプチャー時、外部からの変更は受け付けない
                return;
            }

            if (audioFrame == _controlHandler.CurrentSampleIndex + this.Model.AudioFrameBegin)
            {
                // 現在のオーディオフレームを同じなら、currentSampleIndexを更新しない
                return;
            }

            this.UpdateSamples();
            _controlHandler.SetCurrentSampleIndex(audioFrame - this.Model.AudioFrameBegin);
            Invalidate();
        }

        public void UpdateSamples()
        {
            if (this.Model == null)
            {
                return;
            }

            _controlHandler.LastSampleIndex = this.Model.ProfileList.Count - 1;
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            this.UpdateSamples();
            _controlHandler.OnMouseDown(e);
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            _controlHandler.OnMouseUp(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            _controlHandler.OnMouseMove(e);
        }

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            _controlHandler.OnMouseWheel(e);
        }

        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            base.OnPreviewKeyDown(e);

            _controlHandler.OnPreviewKeyDown(e);
        }

        protected override void OnFontChanged(EventArgs e)
        {
            base.OnFontChanged(e);
            this.InvalidateLayout();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            this.InvalidateLayout();
        }

        private float GetDspLoadValue(long profileIndex, object userData)
        {
            ProfileSpyModel.Profile profile = this.Model.ProfileList[(int)profileIndex];
            if (profile == null)
            {
                return 0;
            }

            return profile.DspFrameProcess.Span;
        }

        private float GetCpuLoadValue(long profileIndex, object userData)
        {
            ProfileSpyModel.Profile profile = this.Model.ProfileList[(int)profileIndex];
            if (profile == null)
            {
                return 0;
            }

            long axLoad = (profile.AuxProcess.End - profile.AxFrameProcess.Begin) +
                     (profile.AxFrameProcess.End - profile.MainMixProcess.Begin);
            long nwLoad = profile.NwFrameProcess.Span + profile.NwVoiceParamUpdate.Span;

            return axLoad + nwLoad;
        }

        private float GetAxLoadValue(long profileIndex, object userData)
        {
            ProfileSpyModel.Profile profile = this.Model.ProfileList[(int)profileIndex];
            if (profile == null)
            {
                return 0;
            }

            long axLoad = (profile.AuxProcess.End - profile.AxFrameProcess.Begin) +
                (profile.AxFrameProcess.End - profile.MainMixProcess.Begin);
            return axLoad;
        }

        private float GetNwLoadValue(long profileIndex, object userData)
        {
            ProfileSpyModel.Profile profile = this.Model.ProfileList[(int)profileIndex];
            if (profile == null)
            {
                return 0;
            }

            long nwLoad = profile.NwFrameProcess.Span + profile.NwVoiceParamUpdate.Span;
            return nwLoad;
        }

        /// <summary>
        /// グラフのレイアウトを再計算させます。再計算はOnPaint()で行われます。
        /// </summary>
        private void InvalidateLayout()
        {
            Disposer.SafeDispose(ref _legendFont);
        }

        private void DrawProfileBar(
            Graphics g,
            Brush br,
            Font font,
            string label,
            ProfileSpyModel.TimeSpan timeSpan,
            long baseTime,
            int y)
        {
            if (timeSpan.Begin - baseTime < TimeRange)
            {
                int x = (int)(_barGraphPosLeft + (timeSpan.Begin - baseTime) * _barGraphWidth / TimeRange);
                Rectangle rect = new Rectangle(
                    x,
                    y,
                    Math.Max(1, (int)(timeSpan.Span * _barGraphWidth / TimeRange)), // 必ず表示されるように幅は>=1にします
                    BarHeight);
                g.FillRectangle(br, rect);
            }

            this.DrawBarGraphLegend(label, g, font, br, timeSpan.Span, y);
        }

        /// <summary>
        /// 棒グラフの凡例ラベルを取得します。
        /// </summary>
        private static string GetLabel(BarGraphLegends e)
        {
            return BarGraphLegendLabels[(int)e];
        }

        /// <summary>
        /// 折れ線グラフの凡例ラベルを取得します。
        /// </summary>
        private static string GetLabel(LineGraphLegends e)
        {
            return LineGraphLegendLabels[(int)e];
        }

        /// <summary>
        /// 棒グラフの凡例を描画します。
        /// </summary>
        private void DrawBarGraphLegend(
            string label,
            Graphics g,
            Font font,
            Brush br,
            long value,
            float y)
        {
            // バーと凡例の中央が一致するようにします。
            float y2 = y + BarHeight / 2 - font.GetHeight() / 2;

            g.DrawString(label, font, br, LegendPosLeft, y2);

            string valueString = string.Format("{0:D}%", value * 100 / AudioFrameNSec);
            g.DrawString(valueString, font, br, _barGraphLegendPosRight, y2, legendValueStringFormat);
        }

        /// <summary>
        /// 折れ線グラフの凡例を描画します。
        /// </summary>
        private void DrawLineGraphLegend(
            string label,
            Graphics g,
            Font font,
            Brush br,
            float value,
            float y)
        {
            g.DrawString(label, font, br, LegendPosLeft, y);

            var valueString = string.Format("{0:F2}%", value * 100.0f / AudioFrameNSec);
            g.DrawString(valueString, font, br, _lineGraphLegendPosRight, y, legendValueStringFormat);
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            if (ClientSize.Width <= NoPaintSize || ClientSize.Height <= NoPaintSize)
            {
                return;
            }

            Graphics g = e.Graphics;

            const int GraphPosY = 100;
            const int GraphMarginBottom = 10;
            const int ScrollBarHeight = 10;

            var br = Brushes.Yellow;
            var brDsp = Brushes.GreenYellow;
            var brNw = Brushes.SkyBlue;

            // グラフのレイアウトを計算します。
            if (_legendFont == null)
            {
                float fontSize = this.Font.Size * LegendFontScale;
                // 凡例の値は右寄せで描画します。
                // プロポーショナルフォントだと文字列によって描画位置が微妙に変わってしまうため、
                // 等幅フォントを使用します。
                _legendFont = new Font(FontFamily.GenericMonospace, fontSize, this.Font.Style);

                // 棒グラフのレイアウトを計算します。
                // 凡例の大きさを求め、残りを棒グラフの領域とします。
                // 棒グラフを表示するのに十分な幅が無いときは棒グラフを凡例と重ねて描画します。
                {
                    int legendWidth = 0;
                    for (int i = 0; i < BarGraphLegendLabels.Length; ++i)
                    {
                        legendWidth = Math.Max(legendWidth, (int)Math.Ceiling(g.MeasureString(BarGraphLegendLabels[i] + " 000%", _legendFont).Width));
                    }

                    _barGraphLegendPosRight = LegendPosLeft + legendWidth;
                    _barGraphWidth = Math.Max(NoPaintSize, ClientSize.Width - LegendPosLeft - legendWidth);
                    _barGraphPosLeft = ClientSize.Width - _barGraphWidth;
                }

                // 折れ線グラフの凡例のレイアウトを計算します。
                {
                    int legendWidth = 0;
                    for (int i = 0; i < LineGraphLegendLabels.Length; ++i)
                    {
                        legendWidth = Math.Max(legendWidth, (int)Math.Ceiling(g.MeasureString(LineGraphLegendLabels[i] + " 000.00%", _legendFont).Width));
                    }

                    _lineGraphLegendPosRight = LegendPosLeft + legendWidth;
                }
            }

            int GraphHeight = ClientSize.Height - GraphPosY - GraphMarginBottom - ScrollBarHeight;

            var profile = FieldNullProfile;
            if (this.Model != null && _controlHandler.LastSampleIndex >= 0)
            {
                var current = this.Model.ProfileList[(int)_controlHandler.CurrentSampleIndex];
                if (current != null)
                {
                    profile = current;
                }
            }

            // 罫線
            {
                Pen penLine = new Pen(Color.DarkGray);
                Pen penLine2 = new Pen(Color.DimGray);

                for (int i = 0; i < 2; ++i)
                {
                    int x0 = (int)(_barGraphPosLeft + i * _barGraphWidth * AudioFrameNSec / TimeRange);
                    Pen pen = new Pen(Color.Gray);
                    g.DrawLine(pen, new Point(x0, 0), new Point(x0, 80));
                }

                for (int i = 0; i <= 100; i += 10)
                {
                    Pen pen1 = ((i % 50) == 0) ? penLine : penLine2;
                    int y1 = GraphPosY + GraphHeight - i * GraphHeight / 100;
                    g.DrawLine(pen1, 0, y1, ClientSize.Width - 1, y1);
                }
            }

            {
                long leftSampleIndex = _controlHandler.LeftSampleIndex;
                long rightSampleIndex = _controlHandler.RightSampleIndex;

                // メーター表示
                {
                    long baseTime = profile.AxFrameProcess.Begin;
                    int y = BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.SyncVoiceParam), profile.SyncVoiceParam, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.PpcVoiceRendering), profile.PpcVoiceRendering, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.AuxProcess), profile.AuxProcess, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, brNw, _legendFont, GetLabel(BarGraphLegends.NwVoiceParamUpdate), profile.NwVoiceParamUpdate, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.MainMixProcess), profile.MainMixProcess, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.FinalMixProcess), profile.FinalMixProcess, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, br, _legendFont, GetLabel(BarGraphLegends.OutputFormatProcess), profile.OutputFormatProcess, baseTime, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, brNw, _legendFont, GetLabel(BarGraphLegends.NwFrameProcess), profile.NwFrameProcess, baseTime != 0 ? baseTime : profile.NwFrameProcess.Begin, y);
                    y += BarHeight + BarMargin;

                    DrawProfileBar(g, brDsp, _legendFont, GetLabel(BarGraphLegends.DspFrameProcess), profile.DspFrameProcess, baseTime, y);
                    y += BarHeight + BarMargin;
                }

                // 時間遷移グラフ表示
                if (rightSampleIndex >= leftSampleIndex && GraphHeight > 0)
                {
                    int x = 0;
                    int y = GraphPosY;
                    int width = ClientSize.Width;
                    int height = GraphHeight;
                    int centerY = GraphHeight;
                    float yScale = (float)GraphHeight / AudioFrameNSec;

                    Bitmap bmp = new Bitmap(width, height, g);
                    Graphics gBmp = Graphics.FromImage(bmp);

                    long drawLeft = leftSampleIndex;
                    long drawRight = rightSampleIndex;

                    _bitmapStock.Apply(
                        gBmp,
                        height,
                        ref drawLeft, ref drawRight,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);

                    // 折れ線グラフ描画
                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.GreenYellow,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetDspLoadValue, null, null);

                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.White,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetCpuLoadValue, null, null);

                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.Yellow,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetAxLoadValue, null, null);

                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.SkyBlue,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetNwLoadValue, null, null);

                    g.DrawImage(bmp, new Point(x, y));

                    gBmp.Dispose();
                    gBmp = null;

                    _bitmapStock.Update(
                        bmp,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);
                }

                // 折れ線グラフに凡例を描画
                {
                    float dspLoad = 0;
                    float cpuLoad = 0;
                    float axLoad = 0;
                    float nwLoad = 0;

                    if (this.Model != null)
                    {
                        dspLoad = this.GetDspLoadValue(_controlHandler.CurrentSampleIndex, null);
                        cpuLoad = this.GetCpuLoadValue(_controlHandler.CurrentSampleIndex, null);
                        axLoad = this.GetAxLoadValue(_controlHandler.CurrentSampleIndex, null);
                        nwLoad = this.GetNwLoadValue(_controlHandler.CurrentSampleIndex, null);
                    }

                    float y = GraphPosY;

                    this.DrawLineGraphLegend(GetLabel(LineGraphLegends.Dsp), g, _legendFont, Brushes.GreenYellow, dspLoad, y);
                    y += BarHeight + BarMargin;

                    this.DrawLineGraphLegend(GetLabel(LineGraphLegends.Cpu), g, _legendFont, Brushes.White, cpuLoad, y);
                    y += BarHeight + BarMargin;

                    this.DrawLineGraphLegend(GetLabel(LineGraphLegends.Ax), g, _legendFont, Brushes.Yellow, axLoad, y);
                    y += BarHeight + BarMargin;

                    this.DrawLineGraphLegend(GetLabel(LineGraphLegends.Nw), g, _legendFont, Brushes.SkyBlue, nwLoad, y);
                }

                // カレント位置描画
                if (_controlHandler.CurrentSampleIndex >= 0 && (PauseFlag || _controlHandler.CaptureFlag))
                {
                    int currentSamplePosX = _controlHandler.CurrentSamplePosX;
                    g.DrawLine(Pens.Red, currentSamplePosX, GraphPosY, currentSamplePosX, ClientSize.Height);
                }

                // スクロールバー描画
                {
                    LineGraphDrawer.DrawScrollBar(g,
                        0, ClientSize.Height - ScrollBarHeight,
                        ClientSize.Width, ScrollBarHeight,
                        leftSampleIndex, rightSampleIndex, _controlHandler.LastSampleIndex);
                }
            }
        }
    }
}
