﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using NintendoWare.Spy;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Shapes;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    /// <summary>
    /// このカスタム コントロールを XAML ファイルで使用するには、手順 1a または 1b の後、手順 2 に従います。
    ///
    /// 手順 1a) 現在のプロジェクトに存在する XAML ファイルでこのカスタム コントロールを使用する場合
    /// この XmlNamespace 属性を使用場所であるマークアップ ファイルのルート要素に
    /// 追加します:
    ///
    ///     xmlns:MyNamespace="clr-namespace:WPFCustomControlTest"
    ///
    ///
    /// 手順 1b) 異なるプロジェクトに存在する XAML ファイルでこのカスタム コントロールを使用する場合
    /// この XmlNamespace 属性を使用場所であるマークアップ ファイルのルート要素に
    /// 追加します:
    ///
    ///     xmlns:MyNamespace="clr-namespace:WPFCustomControlTest;assembly=WPFCustomControlTest"
    ///
    /// また、XAML ファイルのあるプロジェクトからこのプロジェクトへのプロジェクト参照を追加し、
    /// リビルドして、コンパイル エラーを防ぐ必要があります:
    ///
    ///     ソリューション エクスプローラーで対象のプロジェクトを右クリックし、
    ///     [参照の追加] の [プロジェクト] を選択してから、このプロジェクトを参照し、選択します。
    ///
    ///
    /// 手順 2)
    /// コントロールを XAML ファイルで使用します。
    ///
    ///     <MyNamespace:CustomControl/>
    ///
    /// </summary>

    //[TemplatePart(Name = GridPart, Type = typeof(Grid))]
    public class BarControl : Control
    {
        private const string GridName = "grid";
        private const string RectangleName = "rectangle";

        static BarControl()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(BarControl), new FrameworkPropertyMetadata(typeof(BarControl)));
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            var grid = GetTemplateChild(GridName) as Grid;
            grid.SizeChanged += delegate (object sender, SizeChangedEventArgs e)
            {
                UpdateBarWidth();
            };
        }

        public float Value
        {
            get { return (float)GetValue(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }

        public float MaxValue
        {
            get { return (float)GetValue(MaxValueProperty); }
            set { SetValue(MaxValueProperty, value); }
        }

        public Color FillColor
        {
            get { return (Color)GetValue(FillColorProperty); }
            set { SetValue(FillColorProperty, value); }
        }

        public Brush Fill
        {
            get { return (Brush)GetValue(FillProperty); }
            set { SetValue(FillProperty, value); }
        }

        public double BarWidth
        {
            get { return (double)GetValue(BarWidthProperty); }
            set { SetValue(BarWidthProperty, value); }
        }

        private float Ratio
        {
            get
            {
                if (this.MaxValue > 0)
                {
                    return MathUtility.Clamp(this.Value / this.MaxValue, 0.0f, 1.0f);
                }
                else
                {
                    return 0;
                }
            }
        }

        public static readonly DependencyProperty ValueProperty =
            DependencyProperty.Register(
                nameof(Value),
                typeof(float),
                typeof(BarControl),
                new UIPropertyMetadata(0.0F, OnValueChanged));

        public static readonly DependencyProperty MaxValueProperty =
            DependencyProperty.Register(
                nameof(MaxValue),
                typeof(float),
                typeof(BarControl),
                new UIPropertyMetadata(0.0F, OnValueChanged));

        public static readonly DependencyProperty BarWidthProperty =
            DependencyProperty.Register(
                nameof(BarWidth),
                typeof(double),
                typeof(BarControl),
                new UIPropertyMetadata(0.0));

        public static readonly DependencyProperty FillColorProperty =
            DependencyProperty.Register(
                nameof(FillColor),
                typeof(Color),
                typeof(BarControl),
                new UIPropertyMetadata(Colors.DarkGray, OnFillColorChanged));

        public static readonly DependencyProperty FillProperty =
            DependencyProperty.Register(
                nameof(Fill),
                typeof(Brush),
                typeof(BarControl),
                new UIPropertyMetadata(null));

        private void UpdateBarWidth()
        {
            var grid = GetTemplateChild(GridName) as Grid;
            var rectangle = GetTemplateChild(RectangleName) as Rectangle;

            if (grid == null || rectangle == null)
            {
                return;
            }

            this.BarWidth = grid.ActualWidth * this.Ratio;
        }

        private void UpdateFill()
        {
            try
            {
                Color c = this.FillColor;
                c.A = (byte)(255 * (this.Ratio * 0.7f + 0.3f));
                this.Fill = new SolidColorBrush(c);
            }
            catch
            {
                this.Fill = null;
            }
        }

        private static void OnValueChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var control = d as BarControl;
            control.UpdateBarWidth();
            control.UpdateFill();
        }

        private static void OnFillColorChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var control = d as BarControl;
            control.UpdateFill();
        }
    }
}
