﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Drawing;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public partial class LoudnessMeterControl : UserControl
    {
        private const int MarginLeft = 25;
        private const int MarginTop = 30;
        private const int MarginBottom = 10;
        private const int ScrollBarHeight = 10;

        private const int MeterBarWidth = 12;
        private const int MeterMarginWidth = 14;

        private const int LoudnessDisplayWidth = 100;

        private const int MaxDrawChannelCountTv = 6;
        private const int MaxDrawChannelCountDrc = 2;
        private const int ChannelCountMax = MaxDrawChannelCountTv + MaxDrawChannelCountDrc;

        public FinalOutSpyModel Model { get; set; } = null;

        public event EventHandler<DragMoveEventArgs> CurrentDraggedEvent
        {
            add { _controlHandler.CurrentDraggedEvent += value; }
            remove { _controlHandler.CurrentDraggedEvent -= value; }
        }

        public bool PauseFlag
        {
            get { return _controlHandler.PauseFlag; }
            set { _controlHandler.PauseFlag = value; }
        }

        public int DrawChannelsTv
        {
            get { return _drawChannelsTv; }
            set
            {
                _drawChannelsTv = value;
                UpdateChannelIndexes();
                UpdateMeterWidth();
            }
        }
        public int DrawChannelsDrc
        {
            get { return _drawChannelsDrc; }
            set
            {
                _drawChannelsDrc = value;
                UpdateChannelIndexes();
                UpdateMeterWidth();
            }
        }

        private int _drawChannelsTv = 4;
        private int _drawChannelsDrc = 2;
        private int[] _channelIndexes = null;
        private Frame _audioFrameBegin = Frame.InvalidValue;

        private readonly LineGraphControlHandler _controlHandler;
        private LineGraphDrawer.BitmapStock _bitmapStock = new LineGraphDrawer.BitmapStock();

        private struct Channel
        {
            public string Name;
            public PeakHold PeakValue;

            public Channel(string name)
            {
                this.Name = name;
                this.PeakValue = new PeakHold();
            }
        }

        private readonly Channel[] _channels = new Channel[ChannelCountMax]
        {
            new Channel("F-L"),
            new Channel("F-R"),
            new Channel("R-L"),
            new Channel("R-R"),
            new Channel("F-C"),
            new Channel("LFE"),
            new Channel("DRC-L"),
            new Channel("DRC-R")
        };

        public class DoubleBufferedControl : Control
        {
            public DoubleBufferedControl()
            {
                DoubleBuffered = true;
                SetStyle(ControlStyles.Selectable, true);
                SetStyle(ControlStyles.UserMouse, true);
            }
        }

        private readonly DoubleBufferedControl _panel;

        public LoudnessMeterControl()
        {
            _panel = new DoubleBufferedControl();
            _panel.Parent = this;
            _panel.Location = new Point(MarginLeft, MarginTop);
            _panel.Paint += new PaintEventHandler(OnPaintLineGraph);
            _panel.MouseUp += new MouseEventHandler(OnMouseUpLineGraph);
            _panel.MouseDown += new MouseEventHandler(OnMouseDownLineGraph);
            _panel.MouseMove += new MouseEventHandler(OnMouseMoveLineGraph);
            _panel.MouseWheel += new MouseEventHandler(OnMouseWheelLineGraph);
            _panel.PreviewKeyDown += new PreviewKeyDownEventHandler(OnPreviewKeyDownLineGraph);

            // this.InitializeComponent();
            DoubleBuffered = true;
            BackColor = Color.Black;

            _controlHandler = new LineGraphControlHandler(_panel, -8);
            _controlHandler.CurrentChangedEvent += OnChangedMeterValue;

            for (int index = 0; index < _channels.Length; index++)
            {
                _channels[index].PeakValue = new PeakHold();
                _channels[index].PeakValue.ValueChangedEvent += OnChangedMeterValue;
            }

            UpdateChannelIndexes();
            UpdateMeterWidth();
        }

        public void Reset()
        {
            this.Model = null;
            _bitmapStock = new LineGraphDrawer.BitmapStock();
            Invalidate();
            _panel.Invalidate();
        }

        private void OnChangedMeterValue(object sender, EventArgs ea)
        {
            Invalidate();
        }

        private void UpdateChannelIndexes()
        {
            int totalDrawChannelCount = _drawChannelsTv + _drawChannelsDrc;
            int[] indexes = new int[totalDrawChannelCount];

            int index = 0;
            for (int channelIndex = 0; channelIndex < _drawChannelsTv; channelIndex++)
            {
                indexes[index++] = (int)FinalOutSpyModel.ChannelIndex.MainFrontLeft + channelIndex;
            }
            for (int channelIndex = 0; channelIndex < _drawChannelsDrc; channelIndex++)
            {
                indexes[index++] = (int)FinalOutSpyModel.ChannelIndex.DrcLeft + channelIndex;
            }
            _channelIndexes = indexes;
        }

        private void UpdateMeterWidth()
        {
            int totalDrawChannelCount = _drawChannelsTv + _drawChannelsDrc;
            int meterWidth = (MeterBarWidth + MeterMarginWidth) * totalDrawChannelCount;

            // NOTE: MarginLeft は、折れ線グラフの左と、メータの左の２つ分
            _panel.Width = ClientSize.Width - MarginLeft - MarginLeft - meterWidth - LoudnessDisplayWidth;
        }

        /// <summary>
        /// プロットするオーディオフレームの範囲を指定します。
        /// </summary>
        /// <param name="audioFrameBegin"></param>
        /// <param name="audioFrameEnd"></param>
        public void SetAudioFrameRange(Frame audioFrameBegin, Frame audioFrameEnd)
        {
            if (audioFrameBegin != Frame.InvalidValue && audioFrameBegin <= audioFrameEnd)
            {
                _audioFrameBegin = audioFrameBegin;
                _controlHandler.LastSampleIndex = audioFrameEnd - audioFrameBegin;
            }
            else if (_audioFrameBegin != Frame.InvalidValue)
            {
                _audioFrameBegin = Frame.InvalidValue;
                _controlHandler.LastSampleIndex = -1;
            }
        }

        public Frame GetCurrentFrame()
        {
            if (_audioFrameBegin == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            return _audioFrameBegin + _controlHandler.CurrentSampleIndex;
        }

        public void ScrollToAudioFramePos(Frame audioFrame)
        {
            if (this.Model == null || _controlHandler.CaptureFlag)
            {
                // キャプチャー時、外部からの変更は受け付けない
                return;
            }

            if (_audioFrameBegin == Frame.InvalidValue)
            {
                return;
            }

            if (audioFrame == GetCurrentFrame())
            {
                // 現在のオーディオフレームを同じなら、currentSampleIndexを更新しない
                return;
            }

            this.UpdateSamples();
            _controlHandler.SetCurrentSampleIndex(audioFrame - _audioFrameBegin);
            _panel.Invalidate();

            // 縦メータの更新
            Invalidate();
        }

        public void UpdateSamples()
        {
            if (this.Model == null)
            {
                return;
            }

            Invalidate();
            _panel.Invalidate();
        }

        private void OnMouseDownLineGraph(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                if (!PauseFlag && this.Model != null)
                {
                    this.UpdateSamples();
                }
            }
            _controlHandler.OnMouseDown(e);
        }

        private void OnMouseUpLineGraph(object sender, MouseEventArgs e)
        {
            _controlHandler.OnMouseUp(e);
        }

        private void OnMouseMoveLineGraph(object sender, MouseEventArgs e)
        {
            _controlHandler.OnMouseMove(e);
        }

        private void OnMouseWheelLineGraph(object sender, MouseEventArgs e)
        {
            _controlHandler.OnMouseWheel(e);
        }

        private void OnPreviewKeyDownLineGraph(object sender, PreviewKeyDownEventArgs e)
        {
            _controlHandler.OnPreviewKeyDown(e);
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            UpdateMeterWidth();

            _panel.Height = ClientSize.Height - MarginTop;
        }

        private float GetPeakValue(long finalOutIndx, object userData)
        {
            int channelIndex = (int)userData;
            int frameIndex = (int)finalOutIndx;

            FinalOutSpyModel.LoudnessInfo loudness = this.Model.GetLoudness(_audioFrameBegin + frameIndex);
            if (loudness == null)
            {
                return float.NegativeInfinity;
            }

            return loudness.Channels[channelIndex].PeakValue;
        }

        private float GetPeakValueTv(long finalOutIndx, object userData)
        {
            float peakValue = float.NegativeInfinity;
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontLeft));
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontRight));
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainRearLeft));
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainRearRight));
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontCenter));
            peakValue = Math.Max(peakValue, GetPeakValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainLfe));
            return peakValue;
        }

        private float GetRmsValue(long finalOutIndx, object userData)
        {
            int channelIndex = (int)userData;
            int frameIndex = (int)finalOutIndx;

            FinalOutSpyModel.LoudnessInfo loudness = this.Model.GetLoudness(_audioFrameBegin + frameIndex);
            if (loudness == null)
            {
                return float.NegativeInfinity;
            }

            return loudness.Channels[channelIndex].RmsValue;
        }

        private float GetRmsValueTv(long finalOutIndx, object userData)
        {
            float peakValue = float.NegativeInfinity;

            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontLeft));
            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontRight));
            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainRearLeft));
            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainRearRight));
            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainFrontCenter));
            peakValue = Math.Max(peakValue, GetRmsValue(finalOutIndx, FinalOutSpyModel.ChannelIndex.MainLfe));

            return peakValue;
        }

        private float GetIntegratedLoudnessValueTv(long finalOutIndx, object userData)
        {
            int frameIndex = (int)finalOutIndx;
            FinalOutSpyModel.LoudnessInfo loudness = this.Model.GetLoudness(_audioFrameBegin + frameIndex);
            if (loudness == null)
            {
                return float.NegativeInfinity;
            }

            return loudness.RelGatedLoudnessValue;
        }

        private float GetShortTermLoudnessValueTv(long finalOutIndx, object userData)
        {
            int frameIndex = (int)finalOutIndx;
            FinalOutSpyModel.LoudnessInfo loudness = this.Model.GetLoudness(_audioFrameBegin + frameIndex);
            if (loudness == null)
            {
                return float.NegativeInfinity;
            }

            return loudness.ShortTermLoudnessValue;
        }

        private const int DecibelMin = -60;
        private const int DecibelMax = 0;

        private void OnPaintLineGraph(object sender, PaintEventArgs e)
        {
            const int NoPaintSize = 30;

            if (ClientSize.Width <= NoPaintSize || ClientSize.Height <= NoPaintSize)
            {
                return;
            }

            Graphics g = e.Graphics;

            int GraphWidth = _panel.ClientSize.Width;
            int GraphHeight = _panel.ClientSize.Height - MarginBottom - ScrollBarHeight;

            // 罫線
            {
                for (int i = DecibelMin; i <= DecibelMax; i += 10)
                {
                    Pen pen1 = (i == DecibelMin || i == DecibelMax) ? Pens.DarkGray : Pens.DimGray;
                    int y1 = -i * (GraphHeight - 1) / (DecibelMax - DecibelMin);
                    g.DrawLine(pen1, 0, y1, GraphWidth - 1, y1);
                }
            }

            if (this.Model != null)
            {
                long leftSampleIndex = _controlHandler.LeftSampleIndex;
                long rightSampleIndex = _controlHandler.RightSampleIndex;

                // ラウドネス情報の無いフレームは描画しないようにします。
                this.Model.UpdateLoudness();
                var audioFrameEnd = this.Model.LoudnessAudioFrameEnd;
                if (audioFrameEnd == Frame.InvalidValue)
                {
                    rightSampleIndex = -1;
                }
                else
                {
                    rightSampleIndex = Math.Min(rightSampleIndex, audioFrameEnd - _audioFrameBegin);
                }

                // 時間遷移グラフ表示
                if (rightSampleIndex >= leftSampleIndex)
                {
                    int x = 0;
                    int y = 0;
                    int width = GraphWidth;
                    int height = GraphHeight;
                    int centerY = 0;
                    float yScale = (float)GraphHeight / (DecibelMax - DecibelMin);

                    if (height <= 0)
                    {
                        return;
                    }

                    Bitmap bmp = new Bitmap(width, height, g);
                    Graphics gBmp = Graphics.FromImage(bmp);

                    long drawLeft = leftSampleIndex;
                    long drawRight = rightSampleIndex;

                    _bitmapStock.Apply(
                        gBmp,
                        height,
                        ref drawLeft, ref drawRight,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);

                    // 折れ線グラフ描画
                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.Yellow,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetPeakValueTv, null, null);

                    LineGraphDrawer.DrawLineGraph(
                        gBmp, Pens.GreenYellow,
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetRmsValueTv, null, null);

                    LineGraphDrawer.DrawLineGraph(
                        gBmp, new Pen(Color.SkyBlue, 3),
                        0, 0,
                        width, height,
                        _controlHandler.BeginSamplePosX, centerY,
                        drawLeft, drawRight,
                        _controlHandler.ZoomRatio.Index, yScale,
                        GetShortTermLoudnessValueTv, null, _channelIndexes);

                    g.DrawImage(bmp, new Point(x, y));

                    gBmp.Dispose();
                    gBmp = null;

                    _bitmapStock.Update(
                        bmp,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);
                }

                // カレント位置描画
                if (_controlHandler.CurrentSampleIndex >= 0 && (PauseFlag || _controlHandler.CaptureFlag))
                {
                    int currentSamplePosX = _controlHandler.CurrentSamplePosX;
                    g.DrawLine(Pens.Red, currentSamplePosX, 0, currentSamplePosX, GraphHeight);
                }

                // スクロールバー描画
                if (_controlHandler.LastSampleIndex >= 0)
                {
                    LineGraphDrawer.DrawScrollBar(
                        g,
                        0, _panel.ClientSize.Height - ScrollBarHeight,
                        _panel.ClientSize.Width, ScrollBarHeight,
                        leftSampleIndex, rightSampleIndex, _controlHandler.LastSampleIndex);
                }
            }
        }

        private static void DrawMeter(
            Graphics g, Font newFont,
            int posX, int posY,
            string name,
            float currentValue, float peakValue, bool drawPeakFlag,
            int height, int marginWidth, int barWidth,
            int textPosOffsetX, int textPosOffsetY)
        {
            const int PeakBarHeight = 2;

            float barHeight = -currentValue * height / (DecibelMax - DecibelMin);
            barHeight = Math.Min(barHeight, height);

            float peakBarPoxY = posY - peakValue * height / (DecibelMax - DecibelMin) - PeakBarHeight / 2;
            peakBarPoxY = Math.Min(peakBarPoxY, posY + height - PeakBarHeight / 2);

            string str = FormatFloatValue(peakValue);
            posX += marginWidth / 2;

            // チャンネル名
            g.DrawString(name, newFont, Brushes.White, posX + textPosOffsetX, posY + textPosOffsetY * 2);

            // 数値
            g.DrawString(str, newFont, Brushes.White, posX + textPosOffsetX, posY + textPosOffsetY);

            // メーター
            g.FillRectangle(
                Brushes.YellowGreen,
                posX, posY + barHeight,
                barWidth, (height - barHeight));

            // ピークホールド
            if (drawPeakFlag)
            {
                g.FillRectangle(
                    Brushes.Yellow,
                    posX, peakBarPoxY,
                    barWidth, PeakBarHeight);
            }

            posX += barWidth;
            posX += marginWidth / 2;
        }

        private static string FormatFloatValue(float x)
        {
            if (!float.IsNegativeInfinity(x))
            {
                return string.Format("{0,-1:F1}", x);
            }
            else
            {
                return "-∞";
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            const int NoPaintSize = 30;

            if (ClientSize.Width <= NoPaintSize || ClientSize.Height <= NoPaintSize)
            {
                return;
            }

            if (this.Model != null)
            {
                this.Model.UpdateLoudness();
            }

            Graphics g = e.Graphics;

            int GraphHeight = ClientSize.Height - MarginTop - MarginBottom - ScrollBarHeight;
            int GraphWidth = ClientSize.Width - _panel.Right - MarginLeft - LoudnessDisplayWidth;

            int x0 = _panel.Right + MarginLeft;

            // 罫線
            {
                for (int i = DecibelMin; i <= DecibelMax; i += 10)
                {
                    Pen pen1 = (i == DecibelMin || i == DecibelMax) ? Pens.DarkGray : Pens.DimGray;

                    int x1 = x0;
                    int y1 = _panel.Top - i * GraphHeight / (DecibelMax - DecibelMin);

                    g.DrawLine(pen1, x1, y1, x1 + GraphWidth, y1);

                    y1 -= this.Font.Height / 2;

                    string str = string.Format("{0,3}", i);

                    // 数値表記（左）
                    g.DrawString(str, this.Font, Brushes.White, 0, y1);

                    // 数値表記（右）
                    g.DrawString(str, this.Font, Brushes.White, _panel.Right, y1);
                }
            }

            // 縦メータ
            {
                int posX = x0;
                int posY = _panel.Top;
                int width = this.ClientSize.Width - posX;
                int height = _panel.Height - MarginBottom - ScrollBarHeight;

                if (height < 0)
                {
                    height = 0;
                }

                int barCount = _drawChannelsTv + _drawChannelsDrc;

                Font newFont = new Font(this.Font.FontFamily, 6, this.Font.Style);

                int textPosOffsetX = -MeterMarginWidth / 2;
                int textPosOffsetY = -(int)newFont.Size * 2;

                if (this.Model != null && this.Model.AudioFrameBegin != Frame.InvalidValue && _audioFrameBegin != Frame.InvalidValue && _controlHandler.CurrentSampleIndex >= 0)
                {
                    foreach (int channelIndex in _channelIndexes)
                    {
                        float currentValue = GetRmsValue(_controlHandler.CurrentSampleIndex, channelIndex);

                        bool drawPeakFlag;
                        float peakValue;
                        if (this.PauseFlag || _controlHandler.CaptureFlag)
                        {
                            drawPeakFlag = false;
                            peakValue = currentValue;
                        }
                        else
                        {
                            PeakHold peakHold = _channels[channelIndex].PeakValue;
                            peakHold.Update(currentValue);
                            peakValue = peakHold.PeakValue;
                            drawPeakFlag = true;
                        }

                        DrawMeter(
                            g, newFont,
                            posX, posY,
                            _channels[channelIndex].Name,
                            currentValue, peakValue, drawPeakFlag,
                            height, MeterMarginWidth, MeterBarWidth,
                            textPosOffsetX, textPosOffsetY);

                        posX += MeterBarWidth + MeterMarginWidth;
                    }
                }
            }

            // ラウドネス数値表示
            if (_controlHandler.CurrentSampleIndex >= 0 && this.Model != null)
            {
                int x = this.Right - LoudnessDisplayWidth;
                int y = 60;

                Font bigFont = new Font(this.Font.FontFamily, 12, this.Font.Style);
                Font smallFont = new Font(this.Font.FontFamily, 7, this.Font.Style);
                string str;

                float loudnessValue = GetShortTermLoudnessValueTv(_controlHandler.CurrentSampleIndex, null);
                str = FormatFloatValue(loudnessValue);

                g.DrawString(str, bigFont, Brushes.White, x + 20, y);
                y += 20;
                g.DrawString("Short-term(LKFS)", smallFont, Brushes.White, x + 10, y);

                y += 40;
                loudnessValue = GetIntegratedLoudnessValueTv(_controlHandler.CurrentSampleIndex, null);
                str = FormatFloatValue(loudnessValue);

                g.DrawString(str, bigFont, Brushes.White, x + 20, y);
                y += 20;
                g.DrawString("Integrated(LKFS)", smallFont, Brushes.White, x + 10, y);
            }
        }
    }
}
