﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using NintendoWare.Spy;
using System;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public class LineGraphControlHandler
    {
        private enum DragMode
        {
            None,
            Panning,
            Scrolling,
        }

        public event EventHandler CurrentChangedEvent;

        public event EventHandler<DragMoveEventArgs> CurrentDraggedEvent;

        public bool PauseFlag
        {
            get;
            set;
        }
        public bool CaptureFlag
        {
            get { return _captureFlag; }
        }
        public ZoomRatio ZoomRatio
        {
            get { return _zoomRatio; }
        }
        public long CurrentSampleIndex
        {
            get { return _currentSampleIndex; }
        }
        public long LeftSampleIndex
        {
            get
            {
                long leftSampleIndex = _zoomRatio.CalcPointCountFromWidth(0 - this.BeginSamplePosX) - 1;
                leftSampleIndex = MathUtility.Clamp(leftSampleIndex, 0, this.LastSampleIndex);
                return leftSampleIndex;
            }
        }
        public long RightSampleIndex
        {
            get
            {
                long rightSampleIndex = _zoomRatio.CalcPointCountFromWidth(_control.ClientSize.Width - 1 - this.BeginSamplePosX) + 1;
                rightSampleIndex = MathUtility.Clamp(rightSampleIndex, 0, this.LastSampleIndex);
                return rightSampleIndex;
            }
        }
        public int CurrentSamplePosX
        {
            get
            {
                int currentSamplePosX = this.BeginSamplePosX + _zoomRatio.CalcWidthFromPointCount(_currentSampleIndex);
                return currentSamplePosX;
            }
        }

        public long LastSampleIndex { get; set; } = -1;
        public int BeginSamplePosX { get; set; } = 0;

        private long _currentSampleIndex = -1;

        private readonly ZoomRatio _zoomRatio = new ZoomRatio(0);

        private bool _captureFlag = false;

        private Control _control = null;

        private DragMode _dragMode = DragMode.None;
        private int _scrollSpeed = 0;
        private int _dragStartPos;
        private readonly Timer _scrollTimer = new Timer();

        public void Reset()
        {
            LastSampleIndex = -1;
            BeginSamplePosX = 0;
            _currentSampleIndex = -1;
        }

        public void SetCurrentSampleIndex(long sampleIndex)
        {
            _currentSampleIndex = sampleIndex;
            _currentSampleIndex = MathUtility.Clamp(_currentSampleIndex, 0, this.LastSampleIndex);
            EnsureVisibleCurrentSampleIndex();
        }

        public void ScrollToEnd()
        {
            _currentSampleIndex = this.LastSampleIndex;
            _currentSampleIndex = MathUtility.Clamp(_currentSampleIndex, 0, this.LastSampleIndex);
            this.BeginSamplePosX = _control.ClientSize.Width - 1 - _zoomRatio.CalcWidthFromPointCount(_currentSampleIndex);
        }

        public LineGraphControlHandler(Control control, int zoomRatioIndex)
        {
            _control = control;

            _scrollTimer.Interval = 20;
            _scrollTimer.Tick += new EventHandler(ScrollTimerOnTick);

            _zoomRatio.Index = zoomRatioIndex;

            PauseFlag = false;
        }

        private void UpdateCurrentSampleIndex(long index)
        {
            long prevCurrent = _currentSampleIndex;

            _currentSampleIndex = index;
            _currentSampleIndex = MathUtility.Clamp(_currentSampleIndex, 0, LastSampleIndex);

            if (prevCurrent != _currentSampleIndex)
            {
                if (CurrentChangedEvent != null)
                {
                    CurrentChangedEvent(this, EventArgs.Empty);
                }

                if (CurrentDraggedEvent != null)
                {
                    CurrentDraggedEvent(this, DragMoveEventArgs.Update);
                }
            }
        }

        private void ScrollTimerOnTick(object obj, EventArgs ea)
        {
            if (_scrollSpeed != 0)
            {
                this.BeginSamplePosX -= _scrollSpeed;

                int posX;
                if (_scrollSpeed < 0)
                {
                    posX = 0;
                    if (this.BeginSamplePosX > 0)
                    {
                        this.BeginSamplePosX = 0;
                    }
                }
                else
                {
                    posX = _control.ClientSize.Width - 1;
                    if (_zoomRatio.CalcPointCountFromWidth(_control.ClientSize.Width - 1 - BeginSamplePosX) > this.LastSampleIndex)
                    {
                        this.BeginSamplePosX = _control.ClientSize.Width - 1 - _zoomRatio.CalcWidthFromPointCount(this.LastSampleIndex);
                    }
                }
                UpdateCurrentSampleIndex(_zoomRatio.CalcPointCountFromWidth(posX - BeginSamplePosX));
                _control.Invalidate();
            }
        }

        private bool UpdateZoomValue(int newZoomIndex)
        {
            const int ZoomRatioIndexMax = 8; // x24

            if (newZoomIndex < _zoomRatio.Index &&
                _zoomRatio.CalcWidthFromPointCount(0) + BeginSamplePosX > 0 &&
                _zoomRatio.CalcWidthFromPointCount(this.LastSampleIndex) + BeginSamplePosX < _control.ClientSize.Width)
            {
                // これ以上の縮小は禁止
                return false;
            }
            if (newZoomIndex > ZoomRatioIndexMax)
            {
                // これ以上の拡大は禁止
                return false;
            }

            int currentSamplePosX = _zoomRatio.CalcWidthFromPointCount(CurrentSampleIndex) + BeginSamplePosX;

            _zoomRatio.Index = newZoomIndex;

            this.BeginSamplePosX = currentSamplePosX - _zoomRatio.CalcWidthFromPointCount(CurrentSampleIndex);

            return true;
        }

        // X座標から、その位置のサンプルインデックスを計算する
        private long GetSampleIndexFromPosX(int x)
        {
            long index = _zoomRatio.CalcPointCountFromWidth(x - BeginSamplePosX);
            index = MathUtility.Clamp(index, 0, LastSampleIndex);
            return index;
        }

        public void OnMouseMove(MouseEventArgs e)
        {
            if (_dragMode == DragMode.Scrolling)
            {
                if (e.X < 0)
                {
                    // 左スクロール
                    if (this.BeginSamplePosX < 0)
                    {
                        _scrollSpeed = e.X;
                    }
                    long newSampleIndex = GetSampleIndexFromPosX(0);
                    UpdateCurrentSampleIndex(newSampleIndex);
                }
                else if (e.X >= _control.ClientSize.Width)
                {
                    // 右スクロール
                    if (this.BeginSamplePosX + _zoomRatio.CalcWidthFromPointCount(this.LastSampleIndex) > _control.ClientSize.Width - 1)
                    {
                        _scrollSpeed = e.X - _control.ClientSize.Width;
                    }
                    long newSampleIndex = GetSampleIndexFromPosX(_control.ClientSize.Width - 1);
                    UpdateCurrentSampleIndex(newSampleIndex);
                }
                else
                {
                    // カーソル移動
                    _scrollSpeed = 0;
                    long newSampleIndex = GetSampleIndexFromPosX(e.X);
                    UpdateCurrentSampleIndex(newSampleIndex);
                }

                _scrollTimer.Enabled = (_scrollSpeed != 0);
                _control.Invalidate();
            }
            else if (_dragMode == DragMode.Panning)
            {
                BeginSamplePosX = _dragStartPos + e.X;
                _control.Invalidate();
            }
        }

        public void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left && _dragMode == DragMode.None)
            {
                _captureFlag = true;
                _dragMode = DragMode.Scrolling;
                System.Windows.Input.Mouse.OverrideCursor = System.Windows.Input.Cursors.Hand;

                if (CurrentDraggedEvent != null)
                {
                    CurrentDraggedEvent(this, DragMoveEventArgs.Start);
                }

                long newSampleIndex = GetSampleIndexFromPosX(e.X);
                UpdateCurrentSampleIndex(newSampleIndex);

                _scrollSpeed = 0;

                _control.Invalidate();
            }
            else if (e.Button == MouseButtons.Right && _dragMode == DragMode.None)
            {
                _captureFlag = true;
                _dragMode = DragMode.Panning;
                _dragStartPos = BeginSamplePosX - e.X;
                System.Windows.Input.Mouse.OverrideCursor = System.Windows.Input.Cursors.ScrollWE;
            }
        }

        public void EnsureVisibleCurrentSampleIndex()
        {
            int width = _zoomRatio.CalcWidthFromPointCount(CurrentSampleIndex);
            int currentSamplePosX = width + this.BeginSamplePosX;
            if (currentSamplePosX < 0)
            {
                this.BeginSamplePosX = -width;
            }
            else if (currentSamplePosX > _control.ClientSize.Width - 1)
            {
                this.BeginSamplePosX = _control.ClientSize.Width - 1 - width;
            }
        }

        public void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            if (e.KeyCode == Keys.Right)
            {
                UpdateCurrentSampleIndex(CurrentSampleIndex + 1);
                EnsureVisibleCurrentSampleIndex();
                _control.Invalidate();
            }
            else if (e.KeyCode == Keys.Left)
            {
                UpdateCurrentSampleIndex(CurrentSampleIndex - 1);
                EnsureVisibleCurrentSampleIndex();
                _control.Invalidate();
            }
            else if (e.KeyCode == Keys.Up)
            {
                if (UpdateZoomValue(_zoomRatio.Index + 1))
                {
                    _control.Invalidate();
                }
            }
            else if (e.KeyCode == Keys.Down)
            {
                if (UpdateZoomValue(_zoomRatio.Index - 1))
                {
                    _control.Invalidate();
                }
            }
        }

        public void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left && _dragMode == DragMode.Scrolling)
            {
                if (CurrentDraggedEvent != null)
                {
                    CurrentDraggedEvent(this, DragMoveEventArgs.End);
                }
                _dragMode = DragMode.None;
                _captureFlag = false;
                _scrollSpeed = 0;
            }
            else if (e.Button == MouseButtons.Right && _dragMode == DragMode.Panning)
            {
                _dragMode = DragMode.None;
                _captureFlag = false;
            }

            _control.Invalidate();
        }

        public void OnMouseWheel(MouseEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Control) != 0)
            {
                if (e.Delta > 0)
                {
                    if (UpdateZoomValue(_zoomRatio.Index + 1))
                    {
                        _control.Invalidate();
                    }
                }
                else if (e.Delta < 0)
                {
                    if (UpdateZoomValue(_zoomRatio.Index - 1))
                    {
                        _control.Invalidate();
                    }
                }
            }
        }
    }
}
