﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;

namespace NintendoWare.NwSoundSpyPlugin.Models
{
    /// <summary>
    /// NwSnd サウンドステータス Spy モデルです。
    /// </summary>
    public sealed class SoundStatusInfoSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct SoundStatusInfoPacketData
        /// {
        ///     struct SoundInfo
        ///     {
        ///         u32 instanceId;
        ///         u32 soundId;
        ///         u8 statusBitFlag;
        ///         u8 padding1[3];
        ///     };
        ///
        ///     u32 soundCount;
        ///     SoundInfo soundInfoArray[soundCount];
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        // TODO:名前変えたい。○○Infoでなく、○○に。"トーンマップ"と一緒に検討。
        public class SoundInfo
        {
            public const byte StatusStart = (1 << 0);
            public const byte StatusPause = (1 << 1);
            public const byte StatusPrepared = (1 << 2);

            public SpyGlobalTime Timestamp { get; private set; }
            public SpyTime BelongingFrame { get; private set; }
            public uint InstanceId { get; private set; }
            public uint SoundId { get; private set; }
            public byte StatusBitFlag { get; private set; }

            public static SoundInfo CreateSoundInfo(
                SpyGlobalTime timestamp,
                SpyTime belongingFrame,
                BinaryReader reader)
            {
                if (reader == null)
                {
                    return null;
                }

                SoundInfo newInfo = new SoundInfo();

                newInfo.InstanceId = reader.ReadUInt32();
                newInfo.SoundId = reader.ReadUInt32();
                newInfo.StatusBitFlag = reader.ReadByte();
                reader.BaseStream.Position += 3; // パディング分進めておく
                newInfo.Timestamp = timestamp;
                newInfo.BelongingFrame = belongingFrame;

                return newInfo;
            }
        }

        public class SoundInfoChangedEventArgs : EventArgs
        {
            public IList<SoundInfo> SoundInfos { get; internal set; }
            public SpyGlobalTime Timestamp { get; internal set; }
            public SpyTime BelongingFrame { get; internal set; }
        }

        private readonly List<SoundInfo> _newSoundInfos = new List<SoundInfo>();
        public event EventHandler<SoundInfoChangedEventArgs> SoundInfoChanged = null;

        public IList<SoundInfo> NewSoundInfos
        {
            get { return _newSoundInfos; }
        }

        private class SoundStatusInfos
        {
            private readonly List<SoundInfo> _newSoundInfos = new List<SoundInfo>();

            public SpyGlobalTime Timestamp { get; set; }

            public SpyTime BelongingFrame { get; set; }

            public uint SoundCount { get; set; }

            public IList<SoundInfo> NewSoundInfos
            {
                get { return _newSoundInfos; }
            }
        }

        private readonly List<SoundStatusInfos> _allSoundStatusInfos = new List<SoundStatusInfos>();

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                return;
            }

            var reader = CreateDataReader(dataBlock);

            uint soundCount = reader.ReadUInt32();

            var belongingFrame = GetBelongingFrame(dataBlock.Timestamp);

            if (SoundInfoChanged != null)
            {
                NewSoundInfos.Clear();

                for (int i = 0; i < soundCount; i++)
                {
                    SoundInfo newInfo = SoundInfo.CreateSoundInfo(
                        dataBlock.Timestamp,
                        GetBelongingFrame(dataBlock.Timestamp),
                        reader);

                    if (newInfo != null)
                    {
                        NewSoundInfos.Add(newInfo);
                    }
                }

                OnSoundInfoChanged(dataBlock.Timestamp, belongingFrame);
            }
            else
            {
                var infos = new SoundStatusInfos()
                {
                    Timestamp = dataBlock.Timestamp,
                    BelongingFrame = belongingFrame,
                    SoundCount = soundCount
                };

                for (int i = 0; i < soundCount; i++)
                {
                    SoundInfo newInfo = SoundInfo.CreateSoundInfo(
                        dataBlock.Timestamp,
                        belongingFrame,
                        reader);

                    if (newInfo != null)
                    {
                        infos.NewSoundInfos.Add(newInfo);
                    }
                }

                _allSoundStatusInfos.Add(infos);
            }
        }

        private void OnSoundInfoChanged(SpyGlobalTime timestamp, SpyTime belongingFrame)
        {
            if (SoundInfoChanged != null)
            {
                SoundInfoChanged(this, new SoundInfoChangedEventArgs()
                {
                    SoundInfos = NewSoundInfos,
                    Timestamp = timestamp,
                    BelongingFrame = belongingFrame,
                });
            }
        }

        public void ReadAllSoundStatusInfos()
        {
            if (SoundInfoChanged != null)
            {
                foreach (var info in _allSoundStatusInfos)
                {
                    SoundInfoChanged(this, new SoundInfoChangedEventArgs()
                    {
                        SoundInfos = info.NewSoundInfos,
                        Timestamp = info.Timestamp,
                        BelongingFrame = info.BelongingFrame
                    });
                }
            }
        }
    }
}
