﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using NintendoWare.Spy;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class WaveformCache
    {
        private const double FramesPerSecond = 1000000.0; // 1 [frame] : 1 [usec]
        private const double SecondsPerFrame = 1 / FramesPerSecond;

        private readonly Dictionary<double, MinMaxCollection> _minMaxCache = new Dictionary<double, MinMaxCollection>();

        public interface IWaveProvider
        {
            /// <summary>
            /// 総フレーム数です。
            /// </summary>
            int WaveformCount { get; }

            /// <summary>
            /// 指定した時刻が属するフレームのインデックスを取得します。
            /// </summary>
            /// <param name="time"></param>
            /// <returns>
            /// 指定した時刻を含むフレームが存在しないときは次のフレームのインデックスの１の補数（負の値）を返します。
            /// </returns>
            int GetWaveformIndex(SpyGlobalTime time);

            /// <summary>
            /// 指定したインデックスの波形データを取得します。
            /// </summary>
            /// <param name="waveformIndex"></param>
            /// <returns></returns>
            WaveformSpyModel.Waveform GetWaveform(int waveformIndex);
        }

        private struct WaveformSample
        {
            public WaveformSpyModel.Waveform Waveform { get; set; }

            public int Index { get; set; }

            public double Time { get; set; }

            public double GetValue(WaveformSpyModel.ChannelIndex channel)
            {
                return this.Waveform.SampleData.Samples[(int)channel][this.Index];
            }
        }

        private struct MinMaxSample
        {
            public MinMax MinMax { get; set; }

            public int Index { get; set; }

            public double Time { get; set; }

            public double GetMin(int channel)
            {
                return this.MinMax.Min[channel][this.Index];
            }

            public double GetMax(int channel)
            {
                return this.MinMax.Max[channel][this.Index];
            }
        }

        private class MinMax
        {
            public const int MaxSampleCount = 256;

            /// <summary>
            /// 区間の通し番号です。
            /// </summary>
            public long FrameIndex { get; set; }

            /// <summary>
            /// 区間の開始位置の時間です。
            /// </summary>
            public double Time { get; set; }

            /// <summary>
            /// Min, Max 配列内に格納されたサンプルの数です。
            /// </summary>
            public int SampleCount { get; set; }

            /// <summary>
            /// 最小値の配列です。
            /// (Min[channel][sampleIndex])
            /// </summary>
            public double[][] Min { get; private set; }

            /// <summary>
            /// 最大値の配列です。
            /// (Max[channel][sampleIndex])
            /// </summary>
            public double[][] Max { get; private set; }

            /// <summary>
            /// MinMax 生成時の WaveCache.SourceUpdateCount の値です。
            /// MinMax が古くなっているかの判断に使います。
            /// </summary>
            public long SourceUpdateCount { get; set; }

            /// <summary>
            /// 区間内の全サンプルの計算が完了したかを表します。
            /// 元データに欠落があったときは false になります。
            /// その場合、SourceUpdateCount が古くなっていたら再計算を行います。
            /// </summary>
            public bool IsCompleted { get; set; }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="channelCount">保持するチャンネル数です。</param>
            public MinMax(int channelCount)
            {
                this.Min = new double[channelCount][].Populate(_ => new double[MaxSampleCount]);
                this.Max = new double[channelCount][].Populate(_ => new double[MaxSampleCount]);
            }
        }

        private class MinMaxCollection
        {
            public double Scale { get; set; }

            public CacheDictionary<long, MinMax> Items { get; } = new CacheDictionary<long, MinMax>(20);

            public void Clear()
            {
                this.Items.Clear();
            }
        }

        public IWaveProvider WaveProvider { get; set; }

        public List<WaveformSpyModel.ChannelIndex> ChannelMapping { get; } = new List<WaveformSpyModel.ChannelIndex>();

        /// <summary>
        /// モデルデータから更新通知の回数をカウントします。
        /// キャッシュデータが古くなっているかの判断に使用されます。
        /// </summary>
        private long SourceUpdateCount { get; set; }

        public int SampleRate { get; set; }

        public PeakDataManager PeakDataManager { get; set; }

        public WaveformCache()
        {
        }

        /// <summary>
        /// モデルデータが更新されたことを通知します。
        /// </summary>
        public void SourceUpdated()
        {
            this.SourceUpdateCount++;
        }

        public void ClearCache()
        {
            _minMaxCache.Clear();
            this.SourceUpdateCount = 0;
        }

        public IEnumerable<LineChartSample> EnumerateSamples(double beginX, double endX, double scale, int channelNumber)
        {
            Ensure.Argument.Range(channelNumber, 0, this.ChannelMapping.Count);

            var endTime = SecondsPerFrame * endX;
            var onePixelPerSample = this.SampleRate * SecondsPerFrame * scale;

            if (onePixelPerSample < 16)
            {
                if (onePixelPerSample < 2)
                {
                    var channelIndex = this.ChannelMapping[channelNumber];

                    foreach (var sample in this.EnumerateWaveformSamples(SecondsPerFrame * beginX))
                    {
                        var x = FramesPerSecond * sample.Time;
                        var y = sample.GetValue(channelIndex);
                        yield return new LineChartSample(new Range1(x, x), new Range1(y, y));

                        if (sample.Time >= endTime)
                        {
                            yield break;
                        }
                    }
                }
                else
                {
                    var channelIndex = this.ChannelMapping[channelNumber];
                    var accumulatedX = 0.0;
                    var min = double.MaxValue;
                    var max = double.MinValue;
                    var sampleCount = 0;

                    foreach (var sample in this.EnumerateWaveformSamples(SecondsPerFrame * beginX))
                    {
                        var x = FramesPerSecond * sample.Time;
                        var y = sample.GetValue(channelIndex);

                        accumulatedX += x;
                        min = Math.Min(min, y);
                        max = Math.Max(max, y);
                        sampleCount++;

                        if (sampleCount < onePixelPerSample)
                        {
                            continue;
                        }

                        var center = accumulatedX / sampleCount;
                        yield return new LineChartSample(new Range1(center, center + 1), new Range1(min, max));

                        accumulatedX = 0.0;
                        min = y;
                        max = y;
                        sampleCount = 0;

                        if (sample.Time >= endTime)
                        {
                            yield break;
                        }
                    }
                }
            }
            else
            {
#if true
                if (this.PeakDataManager == null)
                {
                    yield break;
                }

                var channelIndex = (int)this.ChannelMapping[channelNumber];

                // beginX～endXまでを scaleの間隔で表示データを作るとLineChartの表示で隙間が入る問題がある
                // scaleの間隔を微妙に減らして、表示データの個数を1つ分多く詰め込むようにすると回避できる
                var interval = scale * 0.9999999;

                for (var x = beginX; x <= endX; x += interval)
                {
                    var sampleX = (long)(x * (SecondsPerFrame * this.SampleRate));
                    var result = this.PeakDataManager.GetMinMaxValue(sampleX, (long)onePixelPerSample);
                    if (result != null)
                    {
                        short min = 0;
                        short max = 0;
                        if (channelIndex < result.Minimum.Length)
                        {
                            min = result.Minimum[channelIndex];
                            max = result.Maximum[channelIndex];
                        }

                        yield return new LineChartSample(new Range1(x, x + 1), new Range1(min, max));
                    }
                }

#else  // 以前のキャッシュ処理を参考、復元のために残しておきます
                foreach (var sample in this.EnumerateMinMaxSamples(beginX, endX, scale, channelNumber))
                {
                    var index = sample.Index;
                    var x = sample.Time * TimeToFrame;
                    var minY = sample.GetMin(channelNumber);
                    var maxY = sample.GetMax(channelNumber);

                    yield return new LineChartSample(new Range1(x, x + 1), new Range1(minY, maxY));

                    if (sample.Time >= endTime)
                    {
                        yield break;
                    }
                }
#endif
            }
        }

        public double Evaluate(double x, int channelNumber)
        {
            Ensure.Argument.Range(channelNumber, 0, this.ChannelMapping.Count);

            var time = SecondsPerFrame * x;
            foreach (var sample in this.EnumerateWaveformSamples(time))
            {
                // イテレータの最初の要素を返します。
                var channelIndex = this.ChannelMapping[channelNumber];
                return sample.GetValue(channelIndex);
            }

            return double.NaN;
        }

        private IEnumerable<WaveformSample> EnumerateWaveformSamples(double startTime)
        {
            var sample = new WaveformSample();
            foreach (var frame in EnumerateWaveforms(SpyGlobalTime.FromSeconds(startTime)))
            {
                sample.Waveform = frame;

                var secPerSample = 1.0 / frame.SampleRate;

                var frameBegin = frame.Time.Seconds;
                var sampleIndex = 0;
                if (frameBegin < startTime)
                {
                    sampleIndex = (int)((startTime - frameBegin) * frame.SampleRate);
                }

                for (int i = sampleIndex; i < frame.SampleCount; i++)
                {
                    sample.Index = i;
                    sample.Time = frameBegin + i * secPerSample;

                    yield return sample;
                }
            }
        }

        private IEnumerable<WaveformSpyModel.Waveform> EnumerateWaveforms(SpyGlobalTime time)
        {
            if (this.WaveProvider == null)
            {
                yield break;
            }

            var waveformIndex = this.WaveProvider.GetWaveformIndex(time);

            if (waveformIndex < 0)
            {
                waveformIndex = ~waveformIndex;
            }

            WaveformSpyModel.Waveform previous = null;
            for (; waveformIndex < this.WaveProvider.WaveformCount; waveformIndex++)
            {
                var waveform = this.WaveProvider.GetWaveform(waveformIndex);
                if (waveform == null)
                {
                    continue;
                }

                if (previous != null && previous.Time >= waveform.Time)
                {
                    continue;
                }

                yield return waveform;
            }
        }

#if false // 以前のキャッシュ処理を参考、復元のために残しておきます
        private IEnumerable<MinMaxSample> EnumerateMinMaxSamples(double beginX, double endX, double scale, int channelNumber)
        {
            long startPos = (long)Math.Floor(beginX / scale);
            var endTime = endX * FrameToTime;

            var sample = new MinMaxSample();
            foreach (var minMax in EnumerateMinMaxFrames(SpyGlobalTime.FromSeconds(startPos * scale * FrameToTime), scale))
            {
                sample.MinMax = minMax;

                long framePos = minMax.FrameIndex * MinMax.MaxSampleCount;
                int sampleIndex = 0;
                if (framePos < startPos)
                {
                    sampleIndex = (int)(startPos - framePos);
                }

                for (int i = sampleIndex; i < minMax.SampleCount; i++)
                {
                    sample.Index = i;
                    sample.Time = (framePos + i) * scale * FrameToTime;

                    yield return sample;

                    if (sample.Time > endTime)
                    {
                        yield break;
                    }
                }
            }
        }

        private IEnumerable<MinMax> EnumerateMinMaxFrames(SpyGlobalTime time, double scale)
        {
            var screenX = (long)(time.Seconds * TimeToFrame / scale);

            var frameIndex = Math.Max(0, screenX / MinMax.MaxSampleCount);

            if (!_minMaxCache.ContainsKey(scale))
            {
                _minMaxCache[scale] = new MinMaxCollection() { Scale = scale, };
            }

            var minMaxCollection = _minMaxCache[scale];

            for (; ; frameIndex++)
            {
                MinMax minMax = null;
                if (minMaxCollection.Items.TryGetValue(frameIndex, out minMax))
                {
                    if (minMax.IsCompleted || minMax.SourceUpdateCount == this.SourceUpdateCount)
                    {
                        yield return minMax;
                        continue;
                    }
                }

                minMax = this.CreateMinMax(frameIndex, scale);

                if (minMax != null)
                {
                    minMaxCollection.Items[frameIndex] = minMax;

                    yield return minMax;
                }

                if (minMax == null || !minMax.IsCompleted)
                {
                    yield break;
                }
            }
        }

        private MinMax CreateMinMax(long frameIndex, double scale)
        {
            var lazyMinMax = new Lazy<MinMax>(() =>
            {
                var minMax = new MinMax(this.ChannelMapping.Count)
                {
                    FrameIndex = frameIndex,
                    SourceUpdateCount = this.SourceUpdateCount,
                };

                return minMax;
            });

            var numChannels = this.ChannelMapping.Count;

            var startX = frameIndex * MinMax.MaxSampleCount * scale;
            int count = 0;
            bool minMaxAvailable = false;

            double[] min = new double[numChannels];
            double[] max = new double[numChannels];

            min.Populate(double.MaxValue);
            max.Populate(double.MinValue);

            Action commit = () =>
            {
                var minMax = lazyMinMax.Value;

                for (int channel = 0; channel < numChannels; channel++)
                {
                    minMax.Min[channel][count] = min[channel];
                    minMax.Max[channel][count] = max[channel];
                }

                minMax.SampleCount++;
                count++;
                minMaxAvailable = false;
            };

            bool isCompleted = false;
            foreach (var data in EnumerateWaveformSamples(startX * FrameToTime))
            {
                minMaxAvailable = true;

                for (int channel = 0; channel < numChannels; channel++)
                {
                    var value = data.GetValue(this.ChannelMapping[channel]);

                    min[channel] = Math.Min(min[channel], value);
                    max[channel] = Math.Max(max[channel], value);
                }

                var endX = startX + (count + 1) * scale;
                if (data.Time * TimeToFrame < endX)
                {
                    continue;
                }

                commit();

                min.Populate(i => data.GetValue(this.ChannelMapping[i]));
                max.Populate(i => data.GetValue(this.ChannelMapping[i]));

                if (count >= MinMax.MaxSampleCount)
                {
                    isCompleted = true;
                    break;
                }
            }

            if (count < MinMax.MaxSampleCount && minMaxAvailable)
            {
                commit();
            }

            if (lazyMinMax.IsValueCreated)
            {
                var minMax = lazyMinMax.Value;
                minMax.IsCompleted = isCompleted;
                return minMax;
            }
            else
            {
                return null;
            }
        }
#endif
    }
}
