﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class StreamSoundPanelViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();

        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;

        private readonly Collection<StreamSoundItemViewModel> _streamSoundItems;

        public ObservableCollection<StreamSoundItemViewModel> StreamSoundViewableItems
        {
            get;
        }

        public StreamSoundPanelViewModel(SpyPlaybackService playbackService)
        {
            _streamSoundItems = new ObservableCollection<StreamSoundItemViewModel>();
            this.StreamSoundViewableItems = new ObservableCollection<StreamSoundItemViewModel>();

            Ensure.Argument.NotNull(playbackService);

            _playbackService = playbackService;

            PropertyChangedObservation.GetObserver(this, _playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateCurrentTime());
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            _streamSoundItems.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            _plotModel = model;
            this.SyncPlotItems();
        }

        private void SyncPlotItems()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            this.StreamSoundViewableItems.Clear();

            _streamSoundItems.ForEach(it => it.Dispose());
            _streamSoundItems.Clear();

            if (_plotModel != null)
            {
                CollectionChangedObservation.GetObserver(_observerOwner, _plotModel.Floats).AddHandlerForAddItems(
                    (sender, e) =>
                    {
                        this.UpdatePlotItems();
                    });
                CollectionChangedObservation.GetObserver(_observerOwner, _plotModel.States).AddHandlerForAddItems(
                    (sender, e) =>
                    {
                        this.UpdatePlotItems();
                    });
            }

            UpdatePlotItems();
            UpdateViewableItems();
        }

        private void UpdatePlotItems()
        {
            if (_plotModel == null)
            {
                return;
            }

            while (_streamSoundItems.All(streamSoundItem => streamSoundItem.FindFillingPercentagePlotFloat(_plotModel)
                                   && streamSoundItem.FindDelayStatePlotState(_plotModel)))
            {
                _streamSoundItems.Add(new StreamSoundItemViewModel(_streamSoundItems.Count));
            }
        }

        private void UpdateViewableItems()
        {
            if (_plotModel == null)
            {
                return;
            }

            foreach (StreamSoundItemViewModel streamSoundItem in _streamSoundItems)
            {
                if (!this.StreamSoundViewableItems.Contains(streamSoundItem) && !streamSoundItem.IsInvalid())
                {
                    this.StreamSoundViewableItems.Add(streamSoundItem);
                }
                else if (this.StreamSoundViewableItems.Contains(streamSoundItem) && streamSoundItem.IsInvalid())
                {
                    this.StreamSoundViewableItems.Remove(streamSoundItem);
                }
            }
        }

        private void UpdateCurrentTime()
        {
            foreach (StreamSoundItemViewModel streamSoundItem in _streamSoundItems)
            {
                if (_plotModel != null)
                {
                    // Cache 関連は atkspy 側の更新のタイミングが異なるため、ここで Find を行う
                    streamSoundItem.FindRemainingCachePercentagePlotFloat(_plotModel);
                }
                streamSoundItem.UpdateCurrentTime(_playbackService);
            }
            UpdateViewableItems();
        }
    }
}
