﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.IO;

namespace NintendoWare.CafeSpyPlugin.Foundation.Communications
{
    /// <summary>
    /// Cafe HostIO のストリームクラスです。
    /// TODO : CachedStream に改名する。
    /// </summary>
    /// <remarks>
    /// WriteCacheSize 分だけキャッシュします。
    /// </remarks>
    public class HostIOCafeStream : Stream
    {
        private const int WriteCacheSize = 1024;

        private readonly Stream _sourceStream;

        private readonly byte[] _writeCache = new byte[WriteCacheSize];
        private int _writeCacheCurrent = 0;

        //-----------------------------------------------------------------

        public HostIOCafeStream(Stream sourceStream)
        {
            Assertion.Argument.NotNull(sourceStream);
            _sourceStream = sourceStream;
        }

        //-----------------------------------------------------------------

        public Stream SourceStream
        {
            get { return _sourceStream; }
        }

        public override bool CanRead
        {
            get { return _sourceStream.CanRead; }
        }

        public override bool CanSeek
        {
            get { return _sourceStream.CanSeek; }
        }

        public override bool CanWrite
        {
            get { return _sourceStream.CanWrite; }
        }

        public override long Length
        {
            get { return _sourceStream.Length; }
        }

        public override long Position
        {
            get { return _sourceStream.Position; }
            set { _sourceStream.Position = value; }
        }

        //-----------------------------------------------------------------

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);

            if (disposing && _sourceStream != null)
            {
                _sourceStream.Close();
            }
        }

        public override void Flush()
        {
            this.FlushCache();
            _sourceStream.Flush();
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            return _sourceStream.Read(buffer, offset, count);
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            return _sourceStream.Seek(offset, origin);
        }

        public override void SetLength(long value)
        {
            _sourceStream.SetLength(value);
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            // 転送回数を削減するためにキャッシュして出力します。
            // バッファサイズを超える場合は、キャッシュ済みの内容を先に出力します。
            // count がバッファより大きい場合は、直接出力します。
            if (_writeCacheCurrent + count < _writeCache.Length)
            {
                Array.Copy(buffer, offset, _writeCache, _writeCacheCurrent, count);
                _writeCacheCurrent += count;
                return;
            }

            this.FlushCache();

            if (count > _writeCache.Length)
            {
                _sourceStream.Write(buffer, 0, count);
                return;
            }

            Array.Copy(buffer, offset, _writeCache, 0, count);
        }

        private void FlushCache()
        {
            if (_writeCacheCurrent > 0)
            {
                _sourceStream.Write(_writeCache, 0, _writeCacheCurrent);
                _writeCacheCurrent = 0;
            }
        }
    }
}
