﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Runtime.Serialization.Formatters.Binary;
using Nintendo.Foundation.Audio;

namespace WaveSoundEditorFormsDemo
{
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Operations;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Views;
    using Nintendo.AudioToolkit.Windows.Controls;
    using NintendoWare.SoundFoundation.Operations;

    public partial class Form1 : Form
    {
        private WaveSoundResourceData waveSoundData = null;
        private OperationHistory operationHistory = null;

        private IOperationExecutor executor = null;
        private ClipboardService clipboardService = null;

        private WaveSoundPresenter waveSoundPresenter = null;
        private WaveSoundPropertyPresenter waveSoundPropertyPresenter = null;

        public Form1()
        {
            InitializeComponent();

            this.operationHistory = new OperationHistory();
            this.operationHistory.ItemsChanged += delegate(object sender, EventArgs e)
            {
                this.UpdateMenu();
            };

            this.executor = new SoundMakerOperationExecutor(this.operationHistory);

            this.clipboardService = new ClipboardService();
            this.clipboardService.AddConverter(new ClipConverter());
            this.clipboardService.AddConverter(new TrackConverter());

            this.waveSoundPresenter = new WaveSoundPresenter();
            this.waveSoundPresenter.Initialize();
            this.waveSoundPresenter.BeginTransaction += (s, e) => this.executor.BeginTransaction();
            this.waveSoundPresenter.EndTransaction += (s, e) => this.executor.EndTransaction();
            this.waveSoundPresenter.CancelTransaction += (s, e) => this.executor.CancelTransaction();
            this.waveSoundPresenter.DropFiles += OnDropFiles;
            this.waveSoundPresenter.RequestDuplicate += OnRequestDuplicate;
            this.waveSoundPresenter.SelectionChanged += OnSelectionChanged;
            this.elementHostLeft.Child = this.waveSoundPresenter.View;

            this.waveSoundPropertyPresenter = new WaveSoundPropertyPresenter();
            this.waveSoundPropertyPresenter.Initialize();
            this.elementHostRight.Child = this.waveSoundPropertyPresenter.View;

            this.waveSoundData = this.CreateWaveSoundData();
            var waveSoundDM = new WaveSoundResource(this.waveSoundData, this.executor);
            this.waveSoundPresenter.SetModel(waveSoundDM);

            this.WaveSoundDataModel = waveSoundDM;

            this.UpdateMenu();
        }

        public WaveSoundResource WaveSoundDataModel { get; set; }

        /// <summary>
        /// テスト用データを作成します。
        /// </summary>
        /// <returns></returns>
        private WaveSoundResourceData CreateWaveSoundData()
        {
            var waveSound = new WaveSoundResourceData();

            for (int index = 0; index < 2; index++)
            {
                var track = new WaveSoundTrackResourceData() { };

                for (int i = 0; i < 1; i++)
                {
                    var clip = new WaveSoundClipResourceData();
                    clip.IsEnabled = true;
                    clip.Position = index * 2000 + i * 10000;
                    clip.StartOffset = 0;
                    clip.Duration = 5000;
                    clip.Volume = 0.5F;
                    clip.Pitch = 1.0F;
                    clip.Pan = 0.5F;
                    clip.Path = "hoge.wav";

                    track.ClipResources.Add(clip);
                }

                waveSound.TrackResources.Add(track);
            }
            return waveSound;
        }

        private void UpdateMenu()
        {
            this.menuItem_Undo.Enabled = this.operationHistory.CanUndo();
            this.menuItem_Redo.Enabled = this.operationHistory.CanRedo();
        }

        private void OnDropFiles(object sender, WaveSoundPresenter.DropFileEventArgs e)
        {
            var filePath = e.FilePaths[0];
            WaveFileInfo fileInfo = null;
            try
            {
                fileInfo = WaveFileReader.ReadWaveFileInfo(filePath);
            }
            catch
            {
            }

            if (fileInfo == null)
            {
                MessageBox.Show(WaveSoundEditorFormsDemo.Properties.Resources.ErrorMessge_NotWaveFile);
                return;
            }

            var clip = new WaveSoundClipResourceData();
            clip.Position = (int)e.Position.X;
            clip.Row = (int)e.Position.Y;
            clip.Duration = (int)fileInfo.Duration.TotalMilliseconds;
            clip.Path = filePath;
            var clipDM = new WaveSoundClipResource(clip);
            waveSoundPresenter.AddClip(e.TrackDataModel, clipDM);

            waveSoundPresenter.UnselectClipAll();
            waveSoundPresenter.SelectClip(clipDM);
        }

        private void OnRequestDuplicate(object sender, WaveSoundPresenter.RequestDuplicateEventArgs e)
        {
            if (e.Instance is WaveSoundClipResource)
            {
                e.NewInstance = ClipDuplicator.Duplicate(e.Instance as WaveSoundClipResource);
            }
        }

        private void OnSelectionChanged(object sender, EventArgs e)
        {
            if (this.waveSoundPresenter.SelectedClips.Count() == 1)
            {
                var clip = waveSoundPresenter.SelectedClips.First();
                this.waveSoundPropertyPresenter.SetModel(clip);
            }
            else
            {
                this.waveSoundPropertyPresenter.SetModel(null);
            }
        }

        /// <summary>
        /// 元に戻す
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Undo(object sender, EventArgs e)
        {
            this.operationHistory.Undo();
            this.UpdateMenu();
        }

        /// <summary>
        /// やり直し
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Redo(object sender, EventArgs e)
        {
            this.operationHistory.Redo();
            this.UpdateMenu();
        }

        /// <summary>
        /// 切り取り
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Cut(object sender, EventArgs e)
        {
            this.OnClick_Copy(sender, e);
            this.OnClick_Delete(sender, e);
        }

        /// <summary>
        /// コピー
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Copy(object sender, EventArgs e)
        {
            // トラックとクリップの両方が選択されている時にはクリップのコピーを優先しています。
            if (this.waveSoundPresenter.SelectedClips.Count() > 0)
            {
                this.clipboardService.To(this.waveSoundPresenter.SelectedClips);
            }
            else if (this.waveSoundPresenter.SelectedTracks.Count() > 0)
            {
                this.clipboardService.To(this.waveSoundPresenter.SelectedTracks);
            }
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Paste(object sender, EventArgs e)
        {
            if (this.clipboardService.Contains == false)
            {
                return;
            }

            this.waveSoundPresenter.UnselectClipAll();

            foreach (var model in this.clipboardService.From())
            {
                if (model is WaveSoundClipResource)
                {
                    var clipDM = model as WaveSoundClipResource;
                    this.waveSoundPresenter.AddClipToSelectedTrack(model as WaveSoundClipResource);
                    this.waveSoundPresenter.SelectClip(clipDM);

                }
                else if (model is WaveSoundTrackResource)
                {
                    this.waveSoundPresenter.AddTrack(model as WaveSoundTrackResource);
                }
            }
        }

        /// <summary>
        /// 削除
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Delete(object sender, EventArgs e)
        {
            this.operationHistory.BeginTransaction();
            this.waveSoundPresenter.RemoveSelectedClips();
            this.operationHistory.EndTransaction();
        }

        /// <summary>
        /// すべて選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_SelectAll(object sender, EventArgs e)
        {
            this.waveSoundPresenter.SelectClipAll();
        }

        /// <summary>
        /// トラックの追加
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_AddTrack(object sender, EventArgs e)
        {
            var track = new WaveSoundTrackResourceData();
            var trackDM = new WaveSoundTrackResource(track);
            this.waveSoundPresenter.AddTrack(trackDM);
        }

        /// <summary>
        /// 選択したトラックを削除
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_DeleteTrack(object sender, EventArgs e)
        {
            this.waveSoundPresenter.RemoveSelectedTracks();
        }

        /// <summary>
        /// デバッグ用のデータ表示
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_ShowModel(object sender, EventArgs e)
        {
            Console.WriteLine("--------");
            foreach (var track in this.waveSoundData.TrackResources)
            {
                Console.WriteLine("  Track");
                foreach (var clip in track.ClipResources)
                {
                    Console.WriteLine("    Clip Position = {0} : StartOffset = {1} : Duration = {2}", clip.Position, clip.StartOffset, clip.Duration);
                }
            }
        }

        public class ClipConverter : ClipboardService.IConverter<WaveSoundClipResource, WaveSoundClipDataExchange>
        {
            public override WaveSoundClipDataExchange To(WaveSoundClipResource source)
            {
                var target = new WaveSoundClipDataExchange();
                target.IsEnabled = source.IsEnabled;
                target.Path = source.Path;
                target.Position = source.Position;
                target.StartOffset = source.StartOffset;
                target.Duration = source.Duration;
                target.Volume = source.Volume;
                target.Pitch = source.Pitch;
                target.Pan = source.Pan;
                return target;
            }

            public override WaveSoundClipResource From(WaveSoundClipDataExchange source)
            {
                var target = new WaveSoundClipResourceData();
                target.IsEnabled = source.IsEnabled;
                target.Path = source.Path;
                target.Position = source.Position;
                target.StartOffset = source.StartOffset;
                target.Duration = source.Duration;
                target.Volume = source.Volume;
                target.Pitch = source.Pitch;
                target.Pan = source.Pan;
                return new WaveSoundClipResource(target);
            }
        }

        public class TrackConverter : ClipboardService.IConverter<WaveSoundTrackResource, WaveSoundTrackDataExchange>
        {
            public override WaveSoundTrackDataExchange To(WaveSoundTrackResource source)
            {
                var target = new WaveSoundTrackDataExchange();

                foreach (var clip in source.Clips)
                {
                    target.Clips.Add(new ClipConverter().To(clip) as WaveSoundClipDataExchange);
                }

                return target;
            }

            public override WaveSoundTrackResource From(WaveSoundTrackDataExchange source)
            {
                var target = new WaveSoundTrackResourceData();

                foreach(var clip in source.Clips)
                {
                    target.ClipResources.Add((new ClipConverter().From(clip) as WaveSoundClipResource).Data);
                }

                return new WaveSoundTrackResource(target);
            }
        }

        [Serializable]
        public class WaveSoundClipDataExchange
        {
            public bool IsEnabled { get; set; }
            public int Position { get; set; }
            public int StartOffset { get; set; }
            public int Duration { get; set; }
            public string Path { get; set; }
            public float Volume { get; set; }
            public float Pitch { get; set; }
            public float Pan { get; set; }
        }

        [Serializable]
        public class WaveSoundTrackDataExchange
        {
            public IList<WaveSoundClipDataExchange> Clips = new List<WaveSoundClipDataExchange>();
        }
    }

    public static class ClipDuplicator
    {
        public static WaveSoundClipResource Duplicate(WaveSoundClipResource source)
        {
            var target = new WaveSoundClipResourceData();
            target.IsEnabled = source.IsEnabled;
            target.Path = source.Path;
            target.Position = source.Position;
            target.StartOffset = source.StartOffset;
            target.Duration = source.Duration;
            target.Volume = source.Volume;
            target.Pitch = source.Pitch;
            target.Pan = source.Pan;
            return new WaveSoundClipResource(target);
        }
    }

    /// <summary>
    /// クリップボードサービス
    /// </summary>
    public class ClipboardService
    {
        private const string DataIdentifier = "WaveSouneEditor";

        private Dictionary<Type, IConverter> toConverters = new Dictionary<Type, IConverter>();
        private Dictionary<Type, IConverter> fromConverters = new Dictionary<Type, IConverter>();

        public bool Contains
        {
            get
            {
                return Clipboard.ContainsData(DataIdentifier);
            }
        }

        public void AddConverter<T1, T2>(IConverter<T1, T2> converter) where T1 : class where T2 : class
        {
            this.toConverters.Add(typeof(T1), converter);
            this.fromConverters.Add(typeof(T2), converter);
        }

        public void To(IEnumerable<object> models)
        {
            var datas = new List<object>();

            foreach (var model in models)
            {
                var converter = this.toConverters[model.GetType()];
                var cdata = converter.To(model);

                using (MemoryStream stream = new MemoryStream())
                {
                    var formatter = new BinaryFormatter();
                    formatter.Serialize(stream, cdata);
                    datas.Add(stream.GetBuffer());
                }
            }

            Clipboard.SetData(DataIdentifier, datas);
        }

        public IEnumerable<object> From()
        {
            var models = new List<object>();

            var datas = Clipboard.GetData(DataIdentifier) as List<object>;
            foreach (var data in datas)
            {
                using (MemoryStream stream = new MemoryStream(data as byte[]))
                {
                    var formatter = new BinaryFormatter();
                    var cdata = formatter.Deserialize(stream);
                    var converter = this.fromConverters[cdata.GetType()];
                    var model = converter.From(cdata);

                    models.Add(model);
                }
            }
            return models;
        }

        public interface IConverter
        {
            Type ToType { get; }
            Type FromType { get; }

            object To(object o);
            object From(object o);
        }

        public abstract class IConverter<T1, T2> : IConverter where T1 : class where T2 : class
        {
            Type IConverter.ToType
            {
                get { return typeof(T1); }
            }

            Type IConverter.FromType
            {
                get { return typeof(T2); }
            }

            object IConverter.To(object o)
            {
                return this.To(o as T1);
            }

            object IConverter.From(object o)
            {
                return this.From(o as T2);
            }

            public abstract T2 To(T1 source);
            public abstract T1 From(T2 target);
        }
    }

    /// <summary>
    /// SoundMaker用のOperationExecutorです。
    /// </summary>
    public class SoundMakerOperationExecutor : IOperationExecutor
    {
        private readonly OperationHistory operationHistory = null;

        public SoundMakerOperationExecutor(OperationHistory operationHistory)
        {
            this.operationHistory = operationHistory;
        }

        void IOperationExecutor.ExecuteOperation(Action executeAction, Action rollbackAction)
        {
            var operation = new ActionOperation(executeAction, rollbackAction);
            operation.Execute();
            this.operationHistory.AddOperation(operation);
        }

        void IOperationExecutor.BeginTransaction()
        {
            this.operationHistory.BeginTransaction();
        }

        void IOperationExecutor.EndTransaction()
        {
            this.operationHistory.EndTransaction();
        }

        void IOperationExecutor.CancelTransaction()
        {
            this.operationHistory.CancelTransaction();
        }
    }

    public class ActionOperation : OperationImpl
    {
        private Action executeAction = null;
        private Action rollbackAction = null;

        public ActionOperation(Action executeAction, Action rollbackAction)
        {
            this.executeAction = executeAction;
            this.rollbackAction = rollbackAction;
        }

        protected override bool ExecuteInternal()
        {
            this.executeAction();
            return true;
        }

        protected override bool RollbackInternal()
        {
            this.rollbackAction();
            return true;
        }
    }
}
