﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Windows.Threading;
using System.Reflection;

namespace Nintendo.AudioToolkit.Views
{
    using Nintendo.Foundation.Audio;
    using Nintendo.Foundation.ComponentModel;
    using Nintendo.Foundation.Collections;
    using Nintendo.Foundation.Windows.Input;
    using Nintendo.Foundation.Windows.Primitives;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.AudioToolkit.Extensions;

    /// <summary>
    ///
    /// </summary>
    public class WaveSoundViewModel : INotifyPropertyChanged
    {
        private readonly WeakReference<WaveSoundResource> model = new WeakReference<WaveSoundResource>(null);

        public event PropertyChangedEventHandler PropertyChanged;
        private CollectionChangedObserver collectionChangedObsever = null;

        private ObservableCollection<TrackViewModel> tracks = new ObservableCollection<TrackViewModel>();
        private double horizontalScale = 0.14;
        private double horizontalSnap = 1.0;
        private double horizontalOffset = 0.0;
        private double verticalSnap = 100.0;
        private double scaledHorizontalOffset = 0.0;
        private Size viewportSize;
        private double horizontalViewRange = 0.0;

        private double maximumHorizontalOffset = 0.0;

        private Point mousePosition;

        public ICommand ResizeStartedCommand { get; set; }
        public ICommand ResizeCompletedCommand { get; set; }
        public ICommand ResizeCanceledCommand { get; set; }
        public ICommand DropStartedCommand { get; set; }
        public ICommand DropCompletedCommand { get; set; }
        public ICommand DropCanceledCommand { get; set; }

        public ICommand DropFileCommand { get; set; }
        public ICommand RequestDuplicateCommand { get; set; }
        public ICommand ZoomToFitCommand { get; set; }
        public ICommand RequestMoveCommand { get; set; }
        public ICommand CheckDragClipCommand { get; set; }

        public event EventHandler SelectionChanged;

        public WaveSoundViewModel(WaveSoundResource model)
        {
            this.model.SetTarget(model);

            this.collectionChangedObsever = new CollectionChangedObserver(this.Model.Tracks, true);
            this.collectionChangedObsever.AddHandlerForAttachItems(
                delegate (System.Collections.IEnumerable collection, CollectionChangedObserver.AttachItemsInfo info)
                {
                    foreach (var item in info.Items)
                    {
                        var trackDM = item as WaveSoundTrackResource;
                        var trackVM = new TrackViewModel(trackDM) { OwnerWaveSound = this };
                        trackVM.UpdateScale(this.horizontalScale);
                        trackVM.PropertyChanged += OnTrackPropertyChanged;
                        trackVM.Clips.CollectionChanged += OnClipCollectionChanged;

                        int index = this.Model.Tracks.IndexOf(trackDM);
                        this.tracks.Insert(index, trackVM);
                    }
                });
            this.collectionChangedObsever.AddHandlerForDetachItems(
                delegate (System.Collections.IEnumerable collection, CollectionChangedObserver.DetachItemsInfo info)
                {
                    foreach (var item in info.Items)
                    {
                        var trackDM = item as WaveSoundTrackResource;
                        this.tracks.Where(t => t.Model == trackDM).ToList().ForEach(t =>
                            {
                                this.tracks.Remove(t);
                                t.PropertyChanged -= OnTrackPropertyChanged;
                                t.Clips.CollectionChanged -= OnClipCollectionChanged;
                            });
                    }
                });

            this.AddPropertyChanged(t => t.ScaledHorizontalOffset,
                delegate (WaveSoundViewModel target)
                {
                    this.UpdateHorizontalOffset();
                });

            this.AddPropertyChanged(t => t.HorizontalScale,
                delegate (WaveSoundViewModel target)
                {
                    this.UpdateScale();
                });

            this.AddPropertyChanged(t => t.ViewportSize,
                delegate (WaveSoundViewModel target)
                {
                    this.UpdateViewportSize();
                });

            this.AddPropertyChanged(t => t.VerticalSnap,
                delegate (WaveSoundViewModel target)
                {
                    this.UpdateSnap();
                });

            foreach (var trackDM in this.Model.Tracks)
            {
                var trackVM = new TrackViewModel(trackDM) { OwnerWaveSound = this };
                trackVM.PropertyChanged += OnTrackPropertyChanged;
                trackVM.Clips.CollectionChanged += OnClipCollectionChanged;
                this.tracks.Add(trackVM);
            }

            this.UpdateMaximumHorizontalOffset();

            this.ZoomToFitCommand = new DelegateCommand((o) => this.ZoomToFit());
        }

        public ObservableCollection<TrackViewModel> Tracks
        {
            get { return this.tracks; }
        }

        public double HorizontalScale
        {
            get { return this.horizontalScale; }
            set { this.SetProperty(this.PropertyChanged, ref this.horizontalScale, value); }
        }

        public double HorizontalSnap
        {
            get { return this.horizontalSnap; }
            set { this.SetProperty(this.PropertyChanged, ref this.horizontalSnap, value); }
        }

        public double ScaledHorizontalOffset
        {
            get { return this.scaledHorizontalOffset; }
            set
            {
                if (value < 0.0)
                {
                    value = 0.0;
                }
                this.SetProperty(this.PropertyChanged, ref this.scaledHorizontalOffset, value);

                // HorizontalOffsetが更新されたことにします。
                this.horizontalOffset = this.scaledHorizontalOffset / this.horizontalScale;
                this.PropertyChanged.Raise(() => HorizontalOffset);

                this.UpdateMaximumHorizontalOffset();
            }
        }

        public double HorizontalOffset
        {
            get { return this.horizontalOffset; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.horizontalOffset, value);

                // ScaledHorizontalOffsetが更新されたことにします。
                this.scaledHorizontalOffset = this.horizontalOffset * this.horizontalScale;
                this.PropertyChanged.Raise(() => ScaledHorizontalOffset);
            }
        }

        public double VerticalSnap
        {
            get { return this.verticalSnap; }
            set { this.SetProperty(this.PropertyChanged, ref this.verticalSnap, value); }
        }

        public double MaximumHorizontalOffset
        {
            get { return this.maximumHorizontalOffset; }
            set { this.SetProperty(this.PropertyChanged, ref this.maximumHorizontalOffset, value); }
        }

        public Size ViewportSize
        {
            get { return this.viewportSize; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.viewportSize, value);

                this.horizontalViewRange = this.viewportSize.Width / this.horizontalScale;
                this.PropertyChanged.Raise(() => HorizontalViewRange);

                this.UpdateMaximumHorizontalOffset();
            }
        }

        public double HorizontalViewRange
        {
            get { return this.horizontalViewRange; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.horizontalViewRange, value);

                this.horizontalScale = viewportSize.Width / value;
                this.PropertyChanged.Raise(() => HorizontalScale);

                this.scaledHorizontalOffset = this.horizontalOffset * this.horizontalScale;
                this.PropertyChanged.Raise(() => ScaledHorizontalOffset);

                this.UpdateMaximumHorizontalOffset();
            }
        }

        public Point MousePosition
        {
            get { return this.mousePosition; }
            set { this.SetProperty(this.PropertyChanged, ref this.mousePosition, value); }
        }

        public void UpdateHorizontalOffset()
        {
            this.tracks.ToList().ForEach(t => t.UpdateHorizontalOffset());
        }

        public void UpdateScale()
        {
            this.tracks.ToList().ForEach(t => t.UpdateScale(this.HorizontalScale));
        }

        public void UpdateSnap()
        {
            this.tracks.ToList().ForEach(t => t.UpdateSnap(this.HorizontalSnap, this.VerticalSnap));
        }

        public void OnSelectionChanged()
        {
            if (this.SelectionChanged != null)
            {
                this.SelectionChanged(this, EventArgs.Empty);
            }
        }

        private WaveSoundResource Model
        {
            get
            {
                WaveSoundResource model = null;
                this.model.TryGetTarget(out model);
                return model;
            }
        }

        private void UpdateViewportSize()
        {
            this.tracks.ToList().ForEach(t => t.UpdateViewportSize());
        }

        private void UpdateMaximumHorizontalOffset()
        {
            var value = this.tracks
                .Select(t => t.MaximumHorizontalOffset)
                .Max();
            var viewRight = this.HorizontalOffset + this.HorizontalViewRange;

            this.MaximumHorizontalOffset = Math.Max(value, viewRight);
        }

        private void ZoomToFit()
        {
            var usingTracks = this.tracks
                .Where(t => t.Clips.Count > 0);
            if (usingTracks.IsEmpty() == false)
            {
                var left = 0.0;
                var right = this.tracks
                    .Select(t => t.MaximumHorizontalOffset)
                    .Max();

                if (left < right)
                {
                    this.HorizontalOffset = left;
                    this.HorizontalViewRange = right - left;
                }
            }
        }

        private void OnTrackPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            var trackVM = sender as TrackViewModel;
            if (e.PropertyName == PropertyUtility.NameOf(() => trackVM.MaximumHorizontalOffset))
            {
                this.UpdateMaximumHorizontalOffset();
            }
        }

        private void OnClipCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.UpdateMaximumHorizontalOffset();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TrackViewModel : INotifyPropertyChanged
    {
        private readonly WeakReference<WaveSoundTrackResource> model = new WeakReference<WaveSoundTrackResource>(null);

        public event PropertyChangedEventHandler PropertyChanged;
        private CollectionChangedObserver collectionChangedObsever = null;

        private ObservableCollection<ClipViewModel> clips = new ObservableCollection<ClipViewModel>();
        private double horizontalScale;
        private double horizontalSnap;
        private double verticalSnap;
        private bool isSelected;
        private double minimumHorizontalOffset;
        private double maximumHorizontalOffset;

        public TrackViewModel(WaveSoundTrackResource model)
        {
            this.model.SetTarget(model);

            this.collectionChangedObsever = new CollectionChangedObserver(this.Model.Clips, true);
            this.collectionChangedObsever.AddHandlerForAttachItems(
                delegate (System.Collections.IEnumerable collection, CollectionChangedObserver.AttachItemsInfo info)
                {
                    foreach (var item in info.Items)
                    {
                        var clipDM = item as WaveSoundClipResource;
                        if (this.Contains(clipDM) == true)
                        {
                            continue;
                        }

                        var clipVM = new ClipViewModel(clipDM) { OwnerTrack = this };
                        clipVM.UpdateScale(this.horizontalScale);
                        clipVM.UpdateSnap(this.horizontalSnap, this.verticalSnap);
                        clipVM.PropertyChanged += OnClipPropertyChanged;

                        int index = this.Model.Clips.IndexOf(clipDM);
                        this.clips.Insert(index, clipVM);
                    }
                });
            this.collectionChangedObsever.AddHandlerForDetachItems(
                delegate (System.Collections.IEnumerable collection, CollectionChangedObserver.DetachItemsInfo info)
                {
                    foreach (var item in info.Items)
                    {
                        var clipDM = item as WaveSoundClipResource;
                        if (this.Contains(clipDM) == false)
                        {
                            continue;
                        }

                        this.clips.Where(t => t.Model == clipDM).ToList().ForEach(t => { this.clips.Remove(t); t.PropertyChanged -= OnClipPropertyChanged; });
                    }
                });

            foreach (var clipDM in this.Model.Clips)
            {
                var clipVM = new ClipViewModel(clipDM) { OwnerTrack = this };
                clipVM.PropertyChanged += OnClipPropertyChanged;
                this.clips.Add(clipVM);
            }

            this.UpdateMinMaxHorizontalOffset();

            this.Clips.CollectionChanged += OnCollectionChanged;
        }

        public WaveSoundViewModel OwnerWaveSound
        {
            get;
            set;
        }

        public WaveSoundTrackResource Model
        {
            get
            {
                WaveSoundTrackResource model = null;
                this.model.TryGetTarget(out model);
                return model;
            }
        }

        public ObservableCollection<ClipViewModel> Clips
        {
            get { return this.clips; }
        }

        public bool IsSelected
        {
            get { return this.isSelected; }
            set { this.SetProperty(this.PropertyChanged, ref this.isSelected, value); }
        }

        public double MinimumHorizontalOffset
        {
            get { return this.minimumHorizontalOffset; }
            set { this.SetProperty(this.PropertyChanged, ref this.minimumHorizontalOffset, value); }
        }

        public double MaximumHorizontalOffset
        {
            get { return this.maximumHorizontalOffset; }
            set { this.SetProperty(this.PropertyChanged, ref this.maximumHorizontalOffset, value); }
        }

        public void UpdateHorizontalOffset()
        {
            this.clips.ToList().ForEach(t => t.UpdateHorizontalOffset());
        }

        public void UpdateScale(double horizontalScale)
        {
            this.horizontalScale = horizontalScale;

            this.clips.ToList().ForEach(t => t.UpdateScale(horizontalScale));
        }

        public void UpdateSnap(double horizontalSnap, double verticalSnap)
        {
            this.horizontalSnap = horizontalSnap;
            this.verticalSnap = verticalSnap;

            this.clips.ToList().ForEach(t => t.UpdateSnap(horizontalSnap, verticalSnap));
        }

        public void UpdateViewportSize()
        {
            this.clips.ToList().ForEach(t => t.UpdateViewportSize());
        }

        public void OnSelectionChanged()
        {
            this.OwnerWaveSound.OnSelectionChanged();
        }

        private bool Contains(WaveSoundClipResource clipDM)
        {
            return this.Clips.Where(c => c.Model == clipDM).Count() > 0 ? true : false;
        }

        private void UpdateMinMaxHorizontalOffset()
        {
            if (this.clips.Count > 0)
            {
                this.MinimumHorizontalOffset = this.clips
                    .Select(c => c.X)
                    .Min();
                this.MaximumHorizontalOffset = this.clips
                    .Select(c => c.X + c.Width)
                    .Max();
            }
            else
            {
                this.MinimumHorizontalOffset = 0.0;
                this.MaximumHorizontalOffset = 0.0;
            }
        }

        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    if (e.NewItems != null)
                    {
                        foreach (var newItem in e.NewItems)
                        {
                            var clipVM = newItem as ClipViewModel;
                            clipVM.OwnerTrack = this;

                            this.Model.AddClip(clipVM.Model);
                        }
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    if (e.OldItems != null)
                    {
                        foreach (var oldItem in e.OldItems)
                        {
                            var clipVM = oldItem as ClipViewModel;
                            clipVM.OwnerTrack = null;

                            this.Model.RemoveClip(clipVM.Model);
                        }
                    }
                    break;
            }

            this.UpdateMinMaxHorizontalOffset();
        }

        private void OnClipPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            var clipVM = sender as ClipViewModel;
            if (e.PropertyName == PropertyUtility.NameOf(() => clipVM.X) ||
                e.PropertyName == PropertyUtility.NameOf(() => clipVM.Width))
            {
                this.UpdateMinMaxHorizontalOffset();
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ClipViewModel : INotifyPropertyChanged
    {
        private readonly WeakReference<WaveSoundClipResource> model = new WeakReference<WaveSoundClipResource>(null);
        public event PropertyChangedEventHandler PropertyChanged;
        private PropertyChangedObserver<WaveSoundClipResource> propertyChangedObserver = null;

        private double x;
        private double y;
        private double offsetX;
        private double width;
        private double height;
        private double volume;
        private double pitch;
        private double pan;
        private string filePath;

        private double actualX;
        private double actualY;
        private double actualWidth;
        private double actualOffsetX;
        private double horizontalScale;
        private double verticalSnap;
        private double? maximumWidth;
        private bool isSelected;
        private bool isEnabled;
        private double temporaryOffsetX;

        private WaveFileInfo waveFileInfo = new WaveFileInfo();
        private bool isValidFilePath = false;

        private bool useThumbnail = true;
        private Image thumbnail = null;
        private double thumbnailBegin = 0.0;
        private double thumbnailEnd = 0.0;
        private double thumbnailHorizontalScale = 0.0;
        private double thumbnailX;

        private DispatcherOperation dispatcherOperation = null;

        public ClipViewModel(WaveSoundClipResource model)
        {
            this.model.SetTarget(model);

            this.X = this.Model.Position;
            this.Y = this.Model.Row;
            this.Width = this.Model.Duration;
            this.Height = 100;
            this.filePath = this.Model.Path;
            this.Volume = this.Model.Volume;
            this.Pitch = this.Model.Pitch;
            this.Pan = this.Model.Pan;
            this.OffsetX = this.Model.StartOffset;
            this.TemporaryOffsetX = this.Model.StartOffset;

            this.propertyChangedObserver = new PropertyChangedObserver<WaveSoundClipResource>(model, true);
            this.propertyChangedObserver.AddHandler(p => p.Position, (t, e) => this.X = t.Position);
            this.propertyChangedObserver.AddHandler(p => p.Row, (t, e) => this.Y = t.Row);
            this.propertyChangedObserver.AddHandler(p => p.Duration, (t, e) => this.Width = t.Duration);
            this.propertyChangedObserver.AddHandler(p => p.StartOffset, (t, e) => this.OffsetX = t.StartOffset);
            this.propertyChangedObserver.AddHandler(p => p.StartOffset, (t, e) => this.TemporaryOffsetX = t.StartOffset);
            this.propertyChangedObserver.AddHandler(p => p.IsEnabled, (t, e) => this.IsEnabled = t.IsEnabled);

            this.propertyChangedObserver.AddHandler(p => p.Volume, (t, e) => this.Volume = t.Volume);
            this.propertyChangedObserver.AddHandler(p => p.Pitch, (t, e) => this.Pitch = t.Pitch);
            this.propertyChangedObserver.AddHandler(p => p.Pan, (t, e) => this.Pan = t.Pan);
            this.propertyChangedObserver.AddHandler(p => p.Path, (t, e) => this.FilePath = t.Path);

            this.AddPropertyChanged(t => t.X, t => this.Model.Position = (int)t.X);
            this.AddPropertyChanged(t => t.Y, t => this.Model.Row = (int)t.Y);
            this.AddPropertyChanged(t => t.Width, t => this.Model.Duration = (int)t.Width);
            this.AddPropertyChanged(t => t.OffsetX, t => this.Model.StartOffset = (int)t.OffsetX);
            this.AddPropertyChanged(t => t.TemporaryOffsetX, t => this.Model.SetStartOffsetWithPositionAndDuration((int)t.TemporaryOffsetX));
            this.AddPropertyChanged(t => t.IsEnabled, t => this.Model.IsEnabled = t.IsEnabled);

            this.AddPropertyChanged(t => t.Volume, t => this.Model.Volume = (float)t.Volume);
            this.AddPropertyChanged(t => t.Pitch, t => this.Model.Pitch = (float)t.Pitch);
            this.AddPropertyChanged(t => t.Pan, t => this.Model.Pan = (float)t.Pan);
            this.AddPropertyChanged(t => t.FilePath, t => this.Model.Path = t.FilePath);

            this.AddPropertyChanged(t => t.IsSelected, t => { this.OnSelectionChanged(); });

            this.UpdateFileInfo();
        }

        public TrackViewModel OwnerTrack
        {
            get;
            set;
        }

        public WaveSoundClipResource Model
        {
            get
            {
                WaveSoundClipResource model = null;
                this.model.TryGetTarget(out model);
                return model;
            }
        }

        public Image Thumbnail
        {
            get { return this.thumbnail; }
            set { this.SetProperty(this.PropertyChanged, ref this.thumbnail, value); }
        }

        public bool IsSelected
        {
            get { return this.isSelected; }
            set { this.SetProperty(this.PropertyChanged, ref this.isSelected, value); }
        }

        public bool IsEnabled
        {
            get { return this.isEnabled; }
            set { this.SetProperty(this.PropertyChanged, ref this.isEnabled, value); }
        }

        public double Volume
        {
            get { return this.volume; }
            set { this.SetProperty(this.PropertyChanged, ref this.volume, value); }
        }

        public double Pitch
        {
            get { return this.pitch; }
            set { this.SetProperty(this.PropertyChanged, ref this.pitch, value); }
        }

        public double Pan
        {
            get { return this.pan; }
            set { this.SetProperty(this.PropertyChanged, ref this.pan, value); }
        }

        public string FilePath
        {
            get { return this.filePath; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.filePath, value) == true)
                {
                    this.UpdateFileInfo();
                    this.UpdateThumbnail();

                    if (this.PropertyChanged != null)
                    {
                        this.PropertyChanged.Raise(() => FileName);
                    }
                }
            }
        }

        public string FileName
        {
            get { return System.IO.Path.GetFileName(this.filePath); }
        }

        public bool IsValidFilePath
        {
            get { return this.isValidFilePath; }
            private set { this.SetProperty(this.PropertyChanged, ref this.isValidFilePath, value); }
        }

        public double X
        {
            get { return this.x; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.x, value) == true)
                {
                    this.UpdateThumbnail();
                }
                this.SetProperty(this.PropertyChanged, ref this.actualX, value * this.horizontalScale, () => ActualX);
            }
        }

        public double Y
        {
            get { return this.y; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.y, value);
                this.SetProperty(this.PropertyChanged, ref this.actualY, value * this.verticalSnap, () => ActualY);
            }
        }

        public double OffsetX
        {
            get { return this.offsetX; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.offsetX, value);
                this.SetProperty(this.PropertyChanged, ref this.actualOffsetX, value * this.horizontalScale, () => ActualOffsetX);
            }
        }

        public double Width
        {
            get { return this.width; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.width, value);
                this.SetProperty(this.PropertyChanged, ref this.actualWidth, value * this.horizontalScale, () => ActualWidth);
            }
        }

        public double Height
        {
            get { return this.height; }
            set { this.SetProperty(this.PropertyChanged, ref this.height, value); }
        }

        public double ActualX
        {
            get { return this.actualX; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.actualX, value) == true)
                {
                    this.UpdateThumbnail();
                }
                this.SetProperty(this.PropertyChanged, ref this.x, value / this.horizontalScale, () => X);
            }
        }

        public double ActualY
        {
            get { return this.actualY; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.actualY, value);
                this.SetProperty(this.PropertyChanged, ref this.y, value / this.verticalSnap, () => Y);
            }
        }

        public double ActualWidth
        {
            get { return this.actualWidth; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.actualWidth, value) == true)
                {
                    this.UpdateThumbnail();
                }
                this.SetProperty(this.PropertyChanged, ref this.width, value / this.horizontalScale, () => Width);
            }
        }

        public double ActualOffsetX
        {
            get { return this.actualOffsetX; }
            set
            {
                this.SetProperty(this.PropertyChanged, ref this.actualOffsetX, value);
                this.SetProperty(this.PropertyChanged, ref this.offsetX, value / this.horizontalScale, () => OffsetX);
            }
        }

        public double ThumbnailX
        {
            get { return this.thumbnailX; }
            private set { this.SetProperty(this.PropertyChanged, ref this.thumbnailX, value); }
        }

        public double TemporaryOffsetX
        {
            get { return this.temporaryOffsetX; }
            set { this.SetProperty(this.PropertyChanged, ref this.temporaryOffsetX, value); }
        }

        public double? MaximumWidth
        {
            get { return this.maximumWidth; }
            set { this.SetProperty(this.PropertyChanged, ref this.maximumWidth, value, () => MaximumWidth); }
        }

        public void UpdateHorizontalOffset()
        {
            this.UpdateThumbnail();
        }

        public void UpdateScale(double horizontalScale)
        {
            if (this.horizontalScale == horizontalScale)
            {
                return;
            }

            this.horizontalScale = horizontalScale;

            // ActualXだけが更新されたことにします。
            // this.ActualXに代入すると誤差によって this.Xが変わるのを避ける為です。
            this.actualX = this.X * this.horizontalScale;
            this.PropertyChanged.Raise(() => ActualX);

            // ActualWidthだけが更新されたことにします。
            this.actualWidth = this.Width * this.horizontalScale;
            this.PropertyChanged.Raise(() => ActualWidth);

            // ActualOffsetだけが更新されたことにします。
            this.actualOffsetX = this.OffsetX * this.horizontalScale;
            this.PropertyChanged.Raise(() => ActualOffsetX);

            this.UpdateThumbnail();
        }

        public void UpdateSnap(double horizontalSnap, double verticalSnap)
        {
            this.verticalSnap = verticalSnap;

            this.actualY = this.Y * this.verticalSnap;
            this.PropertyChanged.Raise(() => ActualY);
        }

        public void UpdateViewportSize()
        {
            this.UpdateThumbnail();
        }

        protected bool UseThumbnail
        {
            get { return this.useThumbnail; }
            set { this.useThumbnail = value; }
        }

        private void OnSelectionChanged()
        {
            this.OwnerTrack.OnSelectionChanged();
        }

        private void UpdateFileInfo()
        {
            this.waveFileInfo = WaveImage.ReadWaveInfo(this.FilePath);
        }

        /// <summary>
        /// 波形のサムネイルを更新します。
        /// </summary>
        private void UpdateThumbnail()
        {
            if (this.UseThumbnail == false)
            {
                return;
            }

            // 既にサムネイル作成処理が実行されていたら中断します。
            if (this.dispatcherOperation != null &&
                this.dispatcherOperation.Status == DispatcherOperationStatus.Executing)
            {
                this.dispatcherOperation.Abort();
            }

            if (File.Exists(this.FilePath) == false)
            {
                this.IsValidFilePath = false;
                this.Thumbnail = null;
                return;
            }

            if (this.waveFileInfo.FileFormat == Foundation.Audio.WaveFileFormat.Unknwon)
            {
                this.IsValidFilePath = false;
                this.Thumbnail = null;
                return;
            }

            this.IsValidFilePath = true;

            this.dispatcherOperation = Dispatcher.CurrentDispatcher.BeginInvoke(DispatcherPriority.Background, new Action(() =>
            {
                var width = this.waveFileInfo.Duration.TotalMilliseconds * this.horizontalScale;

                var viewportX = this.OwnerTrack.OwnerWaveSound.ScaledHorizontalOffset;
                var viewportWidth = this.OwnerTrack.OwnerWaveSound.ViewportSize.Width;

                // 見える範囲を外れているのか？
                if (this.ActualX + this.ActualWidth < viewportX || this.ActualX > viewportX + viewportWidth)
                {
                    return;
                }

                // クリップのうち画面に収まっている部分
                var actualBegin = Math.Max(this.ActualX, viewportX);
                var actualEnd = Math.Min(this.ActualX + this.ActualWidth, viewportX + viewportWidth);

                // 仮想的な波形の先頭位置
                var headPosition = this.ActualX - this.ActualOffsetX;

                // 画面に収まっている波形の範囲
                var begin = MathUtility.Clamp(actualBegin - headPosition, 0, width);
                var end = MathUtility.Clamp(actualEnd - headPosition, 0, width);

                if (begin.Equals(this.thumbnailBegin) == true &&
                    end.Equals(this.thumbnailEnd) == true &&
                    this.horizontalScale.Equals(this.thumbnailHorizontalScale) == true)
                {
                    return;
                }

                // 描画範囲のビットマップを作成します。
                this.Thumbnail = WaveImage.CreateImage(this.FilePath, new Size(width, this.Height), begin, end);

                this.thumbnailBegin = begin;
                this.thumbnailEnd = end;
                this.thumbnailHorizontalScale = this.horizontalScale;
                this.ThumbnailX = begin - this.ActualOffsetX;

                this.dispatcherOperation = null;
            }));
        }
    }

    static class PropertyUtility
    {
        public static string NameOf<T>(Expression<Func<T>> value)
        {
            return ((MemberExpression)value.Body).Member.Name;
        }
    }
}
