﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Globalization;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Xml;


namespace Nintendo.AudioToolkit.Views
{
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.AudioToolkit.Extensions;

    /// <summary>
    /// WaveSoundControl.xaml の相互作用ロジック
    /// </summary>
    public partial class WaveSoundView : UserControl
    {
        private const double MinimumHorizontalScale = 0.001;
        private const double MaximumHorizontalScale = 1000.0;

        private static readonly DependencyProperty HorizontalScaleProperty =
            DependencyProperty.Register("HorizontalScale", typeof(double), typeof(WaveSoundView), new PropertyMetadata(1.0));

        public double HorizontalScale
        {
            get { return (double)GetValue(HorizontalScaleProperty); }
            set { SetValue(HorizontalScaleProperty, value); }
        }

        public WaveSoundView()
        {
            InitializeComponent();

            var horizontalScaleBinding = new Binding("HorizontalScale");
            horizontalScaleBinding.Mode = BindingMode.TwoWay;
            this.SetBinding(WaveSoundView.HorizontalScaleProperty, horizontalScaleBinding);
        }

        public bool GetCenterLogicalPoint(ref int trackNo, ref double position, ref int row)
        {
            var clientPoint = new Point(this.listBox.ActualWidth / 2.0, this.listBox.ActualHeight / 2.0);
            var screenPoint = this.listBox.PointToScreen(clientPoint);
            var screenX = (int)screenPoint.X;
            var screenY = (int)screenPoint.Y;
            return this.GetLogicalPoint(ref trackNo, ref position, ref row, screenX, screenY);
        }

        public bool GetLogicalPoint(ref int trackNo, ref double position, ref int row, int screenX, int screenY)
        {
            var clientPoint = this.PointFromScreen(new Point(screenX, screenY));
            var result = VisualTreeHelper.HitTest(this, clientPoint);
            var control = result.VisualHit.FindParent<TimelineControl>();

            if (control == null)
            {
                return false;
            }

            var point = control.PointFromScreen(new Point(screenX, screenY));
            trackNo = 0;
            position = point.X / control.HorizontalScale;
            row = (int)(point.Y / control.VerticalSnap);
            return true;
        }

        public void ScrollVerticalListBox(double value)
        {
            var border = VisualTreeHelper.GetChild(this.listBox, 0) as Border;
            var scrollViewer = VisualTreeHelper.GetChild(border, 0) as ScrollViewer;
            scrollViewer.ScrollToVerticalOffset(scrollViewer.VerticalOffset + value);
        }

        protected override void OnPreviewMouseWheel(MouseWheelEventArgs e)
        {
            if (Keyboard.IsKeyDown(Key.LeftShift) == true ||
                Keyboard.IsKeyDown(Key.RightShift) == true)
            {
                var value = this.slider_Position.Position - e.Delta / 2.0;
                this.slider_Position.Position = value;
            }
            else if (Keyboard.IsKeyDown(Key.LeftCtrl) == true ||
                     Keyboard.IsKeyDown(Key.RightCtrl) == true)
            {
                var direction = e.Delta > 0 ? 1 : -1;
                this.Zoom(direction, this.slider_Position.PointerX);
            }
            else if (Keyboard.IsKeyDown(Key.LeftAlt) == true ||
                     Keyboard.IsKeyDown(Key.RightAlt) == true)
            {
                //TODO: 縦ズーム
            }
            else
            {
                var scrollViewer = this.listBox.FindChild<ScrollViewer>();
                scrollViewer.ScrollToVerticalOffset(scrollViewer.VerticalOffset - e.Delta / 10.0);
            }
        }

        private int GetScaleIndex(double scale)
        {
            int index = 0;
            double baseScale = GetScale(index);

            if (baseScale <= scale)
            {
                while (true)
                {
                    double nextScale = this.GetScale(index + 1);
                    if (scale < (nextScale + baseScale) / 2.0)
                    {
                        return index;
                    }

                    baseScale = nextScale;
                    index++;
                }
            }
            else
            {
                while (true)
                {
                    double nextScale = this.GetScale(index - 1);
                    if (scale > (nextScale + baseScale) / 2.0)
                    {
                        return index;
                    }

                    baseScale = nextScale;
                    index--;
                }
            }
        }

        private double GetScale(int index)
        {
            if (index == 0)
            {
                return 1.0;
            }

            // スケールインデックスが奇数の場合は、absScale = 2 * 2 ^ ( ( n - 1 ) / 2 )
            // スケールインデックスが偶数の場合は、absScale = 3 * 2 ^ ( ( n - 1 ) / 2 )
            // スケールインデックスが負数の場合は、scale    = 1 / absScale
            var absIndex = Math.Abs(index);
            double value = (absIndex & 0x1) != 0 ? 2 : 3;
            double absScale = value * Math.Pow(2, ((absIndex - 1) / 2));

            return index > 0 ? absScale : 1 / absScale;
        }

        private double GetContentExtent()
        {
            var scrollContentPresenter = this.listBox.FindChild<ScrollContentPresenter>();
            return scrollContentPresenter.ExtentWidth;
        }


        private void Zoom(int direction, double cardinalPositionOffset)
        {
            var scale = this.HorizontalScale;
            int index = this.GetScaleIndex(scale) + (direction > 0 ? 1 : -1);
            var newScale = this.GetScale(index);

            // 値の範囲をここで補正しないとViewModelには補正前の値が渡ってしまいます。
            if (newScale < MinimumHorizontalScale)
            {
                newScale = MinimumHorizontalScale;
            }
            else if (newScale > MaximumHorizontalScale)
            {
                newScale = MaximumHorizontalScale;
            }

            var scrollContentPresenter = this.listBox.FindChild<ScrollContentPresenter>();
            var width = scrollContentPresenter.ExtentWidth;
            var position = this.slider_Position.Position / scale;
            var positionOffset = cardinalPositionOffset / scale;
            var ratio = positionOffset / (width / scale);
            var newPosition = position + positionOffset - ((width / newScale) * ratio);
            var min = this.globalTimeline.Minimum;
            var max = this.globalTimeline.Maximum;

            if (newPosition < min)
            {
                newPosition = min;
            }

            this.globalTimeline.CurrentValue = newPosition;
            this.globalTimeline.CurrentViewRange = width / newScale;
        }


        private void OnClick_ZoomIn(object sender, RoutedEventArgs e)
        {
            this.Zoom(1, this.GetContentExtent() / 2.0);
        }

        private void OnClick_ZoomOut(object sender, RoutedEventArgs e)
        {
            this.Zoom(-1, this.GetContentExtent() / 2.0);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class FrameworkElementExtensions
    {
        public static Rect GetAbsoltutePlacement(this FrameworkElement element, bool relativeToScreen = false)
        {
            var absolutePos = element.PointToScreen(new System.Windows.Point(0, 0));
            if (relativeToScreen)
            {
                return new Rect(absolutePos.X, absolutePos.Y, element.ActualWidth, element.ActualHeight);
            }
            var posMW = Application.Current.MainWindow.PointToScreen(new System.Windows.Point(0, 0));
            absolutePos = new System.Windows.Point(absolutePos.X - posMW.X, absolutePos.Y - posMW.Y);
            return new Rect(absolutePos.X, absolutePos.Y, element.ActualWidth, element.ActualHeight);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class CursorInfo
    {
        [DllImport("user32.dll")]
        private static extern void GetCursorPos(out POINT pt);

        [DllImport("user32.dll")]
        private static extern int ScreenToClient(IntPtr hwnd, ref POINT pt);

        private struct POINT
        {
            public UInt32 X;
            public UInt32 Y;
        }

        public static Point GetPosition()
        {
            POINT point;
            GetCursorPos(out point);
            return new Point(point.X, point.Y);
        }

        public static Point GetPosition(Visual v)
        {
            POINT point;
            GetCursorPos(out point);

            var source = HwndSource.FromVisual(v) as HwndSource;
            var hwnd = source.Handle;

            ScreenToClient(hwnd, ref point);
            return new Point(point.X, point.Y);
        }
    }

    public class BindableViewportScrollViewer : ScrollViewer
    {
        public Size ViewportSize
        {
            get { return (Size)GetValue(ViewportSizeProperty); }
            set { SetValue(ViewportSizeProperty, value); }
        }

        public static readonly DependencyProperty ViewportSizeProperty =
            DependencyProperty.Register("ViewportSize", typeof(Size), typeof(BindableViewportScrollViewer),
            new FrameworkPropertyMetadata(new Size(0.0, 0.0), null));

        public BindableViewportScrollViewer()
        {
            this.SizeChanged += delegate (object sender, SizeChangedEventArgs e)
            {
                this.ViewportSize = e.NewSize;
            };
        }
    }

    public class DoubleToPercentageConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return (double)value * 100.0;
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }
}
