﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Collections.Specialized;
using System.Reflection;

namespace Nintendo.AudioToolkit.DomainModels
{
    using Nintendo.Foundation.ComponentModel;
    using Nintendo.Foundation.Windows.Input;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.Operations;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.AudioToolkit.Extensions;

    public class WaveSoundResource : ObservableObject
    {
        private readonly WaveSoundResourceData data = null;
        private readonly ObservableCollection<WaveSoundTrackResource> tracks = new ObservableCollection<WaveSoundTrackResource>();
        private IOperationExecutor operationExecutor = null;

        public WaveSoundResource(WaveSoundResourceData data, IOperationExecutor executor)
        {
            this.data = data;
            this.BuildDataModel(this.data);

            this.OperationExecutor = executor;

            this.tracks.CollectionChanged += OnCollectionChangedTracks;
        }

        public WaveSoundResourceData DataModel
        {
            get { return this.data; }
        }

        public string Name
        {
            get { return this.data.Name; }
            set
            {
                if (this.Name != value)
                {
                    this.OperationExecutor.SetParameter<string>(this.data.Name, value,
                        (v) =>
                        {
                            this.data.Name = v;
                            this.NotifyPropertyChanged<string>(() => Name);
                        });
                }
            }
        }

        public IList<WaveSoundTrackResource> Tracks
        {
            get { return this.tracks; }
        }

        public void Add(WaveSoundTrackResource trackDM, WaveSoundClipResource clipDM)
        {
            this.OperationExecutor.AddItem(trackDM.Clips, clipDM);
        }

        public void Remove(IEnumerable<WaveSoundClipResource> clips)
        {
            foreach (var clip in clips.ToList())
            {
                this.OperationExecutor.RemoveItem(clip.Owner.Clips, clip);
            }
        }

        public void Add(WaveSoundTrackResource trackDM)
        {
            this.operationExecutor.AddItem(this.Tracks, trackDM);
        }

        public void Remove(IEnumerable<WaveSoundTrackResource> tracks)
        {
            foreach (var track in tracks.ToList())
            {
                this.OperationExecutor.RemoveItem(track.Owner.Tracks, track);
            }
        }

        private void BuildDataModel(WaveSoundResourceData waveSound)
        {
            foreach (var track in waveSound.TrackResources)
            {
                var trackDM = new WaveSoundTrackResource(track);
                trackDM.Owner = this;
                this.Tracks.Add(trackDM);
            }
        }

        private void OnCollectionChangedTracks(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var trackDM = newItem as WaveSoundTrackResource;
                        trackDM.Owner = this;
                        trackDM.OperationExecutor = this.OperationExecutor;

                        this.data.TrackResources.Add(trackDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var trackDM = oldItem as WaveSoundTrackResource;
                        trackDM.Owner = null;
                        trackDM.OperationExecutor = null;

                        this.data.TrackResources.Remove(trackDM.Data);
                    }
                    break;
            }
        }

        internal IOperationExecutor OperationExecutor
        {
            get { return this.operationExecutor; }
            set
            {
                this.operationExecutor = value;
                this.Tracks.ToList().ForEach(c => c.OperationExecutor = value);
            }
        }
    }
}
