﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"
#include <panels/MasterOutputPanel.h>

#if defined(NW_USE_NINTENDO_SDK)
#include <nn/nn_Windows.h>
#elif defined(NW_PLATFORM_CAFE)
#include <cafe/fs.h>
#include <cafe/env.h>
#elif defined(NW_PLATFORM_WIN32)
#include <windows.h>
#endif

#include <nw/snd/fnd/io/sndfnd_File.h>
#include <resources/GlobalResources.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {

const f32 MasterOutputPanel::MAX_SCALE_X = 4.f;
const f32 MasterOutputPanel::MIN_SCALE_X = 1.f / 32.f;

MasterOutputPanel* MasterOutputPanel::s_pThis = NULL;

MasterOutputPanel::MasterOutputPanel() :
m_SampleCount(0),
m_CurrentSampleIndex(0),
m_DrcCurrentSampleIndex(0),
m_CurrentOutputMode(nw::snd::OUTPUT_MODE_STEREO),
m_IsWaveGraphPaused(false),
m_RecordingWavFile(),
m_DrcRecordingWavFile(),
m_RecordingWavFileAdaptor(&m_RecordingWavFile),
m_DrcRecordingWavFileAdaptor(&m_DrcRecordingWavFile)
#if defined(NW_PLATFORM_CAFE)
, m_Allocator(NULL)
, m_FsClient(NULL)
, m_FsCmdBlock(NULL)
, m_FsCmdBlockForDrc(NULL)
, m_HfioVolume(NULL)
#endif
{
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetIsFocusable(true);

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    nw::internal::dw::UIElementList& containerItems = m_Container.GetContents();

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        containerItems.AddItem(&m_Channels[i]);

        if(i < MAX_CHANNEL_COUNT - 1)
        {
            containerItems.AddItem(&m_Separators[i]);
        }
    }

    for(u32 i = 0; i < MAX_CHANNEL_COUNT - 1; ++i)
    {
        m_Separators[i].SetMargin(nw::internal::dw::Thickness(2.f, 1.f));
        m_Separators[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_Separators[i].SetHeight(0.5f);
        m_Separators[i].SetOrientation(nw::internal::dw::HORIZONTAL);
        m_Separators[i].SetBorderColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.455f, 0.486f, 0.560f, 1.f)));
    }

    m_SampleCount = sizeof(m_Samples[0]) / sizeof(s16);

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        memset(m_Samples[i], 0, sizeof(m_Samples[i]));
        m_WaveGraphs[i].SetSamplesPerSec(SAMPLES_PER_SEC);
        m_WaveGraphs[i].SetWaveData(m_Samples[i], m_SampleCount);
        m_WaveGraphs[i].SetWaveCacheBuffer(m_GraphCacheBuffers[i], GRAPH_CACHE_BUFFER_SIZE);

        m_WaveGraphs[i].SetMargin(nw::internal::dw::Thickness(1.f, 1.f, 1.f, 0.f));
        m_WaveGraphs[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_WaveGraphs[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
        m_WaveGraphs[i].SetHeight(30);

        m_ChannelLabels[i].SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO);
        m_ChannelLabels[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_LEFT);
        m_ChannelLabels[i].SetVerticalAlignment(nw::internal::dw::VERTICAL_TOP);
        m_ChannelLabels[i].SetBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_DIM_GRAY()));
        m_ChannelLabels[i].SetTextScale(0.7f);

        m_Channels[i].SetMargin(nw::internal::dw::Thickness(0.f));
        m_Channels[i].SetPadding(nw::internal::dw::Thickness(0.f));
        m_Channels[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_Channels[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
        m_Channels[i].SetHeight(30);

        m_ChannelContentsArray[i].AddItem(&m_WaveGraphs[i]);
        m_ChannelContentsArray[i].AddItem(&m_ChannelLabels[i]);
        m_Channels[i].SetContents(m_ChannelContentsArray[i]);
    }

    m_ChannelLabels[MAIN_FL_INDEX].SetText("Main FL");
    m_ChannelLabels[MAIN_FR_INDEX].SetText("Main FR");
    m_ChannelLabels[MAIN_FC_INDEX].SetText("Main FC");
    m_ChannelLabels[MAIN_LFE_INDEX].SetText("Main LFE");
    m_ChannelLabels[MAIN_RL_INDEX].SetText("Main RL");
    m_ChannelLabels[MAIN_RR_INDEX].SetText("Main RR");
    m_ChannelLabels[DRC_FL_INDEX].SetText("DRC FL");
    m_ChannelLabels[DRC_FR_INDEX].SetText("DRC FR");
    m_ChannelLabels[DRC_RL_INDEX].SetText("DRC RL");
    m_ChannelLabels[DRC_RR_INDEX].SetText("DRC RR");

    // HACK : MainC, MainLfe は SDK未対応のため、グレー表示しています。
    m_ChannelLabels[MAIN_FC_INDEX].SetTextColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_GRAY()));
    m_ChannelLabels[MAIN_LFE_INDEX].SetTextColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_GRAY()));

    // HACK : 現状は２インスタンス以上の生成をサポートしていません。
    NW_ASSERT(s_pThis == NULL);

    if(s_pThis == NULL)
    {
        s_pThis = this;
        internal::driver::HardwareManager::GetInstance().AppendReadOnlyFinalMixCallback(s_pThis);
    }
}

MasterOutputPanel::~MasterOutputPanel()
{
    Finalize();

    if(s_pThis != NULL)
    {
        internal::driver::HardwareManager::GetInstance().EraseReadOnlyFinalMixCallback(s_pThis);
        s_pThis = NULL;
    }
}

void MasterOutputPanel::Initialize(IResourceProvider& resourceProvider)
{
#if defined(NW_PLATFORM_CAFE)
    m_Allocator = reinterpret_cast<ut::IAllocator*>(
        resourceProvider.GetResource(GlobalResources::AllocatorPath));

    m_FsClient = reinterpret_cast<FSClient*>(
        resourceProvider.GetResource(GlobalResources::FsClientPath));

    m_HfioVolume = reinterpret_cast<const char*>(
        resourceProvider.GetConstResource(GlobalResources::HfioVolumePath));

    m_FsCmdBlock = reinterpret_cast<FSCmdBlock*>(m_Allocator->Alloc(sizeof(FSCmdBlock)));
    m_FsCmdBlockForDrc = reinterpret_cast<FSCmdBlock*>(m_Allocator->Alloc(sizeof(FSCmdBlock)));

    NW_ASSERT_NOT_NULL(m_FsCmdBlock);
    NW_ASSERT_NOT_NULL(m_FsCmdBlockForDrc);

    FSInitCmdBlock(m_FsCmdBlock);
    FSInitCmdBlock(m_FsCmdBlockForDrc);
#endif

    m_WavOutRecorder.Initialize(m_RecordingSamples, RECORDING_BUFFER_SIZE);
    m_DrcWavOutRecorder.Initialize(m_DrcRecordingSamples, DRC_RECORDING_BUFFER_SIZE);
}

void MasterOutputPanel::Finalize()
{
    m_WavOutRecorder.Finalize();
    m_DrcWavOutRecorder.Finalize();

#if defined(NW_PLATFORM_CAFE)
    if(m_Allocator != NULL)
    {
        m_Allocator->Free(m_FsCmdBlock);
        m_Allocator->Free(m_FsCmdBlockForDrc);

        m_FsCmdBlock = NULL;
        m_FsCmdBlockForDrc = NULL;
        m_Allocator = NULL;
        m_HfioVolume = NULL;
    }
#endif
}

f32 MasterOutputPanel::GetScaleX() const
{
    return m_WaveGraphs[0].GetScaleX();
}

void MasterOutputPanel::SetScaleX(f32 value)
{
    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetScaleX(value);
    }
}

void MasterOutputPanel::Pause()
{
    m_IsWaveGraphPaused = true;
}

void MasterOutputPanel::Resume()
{
    if(!m_IsWaveGraphPaused)
    {
        return;
    }

    ClearSamples();
    m_IsWaveGraphPaused = false;
}

bool MasterOutputPanel::IsPaused() const
{
    return m_IsWaveGraphPaused;
}

// 保存先ディレクトリを取得します。
char* MasterOutputPanel::GetSaveDir(char* outBuf, size_t outBufSize) const
{
    const s32 MAX_PATH_LEN = 260;

#ifdef NW_PLATFORM_CAFE
    char buf[MAX_PATH_LEN + 1];

    int result = ENVGetEnvironmentVariable("CAFE_SAVE_DIR", buf, MAX_PATH_LEN);
    NW_ASSERT( result == 0 );
    NW_ASSERT( buf[0] != 0 );
    buf[1] = buf[0]; // c:\ を c\ にする

    ut::snprintf(outBuf, outBufSize, "%s/%s", m_HfioVolume, &buf[1]);
    return outBuf;
#else
    char buf[MAX_PATH_LEN + 1];
    char buf2[MAX_PATH_LEN + 1];

    if (GetModuleFileNameA(NULL, buf, sizeof(buf)) > 0 &&
        GetFileAttributesA((nw::ut::snprintf(buf2, sizeof(buf2), "%s/../Contents", buf), buf2)) == FILE_ATTRIBUTE_DIRECTORY )
    {
        // SoundPlayer と同じパスに Contents ディレクトリがあれば NintendoSDK 環境と判断します。

        // Visual Studio ではデバッグ用に設定できる環境変数は１つに制限されます。
        // そのため、本来は CAFE_SAVE_DIR を使いたいところですが、CAFE_CONTENT_DIR からの相対パスとします。
        if (GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, sizeof(buf)) > 0)
        {
            ut::snprintf(outBuf, outBufSize, "%s/../save", buf);
            return outBuf;
        }
        else
        {
            return NULL;
        }
    }
    else
    {
        // NW4F 環境の場合

        DWORD result = ExpandEnvironmentStringsA("%NW4F_ROOT%\\Tool\\SoundPlayer\\save", outBuf, outBufSize);
        NW_ASSERT(result > 0);
        return outBuf;
    }
#endif
}

void MasterOutputPanel::StartRecording()
{
    if(m_RecordingWavFile.IsOpened() ||
        m_DrcRecordingWavFile.IsOpened())
    {
        return;
    }

    // 保存先ディレクトリの取得
    const s32 MAX_PATH_LEN = 260;
    char saveDir[MAX_PATH_LEN + 1];
    char recPath[MAX_PATH_LEN + 1];

    if (GetSaveDir(saveDir, sizeof(saveDir)) == NULL)
    {
        NW_LOG("[MasterOutput] failed to find save directory.\n");
        return;
    }

    NW_LOG("save directory %s\n", saveDir);

    ut::snprintf(recPath, MAX_PATH_LEN, "%s/mainRec.wav", saveDir);

#ifdef NW_PLATFORM_CAFE
    if(internal::fnd::File::Open(
        &m_RecordingWavFile,
        m_FsClient,
        m_FsCmdBlock,
        recPath,
        internal::fnd::File::ACCESS_MODE_WRITE
        ).IsFailed())
#else
    if(internal::fnd::File::Open(
        &m_RecordingWavFile, recPath, internal::fnd::File::ACCESS_MODE_WRITE
        ).IsFailed())
#endif
    {
        NW_LOG("[MasterOutput] failed to open main wav file.\n");
        return;
    }

    ut::snprintf(recPath, MAX_PATH_LEN, "%s/drcRec.wav", saveDir);
#ifdef NW_PLATFORM_CAFE
    if(internal::fnd::File::Open(
        &m_DrcRecordingWavFile,
        m_FsClient,
        m_FsCmdBlockForDrc,
        recPath,
        internal::fnd::File::ACCESS_MODE_WRITE
        ).IsFailed())
#else
    if(internal::fnd::File::Open(
        &m_DrcRecordingWavFile,
        recPath,
        internal::fnd::File::ACCESS_MODE_WRITE
        ).IsFailed())
#endif
    {
        NW_LOG("[MasterOutput] failed to open drc wav file.\n");
        return;
    }

    if(!m_WavOutRecorder.Start(m_RecordingWavFileAdaptor))
    {
        NW_LOG("[MasterOutput] failed to start main recording.\n");
        m_RecordingWavFile.Close();
        return;
    }

#if defined(NW_PLATFORM_CAFE)
    if(!m_DrcWavOutRecorder.Start(m_DrcRecordingWavFileAdaptor))
    {
        NW_LOG("[MasterOutput] failed to start drc recording.\n");
        m_DrcRecordingWavFile.Close();
        return;
    }
#endif
}

void MasterOutputPanel::StopRecording()
{
    m_WavOutRecorder.Stop(false);
    m_DrcWavOutRecorder.Stop(false);
}

bool MasterOutputPanel::IsRecording()
{
    return m_RecordingWavFile.IsOpened() ||
        m_DrcRecordingWavFile.IsOpened() ||
        m_WavOutRecorder.GetState() == WavOutRecorder::STATE_RECORDING ||
        m_DrcWavOutRecorder.GetState() == DrcWavOutRecorder::STATE_RECORDING;
}

void MasterOutputPanel::SetRecordingLamp(Lamp* pLamp)
{
    if(m_pRecordingLamp == pLamp)
    {
        return;
    }

    if(m_pRecordingLamp != NULL)
    {
        m_pRecordingLamp->SetLabelText("");
        m_pRecordingLamp->SetIsLamped(false);
    }

    m_pRecordingLamp = pLamp;

    if(m_pRecordingLamp != NULL)
    {
        m_pRecordingLamp->SetLabelText("REC");
        m_pRecordingLamp->SetIsLamped(IsRecording());
    }
}

bool MasterOutputPanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    // 波形表示スケール * 2
    if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
    {
        f32 scaleX = GetScaleX() * 2;

        if(scaleX > MAX_SCALE_X)
        {
            scaleX = MIN_SCALE_X;
        }

        SetScaleX(scaleX);
        return true;
    }

    // 波形表示スケール / 2
    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT))
    {
        f32 scaleX = GetScaleX() / 2;

        if(scaleX < MIN_SCALE_X)
        {
            scaleX = MAX_SCALE_X;
        }

        SetScaleX(scaleX);
        return true;
    }

    // 波形表示の一時停止、再開
    if(pad.IsHold(nw::dev::Pad::MASK_R) &&
        pad.IsTrig(nw::dev::Pad::MASK_Y))
    {
        if(IsPaused())
        {
            Resume();
        }
        else
        {
            Pause();
        }

        return true;
    }

    // 録音の開始
    if(pad.IsTrig(nw::dev::Pad::MASK_X))
    {
        StartRecording();
        return true;
    }

    // 録音の停止
    if(pad.IsTrig(nw::dev::Pad::MASK_Y))
    {
        StopRecording();
        return true;
    }

    return false;
}

void MasterOutputPanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    nw::snd::OutputMode outputMode = nw::snd::SoundSystem::GetOutputMode();

    if(m_CurrentOutputMode != outputMode)
    {
        m_CurrentOutputMode = outputMode;

        for(u32 i = MAIN_CHANNEL_START; i < MAIN_CHANNEL_START + MAX_MAIN_CHANNEL_COUNT; ++i)
        {
            m_WaveGraphs[i].SetSamplesPerSec(SAMPLES_PER_SEC);
        }

        // Main FC, Main LFE, Main RL, MainRR の波形バッファをクリアします。
        memset(m_Samples[MAIN_FC_INDEX], 0, sizeof(m_Samples[MAIN_FC_INDEX]));
        memset(m_Samples[MAIN_LFE_INDEX], 0, sizeof(m_Samples[MAIN_LFE_INDEX]));
        memset(m_Samples[MAIN_RL_INDEX], 0, sizeof(m_Samples[MAIN_RL_INDEX]));
        memset(m_Samples[MAIN_RR_INDEX], 0, sizeof(m_Samples[MAIN_RR_INDEX]));

        // 録音を停止します。
        if(IsRecording())
        {
            StopRecording();
        }
    }

    if(m_RecordingWavFile.IsOpened() &&
        m_WavOutRecorder.GetState() != WavOutRecorder::STATE_RECORDING &&
        m_WavOutRecorder.GetState() != WavOutRecorder::STATE_STOPPING)
    {
        m_RecordingWavFile.Close();
    }

    if(m_DrcRecordingWavFile.IsOpened() &&
        m_DrcWavOutRecorder.GetState() != DrcWavOutRecorder::STATE_RECORDING &&
        m_DrcWavOutRecorder.GetState() != DrcWavOutRecorder::STATE_STOPPING)
    {
        m_DrcRecordingWavFile.Close();
    }

    if(m_pRecordingLamp != NULL)
    {
        m_pRecordingLamp->SetIsLamped(IsRecording());
    }
}

void MasterOutputPanel::OnFinalMix(OutputDevice device, const FinalMixData* data)
{
    NW_NULL_ASSERT(s_pThis);
    NW_NULL_ASSERT(data);
    if (device == OUTPUT_DEVICE_MAIN)
    {
        s_pThis->UpdateWaveGraph(data);
    }
    else if (device == OUTPUT_DEVICE_DRC0)
    {
        s_pThis->UpdateDrcWaveGraph(data);
    }
}

s16* MasterOutputPanel::GetSamples(u32 channel)
{
    NW_ASSERT(channel < MAX_CHANNEL_COUNT);
    return m_Samples[channel];
}

void MasterOutputPanel::UpdateWaveGraph(const FinalMixData* data)
{
    if(m_IsWaveGraphPaused)
    {
        return;
    }

    bool isSurroundEnabled =
        nw::snd::SoundSystem::GetOutputMode() == nw::snd::OUTPUT_MODE_SURROUND;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // PC版では、サラウンドモードに対応していません。
    isSurroundEnabled = false;
#endif

    const u32 deviceIndex = 0;
    u32 restCopySamples = data[deviceIndex].sampleCount;
    u32 sampleOffset = 0;

    while(restCopySamples > 0)
    {
        const u32 copySamples = nw::ut::Min(restCopySamples, m_SampleCount - m_CurrentSampleIndex);

        for(u32 i = 0; i < copySamples ; ++i)
        {
            u32 srcOffset = sampleOffset + i;
            u32 destOffset = m_CurrentSampleIndex + i;

            if (data[deviceIndex].frontLeft != NULL)
            {
                GetSamples(MAIN_FL_INDEX)[destOffset] = data[deviceIndex].frontLeft[srcOffset];
            }
            if (data[deviceIndex].frontRight != NULL)
            {
                GetSamples(MAIN_FR_INDEX)[destOffset] = data[deviceIndex].frontRight[srcOffset];
            }

            if(isSurroundEnabled)
            {
                if (data[deviceIndex].frontCenter != NULL)
                {
                    GetSamples(MAIN_FC_INDEX)[destOffset]  = data[deviceIndex].frontCenter[srcOffset];
                }
                if (data[deviceIndex].lfe != NULL)
                {
                    GetSamples(MAIN_LFE_INDEX)[destOffset] = data[deviceIndex].lfe[srcOffset];
                }
                if (data[deviceIndex].rearLeft != NULL)
                {
                    GetSamples(MAIN_RL_INDEX)[destOffset]  = data[deviceIndex].rearLeft[srcOffset];
                }
                if (data[deviceIndex].rearRight != NULL)
                {
                    GetSamples(MAIN_RR_INDEX)[destOffset]  = data[deviceIndex].rearRight[srcOffset];
                }
            }
        }

        restCopySamples -= copySamples;
        sampleOffset += copySamples;
        m_CurrentSampleIndex += copySamples;

        if ( m_CurrentSampleIndex >= m_SampleCount )
        {
            m_CurrentSampleIndex = 0;
        }
    }

    for(u32 i = MAIN_CHANNEL_START; i < MAIN_CHANNEL_START + MAX_MAIN_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetWaveDataStartIndex(m_CurrentSampleIndex);
    }
}

void MasterOutputPanel::UpdateDrcWaveGraph(const FinalMixData* data)
{
    if(m_IsWaveGraphPaused)
    {
        return;
    }

    const u32 deviceIndex = 0;
    u32 restCopySamples = data[deviceIndex].sampleCount;
    u32 sampleOffset = 0;

    while(restCopySamples > 0)
    {
        const u32 copySamples = nw::ut::Min(restCopySamples, m_SampleCount - m_DrcCurrentSampleIndex);

        for(u32 i = 0; i < copySamples ; ++i)
        {
            u32 srcOffset = sampleOffset + i;
            u32 destOffset = m_DrcCurrentSampleIndex + i;

            if (data[deviceIndex].frontLeft != NULL)
            {
                GetSamples(DRC_FL_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontLeft[srcOffset]);
            }
            if (data[deviceIndex].frontRight != NULL)
            {
                GetSamples(DRC_FR_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontRight[srcOffset]);
            }
            if (data[deviceIndex].rearLeft != NULL)
            {
                GetSamples(DRC_RL_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].rearLeft[srcOffset]);
            }
            if (data[deviceIndex].rearRight != NULL)
            {
                GetSamples(DRC_RR_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].rearRight[srcOffset]);
            }
        }

        restCopySamples -= copySamples;
        sampleOffset += copySamples;
        m_DrcCurrentSampleIndex += copySamples;

        if ( m_DrcCurrentSampleIndex >= m_SampleCount )
        {
            m_DrcCurrentSampleIndex = 0;
        }
    }

    for(u32 i = DRC_CHANNEL_START; i < DRC_CHANNEL_START + MAX_DRC_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetWaveDataStartIndex(m_DrcCurrentSampleIndex);
    }
}


void MasterOutputPanel::ClearSamples()
{
    // 必ず一時停止中にコールします。
    NW_ASSERT(m_IsWaveGraphPaused);

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        memset(m_Samples[i], 0, sizeof(m_Samples[i]));
        m_WaveGraphs[i].SetWaveDataStartIndex(0);
    }
}

} // snd
} // nw
