﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewSoundArchive.h>

#if defined(NW_PLATFORM_CAFE)
#include <nw/sndaac.h>
#endif

// TODO: [fs]暫定対処なので、devのFS対応が済んだら正式対処をする。
#include <string>

namespace {
    const nw::snd::SoundArchive::ItemId SOUND_ID_MIN = nw::snd::internal::ItemType_Sound << 24;
    const s32 STREAM_READ_CACHE_SIZE = 16 * 1024 * 5;
}

namespace nw {
namespace snd {

#if defined(NW_ENABLE_SNDCTRL)
namespace{

void LoadDataForPlay(ctrl::SoundController& target, SoundArchive::ItemId soundID, void* userParam)
{
    PreviewSoundArchive* previewSoundArchive = reinterpret_cast<PreviewSoundArchive*>(userParam);
    NW_ASSERT_NOT_NULL(previewSoundArchive);

    if(!previewSoundArchive->IsOpened())
    {
        return;
    }

    // ストリームサウンドはロード不要
    if(previewSoundArchive->GetSoundArchive().GetSoundType(soundID) == SoundArchive::SOUND_TYPE_STRM)
    {
        return;
    }

#if defined(NW_ENABLE_SNDEDIT)
    // インゲーム編集対象の場合はロード不要
    if(previewSoundArchive->IsEditedItem(soundID))
    {
        return;
    }
#endif

    previewSoundArchive->LoadData(soundID);
}

}
#endif

PreviewSoundArchive::PreviewSoundArchive() :
#if defined(NW_PLATFORM_CAFE)
m_pFsClient(NULL),
m_pMemoryForFsSoundArchive(NULL),
#endif
#if defined(NW_ENABLE_SNDCTRL)
m_SoundObjectController(NULL),
#endif
m_IsOpened(false),
m_IsExistSeqSound(false),
#if defined(NW_ENABLE_SNDEDIT)
m_pSoundArchiveEditor(NULL),
#endif
m_SoundArchiveOpenedCallback(NULL)
{
}

void PreviewSoundArchive::Initialize(
#if defined(NW_PLATFORM_CAFE)
    FSClient* client,
#endif
    const char* pSoundArchiveFilePath,
    u32 heapSize
#if defined(NW_ENABLE_SNDCTRL)
    , ctrl::SoundObjectController* soundObjectController
#endif
    )
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT(m_pFsClient == NULL);
    NW_ASSERT_NOT_NULL(client);
#endif
    NW_NULL_ASSERT(pSoundArchiveFilePath);

#if defined(NW_PLATFORM_CAFE)
    m_pFsClient = client;
#endif

    // サウンドヒープの構築
    {
#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

#if defined( NW_PLATFORM_CAFE )
        const u32 WAVE_DATA_BOUNDARY = 512 * 1024 * 1024;

        u32 memorySize = heapSize * 2;
        NW_ASSERT(memorySize > WAVE_DATA_BOUNDARY);
        m_pMemoryForSoundHeap = MEMAllocFromDefaultHeap(memorySize);

        u32 memoryOffset = WAVE_DATA_BOUNDARY - nw::ut::GetIntPtr(m_pMemoryForSoundHeap) % WAVE_DATA_BOUNDARY;
        if (memoryOffset < heapSize)
        {
            m_pMemoryForSoundHeap = nw::ut::AddOffsetToPtr(m_pMemoryForSoundHeap, memoryOffset);
        }
#else // defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
        m_pMemoryForSoundHeap = MEMAllocFromDefaultHeap(heapSize);
#endif

        bool result = m_Heap.Create(m_pMemoryForSoundHeap, heapSize);
        NW_ASSERT(result);
    }

#if defined(NW_ENABLE_SNDCTRL)
    m_SoundObjectController = soundObjectController;

    if(m_SoundObjectController != NULL)
    {
        if(m_SoundObjectController->Initialize(&m_ArchivePlayer).IsFailed())
        {
            NW_FATAL_ERROR("failed to initialize SoundObjectController.\n");
            return;
        }

        for(u32 index = 0; index < ctrl::SoundObjectController::SOUND_CONTROLLER_COUNT; ++index)
        {
            reinterpret_cast<ctrl::internal::SoundControllerImpl*>(
                m_SoundObjectController->GetSoundController(index))->
                SetPreplayAction(LoadDataForPlay, this);
        }
    }

    for(u32 i = 0; i < PREVIEW_SOUND_COUNT; ++i)
    {
        m_PreviewSounds[i].Initialize(this, m_SoundObjectController->GetSoundController(i));
    }
#endif
}

void PreviewSoundArchive::Finalize()
{
    Close();

#if defined(NW_ENABLE_SNDCTRL)
    if(m_SoundObjectController != NULL)
    {
        for(u32 index = 0; index < ctrl::SoundObjectController::SOUND_CONTROLLER_COUNT; ++index)
        {
            reinterpret_cast<ctrl::internal::SoundControllerImpl*>(
                m_SoundObjectController->GetSoundController(index))->
                SetPreplayAction(NULL, NULL);
        }

        m_SoundObjectController->Finalize();

        m_SoundObjectController = NULL;
    }
#endif

    for(u32 i = 0; i < PREVIEW_SOUND_COUNT; ++i)
    {
        m_PreviewSounds[i].Finalize();
    }

    if(m_pMemoryForSoundHeap != NULL)
    {
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

        m_Heap.Destroy();
        MEMFreeToDefaultHeap(m_pMemoryForSoundHeap);
        m_pMemoryForSoundHeap = NULL;
    }

#if defined(NW_PLATFORM_CAFE)
    m_pFsClient = NULL;
#endif
}

void PreviewSoundArchive::Update()
{
    if (!m_IsOpened)
    {
        return;
    }

    for(u32 i = 0; i < PREVIEW_SOUND_COUNT; ++i)
    {
        m_PreviewSounds[i].Update();
    }

    for(u32 i = 0; i < PREVIEW_BANK_COUNT; ++i)
    {
        m_PreviewBanks[i].Update();
    }

#if defined(NW_ENABLE_SNDCTRL)
    if(m_SoundObjectController != NULL)
    {
        m_SoundObjectController->Update();
    }
#endif

    m_ArchivePlayer.Update();
}

s32 PreviewSoundArchive::GetPreviewSoundCount()
{
    return PREVIEW_SOUND_COUNT;
}

PreviewSound& PreviewSoundArchive::GetPreviewSound(s32 index)
{
    if(index >= GetPreviewSoundCount())
    {
        NW_FATAL_ERROR("invalid PreviewSound index.");
    }

    return m_PreviewSounds[index];
}

s32 PreviewSoundArchive::GetPreviewBankCount()
{
    return PREVIEW_BANK_COUNT;
}

PreviewBank& PreviewSoundArchive::GetPreviewBank(s32 index)
{
    if(index >= GetPreviewBankCount())
    {
        NW_FATAL_ERROR("invalid PreviewBank index.");
    }

    return m_PreviewBanks[index];
}

#ifndef NW_ENABLE_SNDCTRL
SequenceVariableContainer& PreviewSoundArchive::GetGlobalVariables()
{
    return m_GlobalVariables;
}
#endif

void PreviewSoundArchive::Open(const char* pSoundArchiveFilePath)
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

    NW_NULL_ASSERT(pSoundArchiveFilePath);

#if defined(NW_USE_NINTENDO_SDK)
    if (! m_Archive.Open(pSoundArchiveFilePath))
    {
        NW_WARNING(0, "cannot open bcsar(%s)\n", pSoundArchiveFilePath);
        return;
    }

#elif defined(NW_PLATFORM_CAFE)
    // サウンドアーカイブの初期化
    {
        size_t size = m_Archive.GetRequiredMemSize();
        m_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if(! m_Archive.Open(m_pFsClient, pSoundArchiveFilePath, m_pMemoryForFsSoundArchive, size))
        {
            //NW_ASSERTMSG(0, "cannot open bcsar(%s)\n", pSoundArchiveFilePath);
            NW_WARNING(0, "cannot open bcsar(%s)\n", pSoundArchiveFilePath);
            return;
        }
    }

#elif defined(NW_PLATFORM_WIN32)
    // TODO: [fs]暫定対処なので、devのFS対応が済んだら正式対処をする。
    std::string volume = "/vol/content/";
    std::string path = pSoundArchiveFilePath;
    path = path.substr( volume.size() );

    if (! m_Archive.Open(path.c_str()))
    {
        NW_WARNING(0, "cannot open bcsar(%s)\n", path.c_str());
        return;
    }
    //if (! m_Archive.Open(pSoundArchiveFilePath))
    //{
    //    NW_WARNING(0, "cannot open bcsar(%s)\n", pSoundArchiveFilePath);
    //    return;
    //}
#else
#error "not implemented"
#endif

    // INFO ブロックのロード
    {
        size_t infoBlockSize = m_Archive.GetHeaderSize();
        m_pMemoryForInfoBlock = MEMAllocFromDefaultHeap(infoBlockSize);
        if(! m_Archive.LoadHeader(m_pMemoryForInfoBlock, infoBlockSize))
        {
            NW_ASSERTMSG(0, "cannot load infoBlock(%s)", pSoundArchiveFilePath);
        }
    }

#ifdef NW_PLATFORM_CAFE
    // AAC初期化
    {
        size_t aacDecBufferSize = nw::snd::aac::GetAacDecBufferSize(&m_Archive);
        m_pMemoryForAacDec = MEMAllocFromDefaultHeapEx( aacDecBufferSize, 64 );
        nw::snd::aac::Initialize( m_pMemoryForAacDec, aacDecBufferSize );
    }
#endif

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = m_DataManager.GetRequiredMemSize(&m_Archive);
        m_pMemoryForSoundDataManager = MEMAllocFromDefaultHeap(setupSize);
        m_DataManager.Initialize(&m_Archive, m_pMemoryForSoundDataManager, setupSize);
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = m_ArchivePlayer.GetRequiredMemSize(&m_Archive);
        m_pMemoryForSoundArchivePlayer = MEMAllocFromDefaultHeapEx(setupSize, 32);

        if (m_pMemoryForSoundArchivePlayer == NULL)
        {
            nw::ut::Printf("[SoundPlayer] Allocation failed. (RequiredSize = %u)\n", setupSize);
            nw::ut::Printf("[SoundPlayer] Please adjust player heap size or sound limit.\n");

            m_DataManager.Finalize();
            m_Heap.Clear();

            m_Archive.Close();
        #if defined(NW_PLATFORM_CAFE)
            MEMFreeToDefaultHeap(m_pMemoryForFsSoundArchive);
        #endif

            // Openの処理で確保しているメモリを解放する
            MEMFreeToDefaultHeap(m_pMemoryForSoundDataManager);
            MEMFreeToDefaultHeap(m_pMemoryForInfoBlock);

            return;
        }

        size_t setupStrmBufferSize =
            m_ArchivePlayer.GetRequiredStreamBufferSize(&m_Archive);
        m_pMemoryForStreamBuffer = MEMAllocFromDefaultHeapEx(setupStrmBufferSize, 32);

        u32 setupStrmCacheSize =
            m_ArchivePlayer.GetRequiredStreamCacheSize( &m_Archive, STREAM_READ_CACHE_SIZE );
        m_pMemoryForStreamCacheBuffer = MEMAllocFromDefaultHeapEx(setupStrmCacheSize, 64);

        nw::snd::SoundArchivePlayer::InitializeParam param;
        param.soundArchive = &m_Archive;
        param.soundDataManager = &m_DataManager;
        param.setupBuffer = m_pMemoryForSoundArchivePlayer;
        param.setupBufferSize = setupSize;
        param.streamBuffer = m_pMemoryForStreamBuffer;
        param.streamBufferSize = setupStrmBufferSize;
        param.streamCacheBuffer = m_pMemoryForStreamCacheBuffer;
        param.streamCacheSize = setupStrmCacheSize;
        bool result = m_ArchivePlayer.Initialize(param);
        NW_ASSERT( result );
    }

    // ラベルデータのロード
    {
        size_t setupSize = m_Archive.GetLabelStringDataSize();
        m_pMemoryForLabelData = MEMAllocFromDefaultHeap(setupSize);

        m_Archive.LoadLabelStringData(m_pMemoryForLabelData, setupSize);
    }

    m_IsOpened = true;

    // プレビューサウンドの初期化
    // IsOpenedのフラグが立った後で更新しないと、アーカイブが閉じているときの処理に流れてしまうため
    // 正しいIDを取得することができない
#if defined(NW_ENABLE_SNDCTRL)
    NW_ASSERT(PREVIEW_SOUND_COUNT == ctrl::SoundObjectController::SOUND_CONTROLLER_COUNT);
#endif

#if !defined(NW_ENABLE_SNDCTRL)
    for(u32 i = 0; i < PREVIEW_SOUND_COUNT; ++i)
    {
        m_PreviewSounds[i].Initialize(this);
    }
#endif

    // プレビューバンクの初期化
    for(u32 i = 0; i < PREVIEW_BANK_COUNT; ++i)
    {
        m_PreviewBanks[i].Initialize(*this);
    }

    // シーケンスサウンドの存在確認
    m_IsExistSeqSound = false;
    nw::snd::SoundArchive::ItemId id = nw::snd::SoundArchive::INVALID_ID;
    for ( u32 i = 0; i < m_Archive.GetSoundCount(); i++ )
    {
        id = SOUND_ID_MIN + i;
        if ( m_Archive.GetSoundType( id ) == nw::snd::SoundArchive::SOUND_TYPE_SEQ )
        {
            m_IsExistSeqSound = true;
            break;
        }
    }

    if (m_SoundArchiveOpenedCallback != NULL)
    {
        m_SoundArchiveOpenedCallback();
    }
}

void PreviewSoundArchive::Close()
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif
    if (!m_IsOpened)
    {
        return;
    }

    // プレビューバンクを閉じる
    for(u32 i = 0; i < PREVIEW_BANK_COUNT; ++i)
    {
        m_PreviewBanks[i].Finalize();
    }

    // プレビューサウンドを閉じる
    for(u32 i = 0; i < PREVIEW_SOUND_COUNT; ++i)
    {
        m_PreviewSounds[i].Stop();

#if !defined(NW_ENABLE_SNDCTRL)
        m_PreviewSounds[i].Finalize();
#endif
    }

#ifdef NW_PLATFORM_CAFE
    nw::snd::aac::Finalize();
    MEMFreeToDefaultHeap(m_pMemoryForAacDec);
    m_pMemoryForAacDec = NULL;
#endif

    m_ArchivePlayer.Finalize();
    m_DataManager.Finalize();
    m_Heap.Clear();

    m_Archive.Close();

    // Openの処理で確保しているメモリを解放する
    MEMFreeToDefaultHeap(m_pMemoryForLabelData);
    MEMFreeToDefaultHeap(m_pMemoryForStreamCacheBuffer);
    MEMFreeToDefaultHeap(m_pMemoryForStreamBuffer);
    MEMFreeToDefaultHeap(m_pMemoryForSoundArchivePlayer);
    MEMFreeToDefaultHeap(m_pMemoryForSoundDataManager);
    MEMFreeToDefaultHeap(m_pMemoryForInfoBlock);

    m_IsOpened = false;
    m_IsExistSeqSound = false;
}

bool PreviewSoundArchive::StartSound(PreviewSound& sound, nw::snd::SoundArchive::ItemId id)
{
    if (!m_IsOpened)
    {
        return false;
    }

    sound.GetSoundHandle().Stop(0);

#ifdef NW_DEBUG
    NW_LOG("[Info] StartSound(0x%x).\n", id);
#endif

    if(m_Archive.GetSoundType(id) != nw::snd::SoundArchive::SOUND_TYPE_STRM)
    {
        if(!LoadData(id))
        {
            return false;
        }
    }

    nw::snd::SoundStartable::StartResult result;

#ifdef NW_PLATFORM_CAFE
    if(sound.GetStartOffset() > 0)
#else
    if(m_Archive.GetSoundType(id) != nw::snd::SoundArchive::SOUND_TYPE_STRM &&
        sound.GetStartOffset() > 0)
#endif
    {
        nw::snd::SoundStartable::StartInfo startInfo;
        startInfo.enableFlag = nw::snd::SoundStartable::StartInfo::ENABLE_START_OFFSET;
        startInfo.startOffsetType = nw::snd::SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC;
        startInfo.startOffset = sound.GetStartOffset();

        result = m_ArchivePlayer.StartSound(&sound.GetSoundHandle(), id, &startInfo);
    }
    else
    {
        result = m_ArchivePlayer.StartSound(&sound.GetSoundHandle(), id);
    }

#ifdef NW_DEBUG
    NW_LOG("[Info] StartSound(0x%x) result:%d\n", id, result.GetCode());
#endif

    return result.IsSuccess();
}

const char* PreviewSoundArchive::GetItemLabel(nw::snd::SoundArchive::ItemId itemID) const
{
    if (m_IsOpened)
    {
        return m_Archive.GetItemLabel(itemID);
    }
    return "";
}

s32 PreviewSoundArchive::GetSoundCount() const
{
    if (m_IsOpened)
    {
        return m_Archive.GetSoundCount();
    }
    return 0;
}

nw::snd::SoundArchive::ItemId PreviewSoundArchive::GetSoundID(u32 index) const
{
    if (m_IsOpened)
    {
        return m_Archive.GetSoundIdFromIndex(index);
    }
    return 0;
}

nw::snd::SoundArchive::ItemId PreviewSoundArchive::GetSoundID(const char* label) const
{
    if (m_IsOpened)
    {
        nw::snd::SoundArchive::ItemId id = m_Archive.GetItemId(label);

        if (id != SoundArchive::INVALID_ID)
        {
            return id;
        }
    }
    return 0;
}
s32 PreviewSoundArchive::GetBankCount() const
{
    if (m_IsOpened)
    {
        return m_Archive.GetBankCount();
    }
    return -1;
}

nw::snd::SoundArchive::ItemId PreviewSoundArchive::GetBankID(u32 index) const
{
    if (m_IsOpened)
    {
        return m_Archive.GetBankIdFromIndex(index);
    }
    return 0;
}

nw::snd::SoundArchive::ItemId PreviewSoundArchive::GetBankID(const char* label) const
{
    if (m_IsOpened)
    {
        nw::snd::SoundArchive::ItemId id = m_Archive.GetItemId(label);

        if (id != SoundArchive::INVALID_ID)
        {
            return id;
        }
    }
    return 0;
}

#if defined(NW_ENABLE_SNDEDIT)
bool PreviewSoundArchive::IsEditedItem(nw::snd::SoundArchive::ItemId id) const
{
    if (m_pSoundArchiveEditor == NULL)
    {
        return false;
    }

    const char* itemLabel = m_Archive.GetItemLabel(id);
    if (itemLabel == NULL)
    {
        return false;
    }

    if (m_pSoundArchiveEditor->GetItemCacheState(itemLabel) != nw::snd::edit::CACHE_STATE_CACHED)
    {
        return false;
    }

    return true;
}
#endif

bool PreviewSoundArchive::LoadData(nw::snd::SoundArchive::ItemId id)
{
    // すでにロードされていれば、何もせず終了
    if(m_DataManager.IsDataLoaded(id))
    {
        return true;
    }

    // 分割ロードサイズは適当(半分、デバッグ目的)
    if(m_DataManager.LoadData(id, &m_Heap, nw::snd::SoundDataManager::LOAD_ALL, 1024*8))
    {
        return true;
    }

    // ヒープが足りなかっただけかもしれないので、リセットして再ロード
    m_Heap.LoadState(0);
    NW_WARNING(false, "reset heap");

    if(m_DataManager.LoadData(id, &m_Heap, nw::snd::SoundDataManager::LOAD_ALL, 1024*8))
    {
        return true;
    }

    bool isLoadSuccessGroupAttachedItem = false;
    // アイテム ID → ファイル ID
    switch ( nw::snd::internal::Util::GetItemType( id ) )
    {
    case nw::snd::internal::ItemType_Sound:
        {
            nw::snd::SoundArchive::SoundInfo info;
            if ( ! m_Archive.ReadSoundInfo( id, &info ) )
            {
                break;
            }
            isLoadSuccessGroupAttachedItem = LoadDataWithEmbeddedGroup( info.fileId );
        }
        break;

    case nw::snd::internal::ItemType_Bank:
        {
            nw::snd::SoundArchive::BankInfo info;
            if ( ! m_Archive.ReadBankInfo( id, &info ) )
            {
                break;
            }
            isLoadSuccessGroupAttachedItem = LoadDataWithEmbeddedGroup( info.fileId );
        }
        break;

    #if 0   // SoundPlayer は WSDSET や SEQSET, GROUP を明示的にロードすることはない
    case nw::snd::internal::ItemType_SoundGroup:
        {
        }
        break;
    case nw::snd::internal::ItemType_WaveArchive:
        {
        }
        break;
    #endif
    }
    if ( isLoadSuccessGroupAttachedItem )
    {
        return true;
    }

    NW_WARNING(false, "[Err ] failed to LoadData ... %s[%08X]", GetItemLabel(id), id);
    return false;
}

bool PreviewSoundArchive::LoadDataWithEmbeddedGroup(
    nw::snd::SoundArchive::FileId fileId )
{
    // 埋め込みグループに所属しているファイルをリストアップ
#if defined(NW_PLATFORM_CAFE)
    const nw::snd::internal::Util::Table<ut::ResU32>* table =
#else
    const nw::snd::internal::Util::Table<u32>* table =
#endif
        m_Archive.detail_GetAttachedGroupTable( fileId );

    if ( table == NULL )
    {
        return false;
    }

    // 埋め込みグループに所属していない
    if ( table->count == 0 )
    {
        return false;
    }

    return m_DataManager.LoadData(
        table->item[0],
        &m_Heap,
        nw::snd::SoundDataManager::LOAD_ALL,
        1024*8 );
}



} // nw::snd
} // nw
