﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>
#include <models/PreviewSound.h>
#include <models/PreviewSoundArchive.h>

namespace
{

#if !defined(NW_ENABLE_SNDCTRL)

bool
ReadGlobalVariable(
    nw::snd::SequenceSoundHandle* /*handle*/,
    nw::snd::internal::fnd::BinS32 /*trackNo*/,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( var );

    short nWork = -1;
    bool  br    = nw::snd::SequenceSoundHandle::ReadGlobalVariable( varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteGlobalVariable(
    nw::snd::SequenceSoundHandle* /*handle*/,
    nw::snd::internal::fnd::BinS32 /*trackNo*/,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32 var
)
{
    if( nw::snd::SequenceVariableTraits::InvalidValue == var ) return true;
    return nw::snd::SequenceSoundHandle::WriteGlobalVariable( varNo, (short)var );
}

bool ReadLocalVariable(
    nw::snd::SequenceSoundHandle* handle,
    nw::snd::internal::fnd::BinS32 /*trackNo*/,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = nw::snd::SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadVariable( varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteLocalVariable(
    nw::snd::SequenceSoundHandle* handle,
    nw::snd::internal::fnd::BinS32 /*trackNo*/,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32 var
)
{
    NW_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( nw::snd::SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteVariable( varNo, (short)var );
}

bool ReadTrackVariable(
    nw::snd::SequenceSoundHandle* handle,
    nw::snd::internal::fnd::BinS32 trackNo,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = nw::snd::SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadTrackVariable( trackNo, varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteTrackVariable(
    nw::snd::SequenceSoundHandle* handle,
    nw::snd::internal::fnd::BinS32 trackNo,
    nw::snd::internal::fnd::BinS32 varNo,
    nw::snd::internal::fnd::BinS32 var
)
{
    NW_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( nw::snd::SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteTrackVariable( trackNo, varNo, (short)var );
}

#endif

}

namespace nw {
namespace snd {

#if !defined(NW_ENABLE_SNDCTRL)

PreviewSound::PreviewSound() :
m_pPreviewSoundArchive(NULL),
m_SoundID(nw::snd::SoundArchive::INVALID_ID),
m_State(STATE_STOPPED),
m_IsParameterDirty(false),
m_StartOffset(0),
m_Pitch(1.f),
m_LpfFrequency(0.f),
m_BiquadFilterType(BIQUAD_FILTER_TYPE_NONE),
m_BiquadFilterValue(0.5f),
m_Volume(1.f),
m_Pan(0.f),
m_SurroundPan(0.f),
m_MainSend(0.f),
m_FxSendA(0.f),
m_FxSendB(0.f),
m_FxSendC(0.f),
m_MainOutVolume(1.f),
m_MainPan(0.f),
m_MainSurroundPan(0.f),
m_MainMainSend(0.f),
m_MainFxSendA(0.f),
m_MainFxSendB(0.f),
m_MainFxSendC(0.f),
m_DrcOutVolume(1.f),
m_DrcPan(0.f),
m_DrcSurroundPan(0.f),
m_DrcMainSend(0.f),
m_DrcFxSendA(0.f),
m_DrcFxSendB(0.f),
m_DrcFxSendC(0.f),
m_Remote0OutVolume(1.f),
m_Remote1OutVolume(1.f),
m_Remote2OutVolume(1.f),
m_Remote3OutVolume(1.f),
m_IsMainOutEnabled(true),
m_IsDrcOutEnabled(false),
m_IsRemote0OutEnabled(false),
m_IsRemote1OutEnabled(false),
m_IsRemote2OutEnabled(false),
m_IsRemote3OutEnabled(false)
{
    for (s32 i = 0; i < Constants::PREVIEW_SOUND_TRACK_MAX; ++i)
    {
        m_TracksMute[i] = false;
        m_TracksVolume[i] = 1.0f;
    }
}

void PreviewSound::Initialize(PreviewSoundArchive* previewSoundArchive)
{
    NW_ASSERT_NOT_NULL(previewSoundArchive);
    m_pPreviewSoundArchive = previewSoundArchive;
}

void PreviewSound::Finalize()
{
    // todo:
    // Initialize で、再度初期化する前にリセットが必要な処理（メモリ確保など）が追加されたら
    // こちらに後始末の処理を実装してください。
}

const nw::snd::SoundHandle& PreviewSound::GetSoundHandle() const
{
    return m_SoundHandle;
}

nw::snd::SoundHandle& PreviewSound::GetSoundHandle()
{
    return m_SoundHandle;
}

nw::snd::SoundArchive::ItemId PreviewSound::GetSoundID() const
{
    return m_SoundID;
}

void PreviewSound::SetSoundID(nw::snd::SoundArchive::ItemId soundID)
{
    if(m_SoundID == soundID)
    {
        return;
    }

    m_SoundID = soundID;
    SetParameterDirty(true);
}

u32 PreviewSound::GetSoundIndex() const
{
    // HACK : インデックスを抽出する正式な方法が見つかれば置き換える
    return nw::snd::internal::Util::GetItemIndex(m_SoundID);
}

void PreviewSound::SetSoundIndex(u32 index)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewSound is not initialized.");
    SetSoundID(m_pPreviewSoundArchive->GetSoundID(index));
}

void PreviewSound::SetLabel(const char* label)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewSound is not initialized.");
    SetSoundID(m_pPreviewSoundArchive->GetSoundID(label));
}

const char* PreviewSound::GetLabel() const
{
    //NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewSound is not initialized.");

    if (m_pPreviewSoundArchive == NULL)
    {
        return "";
    }

    if(m_SoundID == nw::snd::SoundArchive::INVALID_ID)
    {
        return "";
    }
    else
    {
        return m_pPreviewSoundArchive->GetItemLabel(m_SoundID);
    }
}

PreviewSound::State PreviewSound::GetState() const
{
    return m_State;
}

void PreviewSound::Update()
{
    // m_SoundIDにマッチするデータがあればそれで設定する
    SetSoundIndex(ValidateSoundID() ? GetSoundIndex() : 0);

    UpdateState();
    ApplyParameters();
}

bool PreviewSound::Play()
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewSound is not initialized.\n");
        return false;
    }

    bool result = m_pPreviewSoundArchive->StartSound(*this, m_SoundID);
    UpdateState();

    // 再生開始に成功したらパラメータを再設定します。
    if(result)
    {
        SetParameterDirty(true);
        ApplyParameters();
    }

    return result;
}

bool PreviewSound::Stop()
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewSound is not initialized.\n");
        return false;
    }

    if(!m_SoundHandle.IsAttachedSound())
    {
        return false;
    }

    m_SoundHandle.Stop(0);
    UpdateState();

    return true;
}

bool PreviewSound::Pause()
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewSound is not initialized.\n");
        return false;
    }

    if(!m_SoundHandle.IsAttachedSound())
    {
        return false;
    }

    bool isPaused = m_SoundHandle.IsPause();

    m_SoundHandle.Pause(!isPaused, 0);
    UpdateState();

    return true;
}

void PreviewSound::Next(s32 offset)
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewSound is not initialized.\n");
        return;
    }

    s32 soundCount = static_cast<s32>(m_pPreviewSoundArchive->GetSoundCount());

    if(soundCount == 0)
    {
        return;
    }

    s32 soundIndex = static_cast<s32>(GetSoundIndex()) + offset;

    if(soundIndex >= soundCount)
    {
        soundIndex = 0;
    }
    else if(soundIndex < 0)
    {
        soundIndex = soundCount - 1;
    }

    SetSoundIndex(soundIndex);
}

void PreviewSound::ApplyParameters()
{
    if(!m_IsParameterDirty)
    {
        return;
    }

    if(!m_SoundHandle.IsAttachedSound())
    {
        return;
    }

    // 全体設定
    m_SoundHandle.SetVolume(m_Volume);
    m_SoundHandle.SetPan(m_Pan);
    m_SoundHandle.SetSurroundPan(m_SurroundPan);
    m_SoundHandle.SetMainSend(m_MainSend);
    m_SoundHandle.SetFxSend(AUX_BUS_A, m_FxSendA);
    m_SoundHandle.SetFxSend(AUX_BUS_B, m_FxSendB);
    m_SoundHandle.SetFxSend(AUX_BUS_C, m_FxSendC);

    // MAIN設定
    m_SoundHandle.SetOutputVolume(OUTPUT_DEVICE_MAIN, m_MainOutVolume);
    m_SoundHandle.SetOutputPan(OUTPUT_DEVICE_MAIN, m_MainPan);
    m_SoundHandle.SetOutputSurroundPan(OUTPUT_DEVICE_MAIN, m_MainSurroundPan);
    m_SoundHandle.SetOutputMainSend(OUTPUT_DEVICE_MAIN, m_MainMainSend);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_MAIN, AUX_BUS_A, m_MainFxSendA);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_MAIN, AUX_BUS_B, m_MainFxSendB);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_MAIN, AUX_BUS_C, m_MainFxSendC);

    // DRC設定
    m_SoundHandle.SetOutputVolume(OUTPUT_DEVICE_DRC, m_DrcOutVolume);
    m_SoundHandle.SetOutputPan(OUTPUT_DEVICE_DRC, m_DrcPan);
    m_SoundHandle.SetOutputSurroundPan(OUTPUT_DEVICE_DRC, m_DrcSurroundPan);
    m_SoundHandle.SetOutputMainSend(OUTPUT_DEVICE_DRC, m_DrcMainSend);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_DRC, AUX_BUS_A, m_DrcFxSendA);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_DRC, AUX_BUS_B, m_DrcFxSendB);
    m_SoundHandle.SetOutputFxSend(OUTPUT_DEVICE_DRC, AUX_BUS_C, m_DrcFxSendC);

    // リモート設定
    m_SoundHandle.SetRemoteOutVolume(0, m_Remote0OutVolume);
    m_SoundHandle.SetRemoteOutVolume(1, m_Remote1OutVolume);
    m_SoundHandle.SetRemoteOutVolume(2, m_Remote2OutVolume);
    m_SoundHandle.SetRemoteOutVolume(3, m_Remote3OutVolume);

    // 共通設定
    m_SoundHandle.SetPitch(m_Pitch);
    m_SoundHandle.SetLpfFreq(m_LpfFrequency);
    m_SoundHandle.SetBiquadFilter(m_BiquadFilterType, m_BiquadFilterValue);

    u32 outputLine = 0;
    if(m_IsMainOutEnabled)
    {
        outputLine |= OUTPUT_LINE_MAIN;
    }
    if(m_IsDrcOutEnabled)
    {
        outputLine |= OUTPUT_LINE_DRC0 | OUTPUT_LINE_DRC1;
    }
    if(m_IsRemote0OutEnabled)
    {
        outputLine |= OUTPUT_LINE_REMOTE0;
    }
    if(m_IsRemote1OutEnabled)
    {
        outputLine |= OUTPUT_LINE_REMOTE1;
    }
    if(m_IsRemote2OutEnabled)
    {
        outputLine |= OUTPUT_LINE_REMOTE2;
    }
    if(m_IsRemote3OutEnabled)
    {
        outputLine |= OUTPUT_LINE_REMOTE3;
    }

    m_SoundHandle.SetOutputLine(outputLine);

    if (m_pPreviewSoundArchive->GetSoundArchive().GetSoundType(m_SoundID) == nw::snd::SoundArchive::SOUND_TYPE_SEQ)
    {
        SequenceSoundHandle handle(&m_SoundHandle);

        u32 trackFlag = 0x1;
        for (s32 i = 0; i < Constants::PREVIEW_SOUND_TRACK_MAX; ++i)
        {
            handle.SetTrackMute(trackFlag, GetIsTrackMuteEnabled(i));
            handle.SetTrackVolume(trackFlag, GetTrackVolume(i));
            trackFlag <<= 1;
        }
    }

    if (m_pPreviewSoundArchive->GetSoundArchive().GetSoundType(m_SoundID) == nw::snd::SoundArchive::SOUND_TYPE_STRM)
    {
        StreamSoundHandle handle(&m_SoundHandle);

        u32 trackFlag = 0x1;
        for (s32 i = 0; i < Constants::PREVIEW_SOUND_STREAM_TRACK_MAX; ++i)
        {
            if (GetIsTrackMuteEnabled(i))
            {
                handle.SetTrackVolume(trackFlag, 0.0f);
            }
            else
            {
                handle.SetTrackVolume(trackFlag, GetTrackVolume(i));
            }

            trackFlag <<= 1;
        }
    }

    // シーケンス変数の初期値を設定する
    ApplySequenceVariables();

    m_IsParameterDirty = false;
}

u32 PreviewSound::GetStartOffset() const
{
    return m_StartOffset;
}

void PreviewSound::SetStartOffset(u32 value)
{
    m_StartOffset = value;
}

f32 PreviewSound::GetPitch() const
{
    return m_Pitch;
}

void PreviewSound::SetPitch(f32 value)
{
    SetParameterDirty(m_Pitch != value);
    m_Pitch = value;
}

f32 PreviewSound::GetLpfFrequency() const
{
    return m_LpfFrequency;
}

void PreviewSound::SetLpfFrequency(f32 value)
{
    SetParameterDirty(m_LpfFrequency != value);
    m_LpfFrequency = value;
}

BiquadFilterType PreviewSound::GetBiquadFilterType() const
{
    return m_BiquadFilterType;
}

void PreviewSound::SetBiquadFilterType(BiquadFilterType value)
{
    SetParameterDirty(m_BiquadFilterType != value);
    m_BiquadFilterType = value;
}

f32 PreviewSound::GetBiquadFilterValue() const
{
    return m_BiquadFilterValue;
}

void PreviewSound::SetBiquadFilterValue(f32 value)
{
    SetParameterDirty(m_BiquadFilterValue != value);
    m_BiquadFilterValue = value;
}

f32 PreviewSound::GetVolume() const
{
    return m_Volume;
}

void PreviewSound::SetVolume(f32 value)
{
    SetParameterDirty(m_Volume != value);
    m_Volume = value;
}

f32 PreviewSound::GetPan() const
{
    return m_Pan;
}

void PreviewSound::SetPan(f32 value)
{
    SetParameterDirty(m_Pan != value);
    m_Pan = value;
}

f32 PreviewSound::GetSurroundPan() const
{
    return m_SurroundPan;
}

void PreviewSound::SetSurroundPan(f32 value)
{
    SetParameterDirty(m_SurroundPan != value);
    m_SurroundPan = value;
}

f32 PreviewSound::GetMainSend() const
{
    return m_MainSend;
}

void PreviewSound::SetMainSend(f32 value)
{
    SetParameterDirty(m_MainSend != value);
    m_MainSend = value;
}

f32 PreviewSound::GetFxSend(AuxBus auxBus) const
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_FxSendA;

    case AUX_BUS_B:
        return m_FxSendB;

    case AUX_BUS_C:
        return m_FxSendC;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
}

void PreviewSound::SetFxSend(AuxBus auxBus, f32 value)
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        SetParameterDirty(m_FxSendA != value);
        m_FxSendA = value;
        break;

    case AUX_BUS_B:
        SetParameterDirty(m_FxSendB != value);
        m_FxSendB = value;
        break;

    case AUX_BUS_C:
        SetParameterDirty(m_FxSendC != value);
        m_FxSendC = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
}

f32 PreviewSound::GetMainOutVolume() const
{
    return m_MainOutVolume;
}

void PreviewSound::SetMainOutVolume(f32 value)
{
    SetParameterDirty(m_MainOutVolume != value);
    m_MainOutVolume = value;
}

f32 PreviewSound::GetMainPan() const
{
    return m_MainPan;
}

void PreviewSound::SetMainPan(f32 value)
{
    SetParameterDirty(m_MainPan != value);
    m_MainPan = value;
}

f32 PreviewSound::GetMainSurroundPan() const
{
    return m_MainSurroundPan;
}

void PreviewSound::SetMainSurroundPan(f32 value)
{
    SetParameterDirty(m_MainSurroundPan != value);
    m_MainSurroundPan = value;
}

f32 PreviewSound::GetMainMainSend() const
{
    return m_MainMainSend;
}

void PreviewSound::SetMainMainSend(f32 value)
{
    SetParameterDirty(m_MainMainSend != value);
    m_MainMainSend = value;
}

f32 PreviewSound::GetMainFxSend(AuxBus auxBus) const
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_MainFxSendA;

    case AUX_BUS_B:
        return m_MainFxSendB;

    case AUX_BUS_C:
        return m_MainFxSendC;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
}

void PreviewSound::SetMainFxSend(AuxBus auxBus, f32 value)
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        SetParameterDirty(m_MainFxSendA != value);
        m_MainFxSendA = value;
        break;

    case AUX_BUS_B:
        SetParameterDirty(m_MainFxSendB != value);
        m_MainFxSendB = value;
        break;

    case AUX_BUS_C:
        SetParameterDirty(m_MainFxSendC != value);
        m_MainFxSendC = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
}

f32 PreviewSound::GetDrcOutVolume() const
{
    return m_DrcOutVolume;
}

void PreviewSound::SetDrcOutVolume(f32 value)
{
    SetParameterDirty(m_DrcOutVolume != value);
    m_DrcOutVolume = value;
}

f32 PreviewSound::GetDrcPan() const
{
    return m_DrcPan;
}

void PreviewSound::SetDrcPan(f32 value)
{
    SetParameterDirty(m_DrcPan != value);
    m_DrcPan = value;
}

f32 PreviewSound::GetDrcSurroundPan() const
{
    return m_DrcSurroundPan;
}

void PreviewSound::SetDrcSurroundPan(f32 value)
{
    SetParameterDirty(m_DrcSurroundPan != value);
    m_DrcSurroundPan = value;
}

f32 PreviewSound::GetDrcMainSend() const
{
    return m_DrcMainSend;
}

void PreviewSound::SetDrcMainSend(f32 value)
{
    SetParameterDirty(m_DrcMainSend != value);
    m_DrcMainSend = value;
}

f32 PreviewSound::GetDrcFxSend(AuxBus auxBus) const
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_DrcFxSendA;

    case AUX_BUS_B:
        return m_DrcFxSendB;

    case AUX_BUS_C:
        return m_DrcFxSendC;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
}

void PreviewSound::SetDrcFxSend(AuxBus auxBus, f32 value)
{
    switch(auxBus)
    {
    case AUX_BUS_A:
        SetParameterDirty(m_DrcFxSendA != value);
        m_DrcFxSendA = value;
        break;

    case AUX_BUS_B:
        SetParameterDirty(m_DrcFxSendB != value);
        m_DrcFxSendB = value;
        break;

    case AUX_BUS_C:
        SetParameterDirty(m_DrcFxSendC != value);
        m_DrcFxSendC = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
}

f32 PreviewSound::GetRemote0OutVolume() const
{
    return m_Remote0OutVolume;
}

void PreviewSound::SetRemote0OutVolume(f32 value)
{
    SetParameterDirty(m_Remote0OutVolume != value);
    m_Remote0OutVolume = value;
}

f32 PreviewSound::GetRemote1OutVolume() const
{
    return m_Remote1OutVolume;
}

void PreviewSound::SetRemote1OutVolume(f32 value)
{
    SetParameterDirty(m_Remote1OutVolume != value);
    m_Remote1OutVolume = value;
}

f32 PreviewSound::GetRemote2OutVolume() const
{
    return m_Remote2OutVolume;
}

void PreviewSound::SetRemote2OutVolume(f32 value)
{
    SetParameterDirty(m_Remote2OutVolume != value);
    m_Remote2OutVolume = value;
}

f32 PreviewSound::GetRemote3OutVolume() const
{
    return m_Remote3OutVolume;
}

void PreviewSound::SetRemote3OutVolume(f32 value)
{
    SetParameterDirty(m_Remote3OutVolume != value);
    m_Remote3OutVolume = value;
}

bool PreviewSound::GetIsMainOutEnabled() const
{
    return m_IsMainOutEnabled;
}

void PreviewSound::SetIsMainOutEnabled(bool value)
{
    SetParameterDirty(m_IsMainOutEnabled != value);
    m_IsMainOutEnabled = value;
}

bool PreviewSound::GetIsDrcOutEnabled() const
{
    return m_IsDrcOutEnabled;
}

void PreviewSound::SetIsDrcOutEnabled(bool value)
{
    SetParameterDirty(m_IsDrcOutEnabled != value);
    m_IsDrcOutEnabled = value;
}

bool PreviewSound::GetIsRemote0OutEnabled() const
{
    return m_IsRemote0OutEnabled;
}

void PreviewSound::SetIsRemote0OutEnabled(bool value)
{
    SetParameterDirty(m_IsRemote0OutEnabled != value);
    m_IsRemote0OutEnabled = value;
}

bool PreviewSound::GetIsRemote1OutEnabled() const
{
    return m_IsRemote1OutEnabled;
}

void PreviewSound::SetIsRemote1OutEnabled(bool value)
{
    SetParameterDirty(m_IsRemote1OutEnabled != value);
    m_IsRemote1OutEnabled = value;
}

bool PreviewSound::GetIsRemote2OutEnabled() const
{
    return m_IsRemote2OutEnabled;
}

void PreviewSound::SetIsRemote2OutEnabled(bool value)
{
    SetParameterDirty(m_IsRemote2OutEnabled != value);
    m_IsRemote2OutEnabled = value;
}

bool PreviewSound::GetIsRemote3OutEnabled() const
{
    return m_IsRemote3OutEnabled;
}

void PreviewSound::SetIsRemote3OutEnabled(bool value)
{
    SetParameterDirty(m_IsRemote3OutEnabled != value);
    m_IsRemote3OutEnabled = value;
}

bool PreviewSound::GetIsTrackMuteEnabled(u32 index) const
{
    return m_TracksMute[index];
}

void PreviewSound::SetIsTrackMuteEnabled(u32 index, bool value)
{
    SetParameterDirty(m_TracksMute[index] != value);
    m_TracksMute[index] = value;
}

f32 PreviewSound::GetTrackVolume(u32 index) const
{
    return m_TracksVolume[index];
}

void PreviewSound::SetTrackVolume(u32 index, f32 value)
{
    SetParameterDirty(m_TracksVolume[index] != value);
    m_TracksVolume[index] = value;
}

bool PreviewSound::GetVariables(ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    switch( pVariables->header.variableType ){
    case SequenceVariableType_Global:
        return GetVariables( pVariables, ReadGlobalVariable );

    case SequenceVariableType_Local:
        return GetVariables( pVariables, ReadLocalVariable );

    case SequenceVariableType_Track:
        return GetVariables( pVariables, ReadTrackVariable );
    }

    InvalidateVariables( pVariables );
    return false;
}

bool PreviewSound::SetVariables(const ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    bool result = false;

    switch( pVariables->header.variableType ){
    case SequenceVariableType_Global:
        GetGlobalVariables().Parse( pVariables );
        result = SetVariables( GetGlobalVariables(), WriteGlobalVariable );
        break;

    case SequenceVariableType_Local:
        m_LocalVariables.Parse( pVariables );
        result = SetVariables( m_LocalVariables, WriteLocalVariable );
        break;

    case SequenceVariableType_Track:
        if( !SequenceVariableTraits::ValidateTrackNo( pVariables->header.trackNo ) ){
            NW_ASSERT( false );
            break;
        }

        m_TrackVariables[ pVariables->header.trackNo ].Parse( pVariables );
        result = SetVariables( m_TrackVariables[ pVariables->header.trackNo ], WriteTrackVariable );
        break;
    }

    return result;
}

void PreviewSound::InvalidateAllVariables()
{
    m_LocalVariables.InvalidateAll();

    for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
        m_TrackVariables[ i ].InvalidateAll();
    }
}

void PreviewSound::ApplySequenceVariables()
{
    SetVariables( GetGlobalVariables(), WriteGlobalVariable );
    SetVariables( m_LocalVariables, WriteLocalVariable );

    for( s32 i = 0; i < SequenceVariableTraits::TrackCount; i++ )
    {
        SetVariables( m_TrackVariables[ i ], WriteTrackVariable );
    }
}

bool PreviewSound::GetVariables(ToolSoundSequenceVariableContainer* pVariables, ReadVariableFunc pFunc)
{
    NW_ASSERT_NOT_NULL( pVariables );
    NW_ASSERT_NOT_NULL( pFunc );

#if defined(NW_ENABLE_SNDCTRL)
    SequenceSoundHandle seqSoundHandle( m_SoundController->GetSoundHandle() );
#else
    SequenceSoundHandle seqSoundHandle( &m_SoundHandle );
#endif

    // シーケンス変数の値を取得する（割り込み禁止）
    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_Lock);

        for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
            pFunc( &seqSoundHandle, (int)pVariables->header.trackNo, i, &pVariables->variables[ i ] );
        }
    }

    return true;
}

bool PreviewSound::SetVariables(SequenceVariableContainer& Variables, WriteVariableFunc pFunc)
{
    NW_ASSERT_NOT_NULL( pFunc );

#if defined(NW_ENABLE_SNDCTRL)
    SequenceSoundHandle seqSoundHandle( m_SoundController->GetSoundHandle() );
#else
    SequenceSoundHandle seqSoundHandle( &m_SoundHandle );
#endif

    // シーケンス変数の値を更新する（割り込み禁止）
    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_Lock);

        for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
            pFunc( &seqSoundHandle, (int)Variables.GetTrackNo(), i, Variables[ i ].GetValue() );
        }
    }

    return true;
}

void PreviewSound::InvalidateVariables(ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    // シーケンス変数の値を無効値に設定する
    for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
        pVariables->variables[ i ] = SequenceVariableTraits::InvalidValue;
    }
}

SequenceVariableContainer& PreviewSound::GetGlobalVariables()
{
    NW_ASSERT_NOT_NULL( m_pPreviewSoundArchive );
    return m_pPreviewSoundArchive->GetGlobalVariables();
}

void PreviewSound::UpdateState()
{
    State newState = m_State;

    if(!m_SoundHandle.IsAttachedSound())
    {
        newState = STATE_STOPPED;
    }
    else if(m_SoundHandle.IsPause())
    {
        newState = STATE_PAUSED;
    }
    else
    {
        newState = STATE_PLAYING;
    }

    if(m_State == newState)
    {
        return;
    }

    m_State = newState;
    SetParameterDirty(true);
}

void PreviewSound::SetParameterDirty(bool isDirty)
{
    m_IsParameterDirty = m_IsParameterDirty || isDirty;
}

bool PreviewSound::ValidateSoundID()
{
    if (m_SoundID == nw::snd::SoundArchive::INVALID_ID)
    {
        return false;
    }

    // 登録されているアーカイブに、m_SoundIDに一致するデータが含まれるかどうかを調べます。
    if (m_pPreviewSoundArchive == NULL ||
        !m_pPreviewSoundArchive->IsOpened() )
    {
        return false;
    }

    const s32 soundCount = m_pPreviewSoundArchive->GetSoundCount();
    if (soundCount < 0)
    {
        return false;
    }

    for (int i = 0; i < soundCount; ++i)
    {
        if (m_pPreviewSoundArchive->GetSoundID(i) == m_SoundID)
        {
            return true;
        }
    }

    // 見つからなかった
    return false;
}

#endif

} // nw::snd
} // nw
