﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#include <nw/snd.h>
#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>
#include <mcs/handler/SetSoundParametersHandler.h>
#include <Constants.h>

namespace
{

/* バスの種類 */
enum ToolSoundBiquadType {
    /* Inherit  */
    BiquadType_Inherit =	-1,
    /* OFF  */
    BiquadType_Off =	0,
    /* LPF  */
    BiquadType_Lpf =	1,
    /* HPF  */
    BiquadType_Hpf =	2,
    /* BPF512  */
    BiquadType_Bpf512 =	3,
    /* BPF1024  */
    BiquadType_Bpf1024 =	4,
    /* BPF2048  */
    BiquadType_Bpf2048 =	5,
    /* enumを4Byte型にするための定義 */
    ToolSoundBiquadType_4BYTE_ENUM =	0xFFFFFFFF
};

/* プレビュープレイヤーパラメータ */
typedef struct {
    /*  4B: 対象プレビュープレイヤーのインデックス */
    nw::snd::internal::fnd::BinU32	previewPlayerIndex;
    /*  4B: 音量 */
    nw::snd::internal::fnd::BinF32	volume;
    /*  4B: ピッチ */
    nw::snd::internal::fnd::BinF32	pitch;
    /*  4B: パン */
    nw::snd::internal::fnd::BinF32	pan;
    /*  4B: サラウンドパン */
    nw::snd::internal::fnd::BinF32	surroundPan;
    /*  4B: パン(DRC) */
    nw::snd::internal::fnd::BinF32	drcPan;
    /*  4B: LPF周波数 */
    nw::snd::internal::fnd::BinF32	lpfFrequency;
    /*  4B: BiquadType */
    nw::snd::internal::fnd::BinS32	biquadType;
    /*  4B: BiquadValue */
    nw::snd::internal::fnd::BinF32	biquadValue;
    /*  4B: 再生開始位置 [ms] */
    nw::snd::internal::fnd::BinU32   startOffsetTime;
    /*  4B: メイン 出力スイッチ */
    nw::snd::internal::fnd::BinS32	mainOutEnabled;
    /*  4B: DRC 出力スイッチ */
    nw::snd::internal::fnd::BinS32	drcOutEnabled;
    /*  4B: Wiiリモコン1 出力スイッチ */
    nw::snd::internal::fnd::BinS32	remote0OutEnabled;
    /*  4B: Wiiリモコン2 出力スイッチ */
    nw::snd::internal::fnd::BinS32	remote1OutEnabled;
    /*  4B: Wiiリモコン3 出力スイッチ */
    nw::snd::internal::fnd::BinS32	remote2OutEnabled;
    /*  4B: Wiiリモコン4 出力スイッチ */
    nw::snd::internal::fnd::BinS32	remote3OutEnabled;
    /*  4B: メイン 出力 */
    nw::snd::internal::fnd::BinF32	mainOutVolume;
    /*  4B: DRC 出力 */
    nw::snd::internal::fnd::BinF32	drcOutVolume;
    /*  4B: Wiiリモコン1 出力 */
    nw::snd::internal::fnd::BinF32	remote0OutVolume;
    /*  4B: Wiiリモコン2 出力 */
    nw::snd::internal::fnd::BinF32	remote1OutVolume;
    /*  4B: Wiiリモコン3 出力 */
    nw::snd::internal::fnd::BinF32	remote2OutVolume;
    /*  4B: Wiiリモコン4 出力 */
    nw::snd::internal::fnd::BinF32	remote3OutVolume;
    /*  4B: メイン センド */
    nw::snd::internal::fnd::BinF32	mainSend;
    /*  4B: エフェクト Aux A センド */
    nw::snd::internal::fnd::BinF32	effectAuxASend;
    /*  4B: エフェクト Aux B センド */
    nw::snd::internal::fnd::BinF32	effectAuxBSend;
    /*  4B: エフェクト Aux C センド */
    nw::snd::internal::fnd::BinF32	effectAuxCSend;
    /*  4B: フェードインフレーム */
    nw::snd::internal::fnd::BinS32	fadeInFrames;
    /*  4B: フェードアウトフレーム */
    nw::snd::internal::fnd::BinS32	fadeOutFrames;
    /*  4B: ポーズフェードフレーム */
    nw::snd::internal::fnd::BinS32	pauseFadeFrames;
    /*  1B*16=16B: トラックミュート */
    nw::snd::internal::fnd::BinU8	tracksMute[nw::snd::Constants::PREVIEW_SOUND_TRACK_MAX];
    /*  4B*16=64B: トラックボリューム */
    nw::snd::internal::fnd::BinF32	tracksVolume[nw::snd::Constants::PREVIEW_SOUND_TRACK_MAX];
    /*  4B: パン(メイン) */
    nw::snd::internal::fnd::BinF32	mainPan;
    /*  4B: サラウンドパン(メイン) */
    nw::snd::internal::fnd::BinF32	mainSurroundPan;
    /*  4B: サラウンドパン(DRC) */
    nw::snd::internal::fnd::BinF32	drcSurroundPan;
    /*  4B: メイン センド(メイン) */
    nw::snd::internal::fnd::BinF32	mainMainSend;
    /*  4B: エフェクト Aux A センド(メイン) */
    nw::snd::internal::fnd::BinF32	mainEffectAuxASend;
    /*  4B: エフェクト Aux B センド(メイン) */
    nw::snd::internal::fnd::BinF32	mainEffectAuxBSend;
    /*  4B: エフェクト Aux C センド(メイン) */
    nw::snd::internal::fnd::BinF32	mainEffectAuxCSend;
    /*  4B: メイン センド(DRC) */
    nw::snd::internal::fnd::BinF32	drcMainSend;
    /*  4B: エフェクト Aux A センド(DRC) */
    nw::snd::internal::fnd::BinF32	drcEffectAuxASend;
    /*  4B: エフェクト Aux B センド(DRC) */
    nw::snd::internal::fnd::BinF32	drcEffectAuxBSend;
    /*  4B: エフェクト Aux C センド(DRC) */
    nw::snd::internal::fnd::BinF32	drcEffectAuxCSend;
} ToolSoundPreviewPlayerParameter;

nw::snd::BiquadFilterType GetBiquadFilterType(ToolSoundBiquadType type)
{
    switch (type)
    {
    case BiquadType_Inherit: return nw::snd::BIQUAD_FILTER_TYPE_INHERIT;
    case BiquadType_Off:     return nw::snd::BIQUAD_FILTER_TYPE_NONE;
    case BiquadType_Lpf:     return nw::snd::BIQUAD_FILTER_TYPE_LPF;
    case BiquadType_Hpf:     return nw::snd::BIQUAD_FILTER_TYPE_HPF;
    case BiquadType_Bpf512:  return nw::snd::BIQUAD_FILTER_TYPE_BPF512;
    case BiquadType_Bpf1024: return nw::snd::BIQUAD_FILTER_TYPE_BPF1024;
    case BiquadType_Bpf2048: return nw::snd::BIQUAD_FILTER_TYPE_BPF2048;
    }

    return nw::snd::BIQUAD_FILTER_TYPE_NONE;
}

}

namespace nw {
namespace snd {

//----------------------------------------------------------
void
SetSoundParametersHandler::Initialize(
    PreviewSoundArchive& previewSoundArchive,
    HioChannel& activeChannel)
{
    m_PreviewSoundArchive = &previewSoundArchive;
    m_ActiveChannel = &activeChannel;
}

//----------------------------------------------------------
void
SetSoundParametersHandler::OnInvoke(
    const HioMessageHeader& header,
    const void* data,
    nw::ut::IoStream& writeStream)
{
#if defined(NW_DEBUG)
    NW_LOG("SetSoundParametersHandler\n");
#endif

    NW_NULL_ASSERT(m_PreviewSoundArchive);
    NW_NULL_ASSERT(data);

    const ToolSoundPreviewPlayerParameter* pParameter =
        reinterpret_cast<const ToolSoundPreviewPlayerParameter*>(data);

    PreviewSound& previewSound = m_PreviewSoundArchive->GetPreviewSound(pParameter->previewPlayerIndex);

    previewSound.SetVolume(pParameter->volume);
    previewSound.SetPitch(pParameter->pitch);
    previewSound.SetPan(pParameter->pan);
    previewSound.SetSurroundPan(pParameter->surroundPan);
    previewSound.SetLpfFrequency(pParameter->lpfFrequency);
    previewSound.SetBiquadFilterType(GetBiquadFilterType(
        static_cast<ToolSoundBiquadType>(s32(pParameter->biquadType))));
    previewSound.SetBiquadFilterValue(pParameter->biquadValue);
    previewSound.SetMainSend(pParameter->mainSend);
    previewSound.SetFxSend(AUX_BUS_A, pParameter->effectAuxASend);
    previewSound.SetFxSend(AUX_BUS_B, pParameter->effectAuxBSend);
    previewSound.SetFxSend(AUX_BUS_C, pParameter->effectAuxCSend);
    previewSound.SetMainOutVolume(pParameter->mainOutVolume);
    previewSound.SetDrcOutVolume(pParameter->drcOutVolume);
    previewSound.SetRemote0OutVolume(pParameter->remote0OutVolume);
    previewSound.SetRemote1OutVolume(pParameter->remote1OutVolume);
    previewSound.SetRemote2OutVolume(pParameter->remote2OutVolume);
    previewSound.SetRemote3OutVolume(pParameter->remote3OutVolume);
    previewSound.SetIsMainOutEnabled(pParameter->mainOutEnabled != 0);
    previewSound.SetIsDrcOutEnabled(pParameter->drcOutEnabled != 0);
    previewSound.SetIsRemote0OutEnabled(pParameter->remote0OutEnabled != 0);
    previewSound.SetIsRemote1OutEnabled(pParameter->remote1OutEnabled != 0);
    previewSound.SetIsRemote2OutEnabled(pParameter->remote2OutEnabled != 0);
    previewSound.SetIsRemote3OutEnabled(pParameter->remote3OutEnabled != 0);

    for (s32 i = 0; i < Constants::PREVIEW_SOUND_TRACK_MAX; ++i)
    {
        previewSound.SetIsTrackMuteEnabled(i, pParameter->tracksMute[i] == 1);
        previewSound.SetTrackVolume(i, f32(pParameter->tracksVolume[i]));
    }

    previewSound.SetMainPan(pParameter->mainPan);
    previewSound.SetMainSurroundPan(pParameter->mainSurroundPan);
    previewSound.SetMainMainSend(pParameter->mainMainSend);
    previewSound.SetMainFxSend(AUX_BUS_A, pParameter->mainEffectAuxASend);
    previewSound.SetMainFxSend(AUX_BUS_B, pParameter->mainEffectAuxBSend);
    previewSound.SetMainFxSend(AUX_BUS_C, pParameter->mainEffectAuxCSend);

    previewSound.SetDrcPan(pParameter->drcPan);
    previewSound.SetDrcSurroundPan(pParameter->drcSurroundPan);
    previewSound.SetDrcMainSend(pParameter->drcMainSend);
    previewSound.SetDrcFxSend(AUX_BUS_A, pParameter->drcEffectAuxASend);
    previewSound.SetDrcFxSend(AUX_BUS_B, pParameter->drcEffectAuxBSend);
    previewSound.SetDrcFxSend(AUX_BUS_C, pParameter->drcEffectAuxCSend);

    u32 reply = HIO_RESULT_SUCCESS;
    HioMessageHeader replyHeader(header.message, sizeof(u32));
    m_ActiveChannel->GetStream().Write(&replyHeader, sizeof(HioMessageHeader));
    m_ActiveChannel->GetStream().Write(&reply, sizeof(u32));
}

} // snd
} // nw
