﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <controls/AuxBusControl.h>
#include <models/PreviewAuxBus.h>
#include <models/PreviewEffectUtility.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace snd {

AuxBusControl::AuxBusControl() :
m_pModel(NULL)
{
    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_Container.SetMargin(nw::internal::dw::Thickness(0.f));
    m_Container.SetOrientation(nw::internal::dw::HORIZONTAL);

    m_Label.SetMargin(nw::internal::dw::Thickness(2.f, 1.f, 12.f, 1.f));

    for(s32 i = 0; i < MAX_EFFECT_COUNT; ++i)
    {
        EffectLamp& effect = m_Effects[i];
        effect.SetMargin(nw::internal::dw::Thickness(3.f, 1.f));
        effect.SetMinimumWidth(75.f);
        effect.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO_HEIGHT);
    }

    SetEffectCount(DEFAULT_EFFECT_COUNT);
}

AuxBusControl::~AuxBusControl()
{
}

void AuxBusControl::Initialize(PreviewAuxBus& model)
{
    m_pModel = &model;
}

void AuxBusControl::Finalize()
{
    m_pModel = NULL;
}

const char* AuxBusControl::GetText() const
{
    return m_Label.GetText();
}

void AuxBusControl::SetText(const char* value)
{
    m_Label.SetText(value);
}

s32 AuxBusControl::GetEffectCount() const
{
    // m_Label 分を差し引きます。
    return m_Container.GetContents().GetCount() - 1;
}

void AuxBusControl::SetEffectCount(s32 value)
{
    NW_ASSERT(value < MAX_EFFECT_COUNT);

    nw::internal::dw::UIElementList& contents = m_Container.GetContents();

    contents.ClearItems();
    contents.AddItem(&m_Label);

    for(s32 i = 0; i < value; ++i)
    {
        contents.AddItem(&m_Effects[i]);
    }
}

s32 AuxBusControl::GetSelectedEffectIndex()
{
    if(!GetIsContainsFocus())
    {
        return INVALID_INDEX;
    }

    for(s32 i = 0; i < GetEffectCount(); ++i)
    {
        if(m_Effects[i].GetIsContainsFocus())
        {
            return i;
        }
    }

    return INVALID_INDEX;
}

EffectLamp* AuxBusControl::SetSelectedEffectIndex(s32 index)
{
    if(0 <= index && index < GetEffectCount())
    {
        EffectLamp& effect = m_Effects[index];

        if(effect.GetIsSelected())
        {
            return &effect;
        }
    }

    for(s32 i = 0; i < GetEffectCount(); ++i)
    {
        m_Effects[i].SetIsSelected(false);
    }

    if( index == INVALID_INDEX)
    {
        return NULL;
    }

    EffectLamp& effect = m_Effects[index];
    effect.SetIsSelected(true);

    return &effect;
}


void AuxBusControl::SelectEffect(s32 index)
{
    EffectLamp* pEffect = SetSelectedEffectIndex(index);

    if(pEffect == NULL)
    {
        return;
    }

    pEffect->SetIsSelected(true);
    pEffect->SetFocus();
}

void AuxBusControl::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    if(m_pModel == NULL)
    {
        return;
    }

    // エフェクトボタンのテキストを更新します。
    for(s32 i = 0; i < GetEffectCount(); ++i)
    {
        EffectLamp& effect = m_Effects[i];
        PreviewEffectSet* pEffectSet = m_pModel->GetEffect(i);
        NW_NULL_ASSERT(pEffectSet);

        effect.SetText(
            PreviewEffectUtility::ToShortString(pEffectSet->GetCurrent().GetEffectType()));

        effect.SetTextColor(
            pEffectSet->GetAssignedEffect() == NULL ?
              nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_DARK_GRAY())
              : nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_YELLOW_GREEN()));
    }
}

void AuxBusControl::OnRender(const nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer, nw::internal::dw::UIElementRenderArgs& args) const
{
    nw::internal::dw::Thickness padding = GetPadding();
    nw::internal::dw::Thickness labelMargin = m_Label.GetMargin();

    f32 fromX = padding.left + m_Label.GetFixedWidth() + labelMargin.GetWidth();
    f32 toX   = padding.left + GetFixedWidth() - padding.right - 4;
    f32 y     = padding.top + (GetFixedHeight() - padding.GetHeight()) / 2;

    // ラインを描画します。
    renderer.DrawLine(
        &context,
        nw::internal::dw::DrawLineArgs().
        SetFrom(fromX, y).
        SetTo(toX, y).
        SetColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::GRAY())));

    // 矢印を描画します。
    f32 arrowFromX  = toX - 6;
    f32 arrowDeltaY = 4;

    renderer.DrawLine(
        &context,
        nw::internal::dw::DrawLineArgs().
        SetFrom(arrowFromX, y - arrowDeltaY).
        SetTo(toX, y).
        SetColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::GRAY())));

    renderer.DrawLine(
        &context,
        nw::internal::dw::DrawLineArgs().
        SetFrom(arrowFromX, y + arrowDeltaY).
        SetTo(toX, y).
        SetColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::GRAY())));
}

} // snd
} // nw
