﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SOUNDPLAYER_HIO_PROTOCOL_H_
#define NW_SOUNDPLAYER_HIO_PROTOCOL_H_

#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>

namespace nw {
namespace snd {

static const u32 HIO_PROTOCOL_VERSION = 0x00010002;     //!< HIO通信プロトコルのバージョン

/*
    headerのmessage部の内訳。
    MessageTypeは1bitのReplyFlag、6bitのCategory
    8bitのCommandからなります。

    15             8 7             0
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    | |R| Category  |    Command    |
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

enum
{
    HIO_MESSAGE_REPLY_MASK          = 0x4000,
    HIO_MESSAGE_CATEGORY_MASK       = 0x3f00,
    HIO_MESSAGE_ID_MASK             = 0x00ff,
    HIO_MESSAGE_CATEGORY_BIT_OFFSET = 0x8,
    HIO_MESSAGE_ID_BIT_OFFSET       = 0x00
};

//---------------------------------------------------------------------------
//! @brief  サウンドで利用するメッセージのカテゴリです。（TODO:★整理する）
//---------------------------------------------------------------------------
enum HioMessageCategory
{
    HIO_MESSAGE_CATEGORY_SYSTEM      = 0x0000,     //!< サウンドシステム用のカテゴリです。
    HIO_MESSAGE_CATEGORY_SOUNDPLAYER = 0x0300,     //!< SoundPlayer 用のカテゴリです。
};

//---------------------------------------------------------------------------
//! @brief  サウンドで利用するチャンネル番号です。
//---------------------------------------------------------------------------
enum HioChannelType
{
    HIO_BASE_CHANNEL                = 0x8000 + 'SD',
    HIO_SOUNDPLAYER_PASSIVE_CHANNEL = HIO_BASE_CHANNEL + 0x0,   //!< PC->実機 方向に要求するチャンネルです。(仮SoundPlayer用)
    HIO_SOUNDPLAYER_ACTIVE_CHANNEL  = HIO_BASE_CHANNEL + 0x1,   //!< 実機->PC 方向に要求するチャンネルです。(仮SoundPlayer用)
    //HIO_PASSIVE_CHANNEL             = HIO_BASE_CHANNEL + 0x2,   //!< PC->実機 方向に要求するチャンネルです。
    //HIO_ACTIVE_CHANNEL              = HIO_BASE_CHANNEL + 0x3,   //!< 実機->PC 方向に要求するチャンネルです。
    HIO_PING_CHANNEL                = HIO_BASE_CHANNEL + 0xf,   //!< 実機の生存確認を行うチャンネルです。
};

//---------------------------------------------------------------------------
//! @brief  SNDEDIT が受け取れる HIO メッセージの種類です。
//---------------------------------------------------------------------------
typedef u32 HioMessageType;

enum
{
    //! @brief  無効なメッセージIDです。
    HIO_MESSAGE_INVALID = 0,

    //---------------------------------------------------------------------------
    // システムカテゴリ
    //---------------------------------------------------------------------------

    //! @brief  接続を要求します。
    HIO_MESSAGE_CONNECT = 0x00 + HIO_MESSAGE_CATEGORY_SYSTEM + HIO_MESSAGE_REPLY_MASK,

    //---------------------------------------------------------------------------
    // SoundPlayer カテゴリ
    //---------------------------------------------------------------------------

    //! @brief  サウンドアーカイブを開きます。
    HIO_MESSAGE_OPEN_SOUNDARCHIVE       = 0x41 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  サウンドアーカイブを閉じます。
    HIO_MESSAGE_CLOSE_SOUNDARCHIVE      = 0x53 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  Echo。
    HIO_MESSAGE_ECHO                    = 0x02 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  全発音を停止します。
    HIO_MESSAGE_STOP_ALL                = 0x03 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  出力モードを設定します。
    HIO_MESSAGE_SET_OUTPUT_MODE         = 0x04 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  サウンドを選択します。
    HIO_MESSAGE_SELECT_SOUNDS           = 0x42 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  サウンド情報を取得します。
    HIO_MESSAGED_GET_SOUND_INFORMATIONS = 0x43 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  指定サウンドを再生します。
    HIO_MESSAGE_PLAY_SOUND              = 0x45 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  指定サウンドを一時停止します。
    HIO_MESSAGE_PAUSE_SOUND             = 0x46 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  指定サウンドを停止します。
    HIO_MESSAGE_STOP_SOUND              = 0x47 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  エフェクトパラメータを設定します。
    HIO_MESSAGE_SET_EFFECT_PARAMETERS   = 0x48 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  シーケンス変数を設定します。
    HIO_MESSAGE_SET_SEQUENCE_VARIABLES  = 0x50 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  シーケンス変数を取得します。
    HIO_MESSAGE_GET_SEQUENCE_VARIABLES  = 0x51 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  サウンドパラメータを設定します。
    HIO_MESSAGE_SET_SOUND_PARAMETERS    = 0x52 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  3Dモデルデータを読み込みます。
    HIO_MESSAGE_LOAD_3D_MODEL           = 0x81 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  アニメーションデータを読み込みます。
    HIO_MESSAGE_LOAD_3D_ANIMATION       = 0x82 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  アニメーションサウンドデータを読み込みます。
    HIO_MESSAGE_LOAD_ANIMSOUND          = 0x83 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  アニメーションフレームを取得します。
    HIO_MESSAGE_GET_3D_ANIMATION_FRAME  = 0x86 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  アニメーションフレームを設定します。
    HIO_MESSAGE_SET_3D_ANIMATION_FRAME  = 0x87 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  現在のアニメーションを再生します。
    HIO_MESSAGE_PLAY_ANIMATION          = 0x88 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  再生中のアニメーションを停止します。
    HIO_MESSAGE_STOP_ANIMATION          = 0x89 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  アニメーション再生レートを設定します。
    HIO_MESSAGE_SET_ANIMATION_RATE       = 0x90 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  アニメーションのループ設定を行います。
    HIO_MESSAGE_SET_ANIMATION_LOOP       = 0x91 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  アニメーションの再生範囲を設定します。
    HIO_MESSAGE_SET_3D_ANIMATION_RANGE   = 0x92 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER + HIO_MESSAGE_REPLY_MASK,

    //! @brief  アニメーションサウンドイベントを編集します。
    HIO_MESSAGE_EDIT_ANIMSOUND_EVENT     = 0x93 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,

    //! @brief  アニメーションサウンドイベントのIDをすべて更新します。
    HIO_MESSAGE_UPDATE_ALL_EVENT_ID      = 0x94 + HIO_MESSAGE_CATEGORY_SOUNDPLAYER,
};

enum HioResult
{
    HIO_RESULT_SUCCESS = 1,     //!< 成功しました。 TODO:★互換性のために 1 としているが、0 に変更にする
    HIO_RESULT_FAILED = 0,      //!< 失敗しました。 TODO:★互換性のために 0 としているが、1 に変更にする
};


//---------------------------------------------------------------------------
//! @brief  HIOメッセージです。
//---------------------------------------------------------------------------
struct HioMessage
{
    const internal::fnd::BinU32 value;

    //---------------------------------------------------------------------------

    /* ctor */ HioMessage() : value(0) { }
    /* ctor */ HioMessage(u32 value) : value(value) { }

    //---------------------------------------------------------------------------
    //! @brief   返信が必要なHIOメッセージかどうかを調べます。
    //!
    //! @return  返信が必要な場合は true、不要な場合は false を返します。
    //---------------------------------------------------------------------------
    bool ShouldReply() const
    {
        return (value & HIO_MESSAGE_REPLY_MASK) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief   HIOメッセージカテゴリを取得します。
    //!
    //! @return  HIOメッセージカテゴリを返します。
    //---------------------------------------------------------------------------
    u32 GetCategory() const
    {
        return (value & HIO_MESSAGE_ID_MASK) >> HIO_MESSAGE_CATEGORY_BIT_OFFSET;
    }

    //---------------------------------------------------------------------------
    //! @brief   HIOメッセージID を取得します。
    //!
    //! @return  HIOメッセージID を返します。
    //---------------------------------------------------------------------------
    u32 GetID() const
    {
        return value & HIO_MESSAGE_ID_MASK;
    }

    //--------------------------------------------------------------------------
    //! @brief   u32 型へのキャスト演算子です。
    //!
    //! @return  u32 で表現した値を返します。
    //---------------------------------------------------------------------------
    operator u32() const
    {
        return value;
    }

#if defined(NW_DEBUG)
    //--------------------------------------------------------------------------
    //! @brief   指定メッセージを文字列に変換します。
    //!
    //! @param[in]  message  メッセージを指定します。
    //!
    //! @return  変換した文字列を返します。
    //---------------------------------------------------------------------------
    static const char* ToString(u32 message)
    {
        switch(message)
        {
        case HIO_MESSAGE_INVALID:
            return "HIO_MESSAGE_INVALID";

        case HIO_MESSAGE_CONNECT:
            return "HIO_MESSAGE_CONNECT";

        case HIO_MESSAGE_OPEN_SOUNDARCHIVE:
            return "HIO_MESSAGE_OPEN_SOUNDARCHIVE";

        case HIO_MESSAGE_CLOSE_SOUNDARCHIVE:
            return "HIO_MESSAGE_CLOSE_SOUNDARCHIVE";

        case HIO_MESSAGE_ECHO:
            return "HIO_MESSAGE_ECHO";

        case HIO_MESSAGE_STOP_ALL:
            return "HIO_MESSAGE_STOP_ALL";

        case HIO_MESSAGE_SET_OUTPUT_MODE:
            return "HIO_MESSAGE_SET_OUTPUT_MODE";

        case HIO_MESSAGE_SELECT_SOUNDS:
            return "HIO_MESSAGE_SELECT_SOUNDS";

        case HIO_MESSAGED_GET_SOUND_INFORMATIONS:
            return "HIO_MESSAGED_GET_SOUND_INFORMATIONS";

        case HIO_MESSAGE_PLAY_SOUND:
            return "HIO_MESSAGE_PLAY_SOUND";

        case HIO_MESSAGE_PAUSE_SOUND:
            return "HIO_MESSAGE_PAUSE_SOUND";

        case HIO_MESSAGE_STOP_SOUND:
            return "HIO_MESSAGE_STOP_SOUND";

        case HIO_MESSAGE_SET_EFFECT_PARAMETERS:
            return "HIO_MESSAGE_SET_EFFECT_PARAMETERS";

        case HIO_MESSAGE_SET_SEQUENCE_VARIABLES:
            return "HIO_MESSAGE_SET_SEQUENCE_VARIABLES";

        case HIO_MESSAGE_GET_SEQUENCE_VARIABLES:
            return "HIO_MESSAGE_GET_SEQUENCE_VARIABLES";

        case HIO_MESSAGE_SET_SOUND_PARAMETERS:
            return "HIO_MESSAGE_SET_SOUND_PARAMETERS";

        case HIO_MESSAGE_LOAD_3D_MODEL:
            return "HIO_MESSAGE_LOAD_3D_MODEL";

        case HIO_MESSAGE_LOAD_3D_ANIMATION:
            return "HIO_MESSAGE_LOAD_3D_ANIMATION";

        case HIO_MESSAGE_LOAD_ANIMSOUND:
            return "HIO_MESSAGE_LOAD_ANIMSOUND";

        case HIO_MESSAGE_GET_3D_ANIMATION_FRAME:
            return "HIO_MESSAGE_GET_3D_ANIMATION_FRAME";

        case HIO_MESSAGE_SET_3D_ANIMATION_FRAME:
            return "HIO_MESSAGE_SET_3D_ANIMATION_FRAME";

        case HIO_MESSAGE_PLAY_ANIMATION:
            return "HIO_MESSAGE_PLAY_ANIMATION";

        case HIO_MESSAGE_STOP_ANIMATION:
            return "HIO_MESSAGE_STOP_ANIMATION";

        case HIO_MESSAGE_SET_ANIMATION_RATE:
            return "HIO_MESSAGE_SET_ANIMATION_RATE";

        case HIO_MESSAGE_SET_ANIMATION_LOOP:
            return "HIO_MESSAGE_SET_ANIMATION_LOOP";

        case HIO_MESSAGE_SET_3D_ANIMATION_RANGE:
            return "HIO_MESSAGE_SET_3D_ANIMATION_RANGE";

        case HIO_MESSAGE_EDIT_ANIMSOUND_EVENT:
            return "HIO_MESSAGE_EDIT_ANIMSOUND_EVENT";

        case HIO_MESSAGE_UPDATE_ALL_EVENT_ID:
            return "HIO_MESSAGE_UPDATE_ALL_EVENT_ID";
        }

        return "Unknown Message";
    }

    //--------------------------------------------------------------------------
    //! @brief   メッセージを文字列に変換します。
    //!
    //! @return  変換した文字列を返します。
    //---------------------------------------------------------------------------
    const char* ToString() const
    {
        return ToString(value);
    }
#endif
};

//---------------------------------------------------------------------------
//! @brief  HIOメッセージヘッダです。
//---------------------------------------------------------------------------
struct HioMessageHeader
{
    HioMessage message;
    internal::fnd::BinU32 size;

    //---------------------------------------------------------------------------

    /* ctor */ HioMessageHeader()
        : message(HIO_MESSAGE_INVALID),
          size(0)
    {}

    /* ctor */ HioMessageHeader(HioMessage message)
        : message(message),
          size(0)
    {}

    /* ctor */ HioMessageHeader(HioMessage message, u32 size)
        : message(message),
          size(size)
    {}
};

//---------------------------------------------------------------------------
//! @brief  HIO返信メッセージヘッダです。
//---------------------------------------------------------------------------
struct HioReplyMessageHeader : HioMessageHeader
{
    HioResult result;

    //---------------------------------------------------------------------------

    /* ctor */ HioReplyMessageHeader()
        : HioMessageHeader(HIO_MESSAGE_INVALID, 0)
    {}

    /* ctor */ HioReplyMessageHeader(HioMessage message)
        : HioMessageHeader(message, 0)
    {}

    /* ctor */ HioReplyMessageHeader(HioMessage message, u32 size)
        : HioMessageHeader(message, size)
    {}
};

} // namespace nw::snd
} // namespace nw

#endif // NW_SOUNDPLAYER_HIO_PROTOCOL_H_
