﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMakerPlugin;
    using Runtime = NintendoWare.SoundRuntime;

    public class PreviewStreamChannelGeneric : PreviewChannelGeneric, IPreviewStreamChannel
    {
        private Runtime.ChannelCafe _channel = null;
        private Runtime.WaveInfo _waveInfo;
        private StreamSoundTrackBase streamSoundTrack = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="streamTrack">対象のトラックを指定します。</param>
        /// <param name="baseTrack">基準となるトラックを指定します。</param>
        /// <param name="channelIndexStart">チャンネルインデックスの開始番号</param>
        /// <param name="volume">音量を指定します。</param>
        /// <param name="panMode">パンモードを指定します。</param>
        /// <param name="panCurve">パンカーブを指定します。</param>
        public PreviewStreamChannelGeneric(
            StreamSoundTrackBase streamTrack,
            StreamSoundTrackBase baseTrack,
            int channelIndexStart,
            float volume,
            float pitch,
            PanMode panMode,
            PanCurve panCurve,
            float mainSend,
            Runtime.ChannelCafe.RendererType rendererType,
            bool forceNoLoop)
        {
            Initialize(streamTrack, baseTrack, channelIndexStart, volume, pitch, panMode, panCurve, mainSend, rendererType, forceNoLoop);

            this.streamSoundTrack = streamTrack;
        }

        public override Runtime.ChannelCafe Channel
        {
            get { return _channel; }
        }

        public void ChannelSetUserVolume(float volume)
        {
            _channel.SetUserVolume(volume);
        }

        public StreamSoundTrackBase StreamSoundTrack
        {
            get
            {
                return this.streamSoundTrack;
            }
        }

        protected override Runtime.WaveInfo WaveInfo
        {
            get { return _waveInfo; }
        }

        private void Initialize(
            StreamSoundTrackBase streamTrack,
            StreamSoundTrackBase baseTrack,
            int channelIndexStart,
            float volume,
            float pitch,
            PanMode panMode,
            PanCurve panCurve,
            float mainSend,
            Runtime.ChannelCafe.RendererType rendererType,
            bool forceNoLoop)
        {
            if (null == streamTrack) { throw new ArgumentNullException("streamTrack"); }
            if (null == baseTrack) { throw new ArgumentNullException("baseTrack"); }

            string baseTrackFilePath = baseTrack.FilePath;
            string streamTrackFilePath = streamTrack.FilePath;

            // AAC, Opus ファイルの場合には、同階層フォルダの波形ファイルで代用します。
            if (AACUtil.IsAACFile(baseTrackFilePath) == true || FileUtil.IsOpusFile(baseTrackFilePath) == true)
            {
                if ((baseTrackFilePath = ChangeFilePathForWaveEncodeNoConvert(baseTrackFilePath)) == null)
                {
                    throw new CompressionAudioPreviewFileNotFoundException("file not found.", baseTrack.FilePath);
                }
            }

            if (AACUtil.IsAACFile(streamTrackFilePath) == true || FileUtil.IsOpusFile(streamTrackFilePath) == true)
            {
                if ((streamTrackFilePath = ChangeFilePathForWaveEncodeNoConvert(streamTrackFilePath)) == null)
                {
                    throw new CompressionAudioPreviewFileNotFoundException("file not found.", streamTrack.FilePath);
                }
            }

            //
            if (!File.Exists(baseTrackFilePath))
            {
                throw new FileNotFoundException("file not found.", baseTrack.FilePath);
            }

            if (!File.Exists(streamTrackFilePath))
            {
                throw new FileNotFoundException("file not found.", streamTrackFilePath);
            }

            bool? isLoop = null;
            int? loopStart = null;
            int? loopEnd = null;

            if (forceNoLoop == true)
            {
                isLoop = false;
            }

            WaveFileInfo baseWaveFileInfo = null;
            StreamSoundBase sound = streamTrack.Parent as StreamSoundBase;

            int? sampleRate = null;
            int? channelCount = null;

            if (sound != null)
            {
                sampleRate = sound.GetTargetSampleRate();
                channelCount = sound.GetTargetChannelCount();

                baseWaveFileInfo = (WaveFileInfo)WaveFileManager.Instance.LoadFile(baseTrackFilePath, isLoop, null, null, sampleRate, channelCount);

                if (baseWaveFileInfo == null)
                {
                    throw new Exception();
                }

                if (forceNoLoop == false)
                {
                    isLoop = baseWaveFileInfo.IsLoop;
                }
                loopStart = (int)baseWaveFileInfo.LoopStartFrame;
                loopEnd = (int)baseWaveFileInfo.LoopEndFrame;
            }

            if (null == baseWaveFileInfo) { throw new Exception(); }


            WaveFileInfo waveFileInfo = null;

            if (streamTrack == baseTrack)
            {
                waveFileInfo = baseWaveFileInfo;
            }
            else
            {
                waveFileInfo = (WaveFileInfo)WaveFileManager.Instance.LoadFile(streamTrackFilePath, isLoop, loopStart, loopEnd, sampleRate, channelCount);
            }

            if (null == waveFileInfo) { throw new Exception(); }


            Runtime.ChannelCafe newChannel = new Runtime.ChannelCafe();

            int channelCountValue = streamTrack.ChannelCount;
            if (channelCount.HasValue == true)
            {
                channelCountValue = channelCount.Value;
            }

            bool result = newChannel.AllocChannel(
                channelCountValue,
                32, // priority
                rendererType
            );
            if (!result) { throw new Exception(); }

            newChannel.SetUserVolume((volume / 127.0f) * (streamTrack.Volume / 127.0f));
            newChannel.SetUserPitchRatio(pitch);
            newChannel.SetUserPan((streamTrack.Pan - 64) / 63.0f);
            newChannel.SetMainSend(((mainSend / 127.0f) - 1.0f) + ((streamTrack.Sends.MainSend / 127.0f) - 1.0f));
            newChannel.SetPanMode(panMode.ToRuntimePanMode());
            newChannel.SetPanCurve(panCurve.ToRuntimePanCurve());
            newChannel.SetBiquadFilter(streamTrack.BiquadType.ToRuntimeBiquadFilterType(),
                                        (float)streamTrack.Biquad / 127.0F);
            newChannel.SetUserLpfFreq((float)(streamTrack.LPF - 64) / 64.0F);

            _channel = newChannel;
            _waveInfo = waveFileInfo.ToWaveInfo(channelIndexStart, channelCountValue);
        }

        /// <summary>
        ///
        /// </summary>
        private string ChangeFilePathForWaveEncodeNoConvert(string filePath)
        {
            string[] extensions = new string[] {
                "wav",
                "aiff",
                "aif",
            };

            string tempFilePath = filePath;
            foreach (string extension in extensions)
            {
                tempFilePath = Path.ChangeExtension(filePath, extension);
                if (File.Exists(tempFilePath) != false)
                {
                    return tempFilePath;
                }
            }

            return null;
        }
    }
}

