﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StdAfx.h"
#include "MidiManager.h"

UnmanagedMidiManager::UnmanagedMidiManager()
: mRegisterCallbackFlag( false ),
  mNoteOnCallback( NULL ),
  mUserData( NULL )
{
    ::InitializeCriticalSection( &mCritSec );
}

UnmanagedMidiManager::~UnmanagedMidiManager()
{
    Shutdown();

    ::DeleteCriticalSection( &mCritSec );
}

void UnmanagedMidiManager::Setup( NoteOnCallback noteOnCallback, void* userData )
{
    nn::atk::detail::driver::SoundThread::GetInstance().Lock();
    EnterCriticalSection( &mCritSec );

    Shutdown();

    mNoteOnCallback = noteOnCallback;
    mUserData = userData;

    nn::atk::detail::driver::SoundThread::GetInstance().RegisterPlayerCallback( this );
    mRegisterCallbackFlag = true;

    mGuiPlayer.Start( this );

    LeaveCriticalSection( &mCritSec );
    nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
}

void UnmanagedMidiManager::Shutdown()
{
    nn::atk::detail::driver::SoundThread::GetInstance().Lock();
    EnterCriticalSection( &mCritSec );

    mGuiPlayer.Stop();

    for( PlayerList::iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        player->midiIn.Close();
        player->midiPlayer.Stop();
        delete player;
    }
    mPlayerList.clear();

    if ( mRegisterCallbackFlag ) {
        nn::atk::detail::driver::SoundThread::GetInstance().UnregisterPlayerCallback( this );
        mRegisterCallbackFlag = false;
    }

    mNoteOnCallback = NULL;

    LeaveCriticalSection( &mCritSec );
    nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
}

bool UnmanagedMidiManager::OpenMidiIn( UINT deviceID )
{
    nn::atk::detail::driver::SoundThread::GetInstance().Lock();
    EnterCriticalSection( &mCritSec );

    Player* player = new Player();
    player->deviceID = deviceID;

    if ( ! player->midiIn.Open( deviceID ) ) {
        delete player;
        LeaveCriticalSection( &mCritSec );
        nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
        return false;
    }

    if ( ! player->midiPlayer.Start( this ) ) {
        delete player;
        LeaveCriticalSection( &mCritSec );
        nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
        return false;
    }

    mPlayerList.push_back( player );

    LeaveCriticalSection( &mCritSec );
    nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
    return true;
}

void UnmanagedMidiManager::UpdateMidiPlayers()
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        if ( player->midiIn.IsErrorOccuered() ) {
            player->midiPlayer.Reset();
            player->midiIn.ResetErrorFlag();
            continue;
        }

        MidiIn::MidiMsg midiMsg;
        while( player->midiIn.ReadMidiMsg( &midiMsg ) ) {
            player->midiPlayer.SendMidiMessage(
                midiMsg.status,
                midiMsg.data1,
                midiMsg.data2
            );
        }
    }

    LeaveCriticalSection( &mCritSec );
}

nn::atk::detail::driver::Channel* UnmanagedMidiManager::NoteOn(
    nn::atk::detail::driver::SequenceSoundPlayer* seqPlayer,
    uint8_t bankIndex,
    const nn::atk::detail::driver::NoteOnInfo& noteOnInfo
)
{
    if ( mNoteOnCallback == NULL ) {
        return NULL;
    }

    return mNoteOnCallback( seqPlayer, bankIndex, noteOnInfo, mUserData );
}

void UnmanagedMidiManager::SendMidiMessage( int status, int data1, int data2 )
{
    mGuiPlayer.SendMidiMessage( status, data1, data2 );
}

void UnmanagedMidiManager::ProgramChange( int channelIndex, int prgNo )
{
    mGuiPlayer.ProgramChange( channelIndex, prgNo );
}

void UnmanagedMidiManager::SendMidiMessage( UINT deviceID, int status, int data1, int data2 )
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        if ( player->deviceID == deviceID ) {
            player->midiPlayer.SendMidiMessage( status, data1, data2 );
        }
    }

    LeaveCriticalSection( &mCritSec );
}

void UnmanagedMidiManager::ProgramChange( UINT deviceID, int channelIndex, int prgNo )
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        if ( player->deviceID == deviceID ) {
            player->midiPlayer.ProgramChange( channelIndex, prgNo );
        }
    }

    LeaveCriticalSection( &mCritSec );
}

bool UnmanagedMidiManager::IsActive( DWORD msec ) const
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::const_iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        if ( player->midiIn.IsActive( msec ) ) {
            LeaveCriticalSection( &mCritSec );
            return true;
        }
    }

    LeaveCriticalSection( &mCritSec );
    return false;
}

void UnmanagedMidiManager::Reset()
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        player->midiIn.Reset();
        player->midiPlayer.Reset();
    }

    LeaveCriticalSection( &mCritSec );
}

const nn::atk::detail::driver::MidiSequencePlayer* UnmanagedMidiManager::GetSequenceSoundPlayer( UINT devideID ) const
{
    EnterCriticalSection( &mCritSec );

    for( PlayerList::const_iterator p = mPlayerList.begin(); p != mPlayerList.end() ; p++ ) {
        Player* player = *p;
        if ( devideID == player->deviceID ) {
            LeaveCriticalSection( &mCritSec );
            return &player->midiPlayer.GetSequenceSoundPlayer();
        }
    }

    LeaveCriticalSection( &mCritSec );
    return NULL;
}
