﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StdAfx.h"
#include "MidiIn.h"

MidiIn::MidiIn()
: mMidiInHandle( NULL ),
  mMidiInActive( false ),
  mLastMidiInTime( 0 ),
  mMidiMsgOverFlag( false ),
  mMidiMsgReadPos( 0 ),
  mMidiMsgWritePos( 0 )
{
    ::InitializeCriticalSection( &mCritSec );
}

MidiIn::~MidiIn()
{
    Close();

    ::DeleteCriticalSection( &mCritSec );
}

bool MidiIn::Open( UINT nDeviceID )
{
    Close();

    MMRESULT result = midiInOpen(
        &mMidiInHandle,
        nDeviceID,
        (DWORD_PTR)MidiInProcFunc,
        (DWORD_PTR)this,
        CALLBACK_FUNCTION
    );
    if ( result != MMSYSERR_NOERROR ) {
        return false;
    }

    mMidiMsgOverFlag = false;
    mMidiMsgReadPos = 0;
    mMidiMsgWritePos = 0;

    result = midiInStart( mMidiInHandle );
    if ( result != MMSYSERR_NOERROR ) {
        midiInClose( mMidiInHandle );
        return false;
    }
    return true;
}

void MidiIn::Close()
{
    if ( mMidiInHandle == NULL ) return;

    MMRESULT ret = midiInStop( mMidiInHandle );
    if ( ret != MMSYSERR_NOERROR ) {
        assert( false );
    }
    ret = midiInReset( mMidiInHandle );
    if ( ret != MMSYSERR_NOERROR ) {
        assert( false );
    }
    ret = midiInClose( mMidiInHandle );
    if ( ret != MMSYSERR_NOERROR ) {
        assert( false );
    }

    mMidiInActive = false;
    mMidiInHandle = NULL;
}

void MidiIn::Reset()
{
    EnterCriticalSection( &mCritSec );

    mMidiMsgOverFlag = false;
    mMidiMsgReadPos = 0;
    mMidiMsgWritePos = 0;

    LeaveCriticalSection( &mCritSec );
}

bool MidiIn::IsActive( DWORD msec ) const
{
    if ( ! mMidiInActive ) return false;

    if ( timeGetTime() - mLastMidiInTime < msec ) return true;

    return false;
}

void CALLBACK MidiIn::MidiInProcFunc(
    HMIDIIN hMidiIn,
    UINT wMsg,
    DWORD dwInstance,
    DWORD dwParam1,
    DWORD dwParam2
)
{
    MidiIn* midiIn = (MidiIn*)(DWORD_PTR)dwInstance;
    midiIn->MidiInProc( wMsg, dwParam1, dwParam2 );
}

void MidiIn::MidiInProc(
    UINT wMsg,
    DWORD dwParam1,
    DWORD dwParam2
)
{
    EnterCriticalSection( &mCritSec );

    switch( wMsg ) {
    case MM_MIM_DATA:
        if ( ! mMidiMsgOverFlag ) {
            MidiMsg* msg = &mMidiMsgBuffer[mMidiMsgWritePos];

            msg->status = LOBYTE( LOWORD( dwParam1 ) );
            msg->data1 = HIBYTE( LOWORD( dwParam1 ) );
            msg->data2 = LOBYTE( HIWORD( dwParam1 ) );

            mMidiMsgWritePos++;
            if ( mMidiMsgWritePos >= MIDI_MSG_BUFFER_SIZE ) {
                mMidiMsgWritePos = 0;
            }
            if ( mMidiMsgReadPos == mMidiMsgWritePos ) {
                mMidiMsgOverFlag = true;
            }
        }
        mMidiInActive = true;
        mLastMidiInTime = timeGetTime();
        break;

    case MM_MIM_OPEN:
        break;
    }

    LeaveCriticalSection( &mCritSec );
}

bool MidiIn::ReadMidiMsg( MidiMsg* msg )
{
    EnterCriticalSection( &mCritSec );

    if ( mMidiMsgOverFlag ) {
        LeaveCriticalSection( &mCritSec );
        return false;
    }

    if ( mMidiMsgReadPos == mMidiMsgWritePos ) {
        LeaveCriticalSection( &mCritSec );
        return false;
    }

    *msg = mMidiMsgBuffer[ mMidiMsgReadPos ];
    mMidiMsgReadPos++;
    if ( mMidiMsgReadPos >= MIDI_MSG_BUFFER_SIZE ) {
        mMidiMsgReadPos = 0;
    }

    LeaveCriticalSection( &mCritSec );

    return true;
}

