﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// Nintendo.Sound.Nw.h

#pragma once

#include "Atk.h"

namespace NintendoWare { namespace SoundRuntime {

using namespace System;
using namespace System::Runtime::InteropServices;
using namespace System::Threading;

static const int VOICE_CHANNEL_MAX = 2;

public enum class SampleFormat
{
    SAMPLE_FORMAT_PCM_S32,  // signed 32bit
    SAMPLE_FORMAT_PCM_S16,  // signed 16bit
    SAMPLE_FORMAT_PCM_S8,   // signed 8bit
    SAMPLE_FORMAT_DSP_ADPCM // DSP ADPCM
};

[StructLayout(LayoutKind::Sequential)]
public value struct AdpcmParam
{
    [MarshalAs(UnmanagedType::ByValArray, SizeConst = 16)]
    array<uint16_t>^ coef;
    uint16_t pred_scale;
    uint16_t yn1;
    uint16_t yn2;
};

[StructLayout(LayoutKind::Sequential)]
public value struct AdpcmLoopParam
{
    uint16_t loop_pred_scale;
    uint16_t loop_yn1;
    uint16_t loop_yn2;
};

[StructLayout(LayoutKind::Sequential)]
public value struct ChannelParam
{
    IntPtr dataAddr;
    int dataSize;
    AdpcmParam adpcmParam;
    AdpcmLoopParam adpcmLoopParam;
};

[StructLayout(LayoutKind::Sequential)]
public value struct WaveInfo
{
    SampleFormat format;
    bool loopFlag;
    int numChannels;
    int sampleRate;
    uint32_t loopStart;
    uint32_t loopEnd;
    uint32_t originalLoopStartFrame;
    size_t dataSize;

    [MarshalAs(UnmanagedType::ByValArray, SizeConst = VOICE_CHANNEL_MAX)]
    array<ChannelParam>^ channelParam;
};

public enum class PanCurve
{
    PAN_LAW_SQRT,               // 平方根のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
    PAN_LAW_SQRT_0DB,           // 平方根のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
    PAN_LAW_SQRT_0DB_CLAMP,     // 平方根のカーブ。中央で0dB、両端に振り切ると0dBとなります。
    PAN_LAW_SINCOS,             // 三角関数のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
    PAN_LAW_SINCOS_0DB,         // 三角関数のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
    PAN_LAW_SINCOS_0DB_CLAMP,   // 三角関数のカーブ。中央で0dB、両端に振り切ると0dBとなります。
    PAN_LAW_LINEAR,             // 線形のカーブ。中央で-6dB、両端に振り切ると0dBとなります。
    PAN_LAW_LINEAR_0DB,         // 線形のカーブ。中央で0dB、両端に振り切ると+6dBとなります。
    PAN_LAW_LINEAR_0DB_CLAMP    // 線形のカーブ。中央で0dB、両端に振り切ると0dBとなります。
};

public enum class PanMode
{
    PAN_MODE_DUAL,      // ステレオを２本のモノラルとしてそれぞれに定位処理を行います。
    PAN_MODE_BALANCE    // 左右チャンネルの音量バランスを処理します。
};

public ref class ChannelCafe
{
  public:
    ChannelCafe() : umChannel(NULL)
    {
        mChannelCallback = gcnew ChannelCallback(ChannelCallbackFunc);
        mhCallback = GCHandle::Alloc(mChannelCallback);
        mhChannel = GCHandle::Alloc(this);

        mUserChannelCallback = IntPtr::Zero;
        mUserChannelCallbackData = IntPtr::Zero;
    }
    ~ChannelCafe()
    {
        FreeChannel();
        mhCallback.Free();
        mhChannel.Free();
    }

    enum class RendererType
    {
        k32KHz,
        k48KHz
    };
    bool AllocChannel(
        int voiceChannelCount,
        int priority,
        RendererType renderer
    )
    {
        return AllocChannel(
            voiceChannelCount,
            priority,
            IntPtr::Zero,
            IntPtr::Zero,
            renderer
        );
    }

    static nn::atk::OutputReceiver* GetOutputReceiver()
    {
        return &nn::atk::detail::driver::HardwareManager::GetInstance().GetSubMix( 0 );
    }

    bool AllocChannel(
        int voiceChannelCount,
        int priority,
        IntPtr channelCallback,
        IntPtr channelCallbackData,
        RendererType renderer
    )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) FreeChannel();

        IntPtr ip = Marshal::GetFunctionPointerForDelegate(mChannelCallback);
        nn::atk::detail::driver::Channel::ChannelCallback cb =
            static_cast<nn::atk::detail::driver::Channel::ChannelCallback>(ip.ToPointer());

        nn::atk::detail::driver::Channel*  cp =
            nn::atk::detail::driver::Channel::AllocChannel(
                voiceChannelCount,
                priority,
                cb,
                (void*)GCHandle::ToIntPtr( mhChannel )
            );
        if ( cp == NULL ) {
            AtkSoundSystem::UnlockSoundThread();
            return false;
        }
        umChannel = cp;
        mUserChannelCallback = channelCallback;
        mUserChannelCallbackData = channelCallbackData;

        cp->SetOutputReceiver( GetOutputReceiver() );

        AtkSoundSystem::UnlockSoundThread();
        return true;
    }

    void FreeChannel()
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            nn::atk::detail::driver::Channel::FreeChannel( umChannel );
            umChannel = NULL;
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    IntPtr GetUnmanagedPtr() { return IntPtr( umChannel ); }

    void Start( WaveInfo^ waveData, int length )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            IntPtr p = Marshal::AllocCoTaskMem( Marshal::SizeOf(waveData) );
            Marshal::StructureToPtr( waveData, p, true );

            umChannel->Start(
                *static_cast<nn::atk::detail::WaveInfo*>(p.ToPointer()),
                length,
                0,
                false
            );

            Marshal::FreeCoTaskMem(p);
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void Pause(bool flag)
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->Pause( flag );
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void Stop()
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            AtkSoundSystem::LockSoundThread();
            umChannel->Stop();
            FreeChannel();
            AtkSoundSystem::UnlockSoundThread();
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    bool IsActive()
    {
        bool active = false;

        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            active = umChannel->IsActive();
        }

        AtkSoundSystem::UnlockSoundThread();

        return active;
    }

    bool IsPause()
    {
 		bool paused = false;

        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            paused = umChannel->IsPause();
        }

        AtkSoundSystem::UnlockSoundThread();

        return paused;
    }

    void Release()
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->Release();
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void NoteOff()
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->NoteOff();
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void SetUserPitchRatio( float pitch )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetUserPitchRatio( pitch );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetUserVolume( float volume )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetUserVolume( volume );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetUserPan( float pan )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetUserPan( pan );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetInitPan( float pan )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetInitPan( pan );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetTune( float tune )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetTune( tune );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetKey( int key )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetKey( key );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetKey( int key, int originalKey )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetKey( key, originalKey );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetInstrumentVolume( float instrumentVolume )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetInstrumentVolume( instrumentVolume );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetVelocity( float velocity )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetVelocity( velocity );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetEnvelope( int a, int h, int d, int s, int r )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetAttack( a );
            umChannel->SetHold( h );
            umChannel->SetDecay( d );
            umChannel->SetSustain( s );
            umChannel->SetRelease( r );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetRelease( int release )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetRelease( release );
        }

        AtkSoundSystem::UnlockSoundThread();
    }
    void SetMainSend( float mainSend )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetMainSend( mainSend );
        }

        AtkSoundSystem::UnlockSoundThread();
   }

    void SetPanCurve( PanCurve panCurve )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetPanCurve( static_cast<nn::atk::PanCurve>(panCurve) );
        }

        AtkSoundSystem::UnlockSoundThread();
   }

    void SetPanMode( PanMode panMode )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetPanMode( static_cast<nn::atk::PanMode>(panMode) );
        }

        AtkSoundSystem::UnlockSoundThread();
   }

    void SetReleaseIgnore( bool ignore )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            // umChannel->SetReleaseIgnore( ignore );
            umChannel->SetIsIgnoreNoteOff( ignore );
        }

        AtkSoundSystem::UnlockSoundThread();
   }

    void SetAlternateAssignId( int id )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            // umChannel->SetAlternateAssignId( id );
            umChannel->SetKeyGroupId( id );
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void SetBiquadFilter( int type, float value )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetBiquadFilter( type, value );
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void SetUserLpfFreq( float value )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetUserLpfFreq( value );
        }

        AtkSoundSystem::UnlockSoundThread();
    }

    void SetInterpolationType( int type )
    {
        AtkSoundSystem::LockSoundThread();

        if ( umChannel != NULL ) {
            umChannel->SetInterpolationType( type );
        }

        AtkSoundSystem::UnlockSoundThread();
    }

  private:
    [UnmanagedFunctionPointer(CallingConvention::Cdecl)]
    delegate void ChannelCallback(
        IntPtr channel,
        nn::atk::detail::driver::Channel::ChannelCallbackStatus status,
        IntPtr userData
    );

    static void ChannelCallbackFunc(
        IntPtr channelPtr,
        nn::atk::detail::driver::Channel::ChannelCallbackStatus status,
        IntPtr userData
    )
    {
        nn::atk::detail::driver::Channel*  cp =
            static_cast<nn::atk::detail::driver::Channel*>( channelPtr.ToPointer() );

        GCHandle gch = GCHandle::FromIntPtr( userData );
        ChannelCafe^ channel = (ChannelCafe^)gch.Target;

        if ( cp != channel->umChannel ) {
            return;
        }

        if ( channel->mUserChannelCallback != IntPtr::Zero ) {
            nn::atk::detail::driver::Channel::ChannelCallback  channelCallback =
                static_cast<nn::atk::detail::driver::Channel::ChannelCallback>( channel->mUserChannelCallback.ToPointer() );

            channelCallback( cp, status, channel->mUserChannelCallbackData.ToPointer() );
        }

#if 0
        switch ( status )
        {
        case nn::atk::detail::driver::Channel::CALLBACK_STATUS_STOPPED:
        case nn::atk::detail::driver::Channel::CALLBACK_STATUS_FINISH:
            nn::atk::detail::driver::Channel::FreeChannel( channel->umChannel );
            break;
        }
#endif

        channel->umChannel = NULL;
    }

    nn::atk::detail::driver::Channel* umChannel;
    GCHandle mhChannel;
    GCHandle mhCallback;
    ChannelCallback^ mChannelCallback;
    IntPtr mUserChannelCallback;
    IntPtr mUserChannelCallbackData;
};

}}
