﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <cmath>
#include "SmfEvent.h"
#include "smfconv.h"
#include "SmfParser.h"

#ifdef max
#undef max
#endif

using namespace std;
using namespace sndlib;

namespace
{
const int BANK_SELECT_MAX = 3;
} // anonymous namespace

//////////////////////////////////////////////////////////////////////
//
// BeatEvent
//
//////////////////////////////////////////////////////////////////////
BeatEvent::BeatEvent(tick_t t, int len, const byte_t* data, int tpqn)
: MetaEvent(t),
  measure( numeric_limits<unsigned int>::max() )
{
    numerator = data[0];
    denomirator = (1 << data[1]);
    click_interval = data[2];
    midi_clock_duration = data[3];

    division = tpqn * 4 / denomirator;
    if (tpqn * 4 != division * denomirator) {
        throw MetaEventException("Too small TPQN", t);
    }
}

void BeatEvent::setNext(tick_t next)
{
    long delta = next - tick;
    int beat = delta / division; // 単位音符数
    delta -= beat * division;
    measure = beat / numerator;
    beat -= measure * numerator;

    if (delta != 0 || beat != 0) {
        throw MetaEventException("Time Signature is not on a measure border", next);
    }
}

bool BeatEvent::isEqual(const Event* rhs) const
{
    const BeatEvent* e = dynamic_cast<const BeatEvent*>(rhs);
    if (e == NULL) return false;

    if (numerator != e->numerator) return false;
    if (denomirator != e->denomirator) return false;
    if (click_interval != e->click_interval) return false;
    if (midi_clock_duration != e->midi_clock_duration) return false;

    return true;
}

//////////////////////////////////////////////////////////////////////
//
// TempoEvent
//
//////////////////////////////////////////////////////////////////////
TempoEvent::TempoEvent(tick_t t, int len, const byte_t* data)
: MetaEvent(t)
{
    tempo = data[0];
    tempo <<= 8;
    tempo |= data[1];
    tempo <<= 8;
    tempo |= data[2];

    // [nw_sndsup:0545] [SMF] テンポを四捨五入するか (#11) への対応
    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware/kagemai/html/user.cgi?project=nw_sndsup&action=view_report&id=545#11
    float ceil = std::ceil(60.f * 1000000 / tempo);
    float floor = std::floor(60.f * 1000000 / (tempo-1));
    if ( ceil == floor )
    {
        tempo = static_cast<unsigned long>(ceil);
    }
    else
    {
        tempo = static_cast<unsigned long>(ceil) - 1;
    }
    // tempo = 60 * 1000000 / tempo;
}

ostream& TempoEvent::put(ostream& s) const
{
    return s << "tempo\t" << tempo;
}
bool TempoEvent::isEqual(const Event* rhs) const
{
    const TempoEvent* e = dynamic_cast<const TempoEvent*>(rhs);
    if (e == NULL) return false;

    if (tempo != e->tempo) return false;

    return true;
}

//////////////////////////////////////////////////////////////////////
//
// TextEvent
//
//////////////////////////////////////////////////////////////////////
TextEvent::TextEvent(tick_t t, string text_)
: MetaEvent(t),
  text(text_)
{
}

ostream& TextEvent::put(ostream& s) const
{
    return s << text;
}
bool TextEvent::isEqual(const Event* rhs) const
{
    return false;
}
//////////////////////////////////////////////////////////////////////
//
// WaitEvent
//
//////////////////////////////////////////////////////////////////////
WaitEvent::WaitEvent(tick_t t, unsigned long d)
: Event(t),
  duration(d)
{
}

ostream& WaitEvent::put(ostream& s) const
{
    return s << "wait\t" << duration;
}
bool WaitEvent::isEqual(const Event* rhs) const
{
    const WaitEvent* e = dynamic_cast<const WaitEvent*>(rhs);
    if (e == NULL) return false;

    if (duration != e->duration) return false;

    return true;
}
//////////////////////////////////////////////////////////////////////
//
// PatternEvent
//
//////////////////////////////////////////////////////////////////////
PatternEvent::PatternEvent(tick_t t, const struct Pattern* patt)
: Event(t),
  pattern(patt)
{
}

ostream& PatternEvent::put(ostream& s) const
{
    return s << "call\t" << pattern->label;
}
bool PatternEvent::isEqual(const Event* rhs) const
{
    const PatternEvent* e = dynamic_cast<const PatternEvent*>(rhs);
    if (e == NULL) return false;

    if (pattern != e->pattern) return false;

    return true;
}
//////////////////////////////////////////////////////////////////////
//
// NoteEvent
//
//////////////////////////////////////////////////////////////////////


NoteEvent::NoteEvent(tick_t t, byte_t data1, byte_t data2)
: MidiEvent(t),
  key(data1),
  velocity(data2),
  duration(-1)
{
}

void NoteEvent::noteOff(tick_t t)
{
    duration = t - tick;
}

ostream& NoteEvent::put(ostream& s) const
{
    return s << convKey2String(key) <<
        "\t" << velocity << ",\t" << duration;
}
void NoteEvent::updateDuration( unsigned int in, unsigned int out)
{
    duration = duration * out / in;
    if ( duration == 0 ) duration = 1;
}

bool NoteEvent::isEqual(const Event* rhs) const
{
    const NoteEvent* e = dynamic_cast<const NoteEvent*>(rhs);
    if (e == NULL) return false;

    if (key != e->key) return false;
    if (velocity != e->velocity) return false;
    if (duration != e->duration) return false;

    return true;
}

//////////////////////////////////////////////////////////////////////
//
// ProgramChangeEvent
//
//////////////////////////////////////////////////////////////////////
ProgramChangeEvent::ProgramChangeEvent(tick_t t, byte_t p)
: MidiEvent(t),
  program(p)
{
}

ostream& ProgramChangeEvent::put(ostream& s) const
{
    return s << "prg\t" << program;
}
bool ProgramChangeEvent::isEqual(const Event* rhs) const
{
    const ProgramChangeEvent* e = dynamic_cast<const ProgramChangeEvent*>(rhs);
    if (e == NULL) return false;

    if (program != e->program) return false;

    return true;
}
//////////////////////////////////////////////////////////////////////
//
// PitchBendEvent
//
//////////////////////////////////////////////////////////////////////
PitchBendEvent::PitchBendEvent(tick_t t, int bend_)
: MidiEvent(t),
  bend( bend_ >> 6 )
{
}

ostream& PitchBendEvent::put(ostream& s) const
{
    return s << "pitchbend\t" << bend;
}
bool PitchBendEvent::isEqual(const Event* rhs) const
{
    const PitchBendEvent* e = dynamic_cast<const PitchBendEvent*>(rhs);
    if (e == NULL) return false;

    if (bend != e->bend) return false;

    return true;
}
//////////////////////////////////////////////////////////////////////
//
// ControlChangeEvent
//
//////////////////////////////////////////////////////////////////////
ControlChangeEvent::ControlChangeEvent(tick_t t, byte_t control_, byte_t value_, const std::string& labelPrefix_)
: MidiEvent(t),
  control(control_),
  value(value_),
  labelPrefix( labelPrefix_ )
{
}

ostream& ControlChangeEvent::put(ostream& s) const
{
    int c = control;

    switch( c ) {
    case 0: // bank_select
        if ( value > BANK_SELECT_MAX || value < 0 )
        {
            // throw ControlChangeEventException("Invalid bank_select value", c, value, tick );
            return s << "; bank_select\t" << value << "\t;; valid value is [0-3]";
        }
        return s << "bank_select\t" << value;
    case 1: // modulation depth
        return s << "mod_depth\t" << value;
    case 3: // init pan
        return s << "init_pan\t" << value;
    case 5: // porta time
        return s << "porta_time\t" << value;
    case 7: // volume
        return s << "volume\t" << value;
    case 9: // surround pan
        return s << "span\t" << value;
    case 10: // pan
        return s << "pan\t" << value;
    case 11: // expression
        return s << "volume2\t" << value;

    case 12: // main_volume
        return s << "main_volume\t" << value;
    case 13: // transpose
        return s << "transpose\t" << ( value - 64 );
    case 14: // prio
        return s << "prio\t" << value;

    case 16: // 汎用操作子
        return s << "setvar\tVAR_0,\t" << value;
    case 17: // 汎用操作子
        return s << "setvar\tVAR_1,\t" << value;
    case 18: // 汎用操作子
        return s << "setvar\tVAR_2,\t" << value;
    case 19: // 汎用操作子
        return s << "setvar\tVAR_3,\t" << value;

    case 20: // bend range
        return s << "bendrange\t" << value;
    case 21: // modulation speed
        return s << "mod_speed\t" << value;
    case 22: // modulation type
        return s << "mod_type\t" << value;
    case 23: // modulation range
        return s << "mod_range\t" << value;
    case 26: // modulation delay
        return s << "mod_delay\t" << value;
    case 27: // modulation delay x10
        return s << "mod_delay\t" << value * 10;
    case 28: // sweep pitch
        return s << "sweep_pitch\t" << ( value - 64 );
    case 29: // sweep pitch x 24
        return s << "sweep_pitch\t" << ( value - 64 ) * 24;
    case 30: // biquad filter type
        return s << "biquad_type\t" << value;
    case 31: // biquad filter value
        return s << "biquad_value\t" << value;

    case 64: // damper pedal
        return s << ( ( value >= 64 ) ? "damper_on" : "damper_off" ) ;
    case 65: // portament
        return s << ( ( value >= 64 ) ? "porta_on" : "porta_off" ) ;
    case 68: // monophonic
        return s << ( ( value >= 64 ) ? "monophonic_on" : "monophonic_off" ) ;

    case 74: // lpf cutoff
        return s << "lpf_cutoff\t" << value;

    case 80: // 汎用操作子
        return s << "setvar\tTVAR_0,\t" << value;
    case 81: // 汎用操作子
        return s << "setvar\tTVAR_1,\t" << value;
    case 82: // 汎用操作子
        return s << "setvar\tTVAR_2,\t" << value;
    case 83: // 汎用操作子
        return s << "setvar\tTVAR_3,\t" << value;

    case 84: // portament control
        return s << "porta\t" << value;
    case 85: //
        return s << "attack\t" << value;
    case 86: //
        return s << "decay\t" << value;
    case 87: //
        return s << "sustain\t" << value;
    case 88: //
        return s << "release\t" << value;

    case 89: // loop start
        return s << "loop_start\t" << value;
    case 90: // loop end
        return s << "loop_end\t";
    case 91: // fx send A
        return s << "fxsend_a\t" << value;
    case 92: // fx send B
        return s << "fxsend_b\t" << value;
    case 93: // fx send C
       return s << "fxsend_c\t" << value;
    case 95: // main bus send
        return s << "mainsend\t" << value;

    case 119: // front bypass
        return s << ( ( value >= 64 ) ? "frontbypass_on" : "frontbypass_off" );
    }

    return s;
}
bool ControlChangeEvent::isEqual(const Event* rhs) const
{
    const ControlChangeEvent* e = dynamic_cast<const ControlChangeEvent*>(rhs);
    if (e == NULL) return false;

    if (control != e->control) return false;
    if (value != e->value) return false;

    return true;
}

//////////////////////////////////////////////////////////////////////
//
// RpnNrpnEvent
//
//////////////////////////////////////////////////////////////////////
RpnNrpnEvent::RpnNrpnEvent(tick_t t, Type type_, byte_t msb_, byte_t lsb_, byte_t value_)
: MidiEvent(t),
  type(type_),
  msb(msb_),
  lsb(lsb_),
  value(value_)
{
}

ostream& RpnNrpnEvent::put(ostream& s) const
{
    int c = ( msb << 8 ) + lsb;

    switch ( type )
    {
    case RPN:
        switch ( c )
        {
        case ( 0 << 8 ) + 0: // bend range
            return s << "bendrange\t" << value;
        default:
            break;
        }
        break;

    case NRPN:
        switch ( c )
        {
        case ( 0 << 8 ) + 0: // env reset
            return s << "env_reset";
        default:
            break;
        }
        break;
    }

    return s;
}
bool RpnNrpnEvent::isEqual(const Event* rhs) const
{
    const RpnNrpnEvent* e = dynamic_cast<const RpnNrpnEvent*>(rhs);
    if (e == NULL) return false;

    if (type != e->type) return false;
    if (msb != e->msb) return false;
    if (lsb != e->lsb) return false;
    if (value != e->value) return false;

    return true;
}

//////////////////////////////////////////////////////////////////////
//
// var_t
//
//////////////////////////////////////////////////////////////////////
strm::fstream& operator>>(strm::fstream& s, var_t& a)
{
    unsigned long num = 0;
    unsigned char u;

    for (int i=0; ; ++i) {
        if ( i >= 4 ) {
            throw ParseException("Invalid file format");
        }
        s.read(&u, 1);
        num <<= 7;
        num |= u & 0x7f;
        if (! (u & 0x80) ) break;
    }
    a = num;
    return s;
}

string convKey2String(int key)
{
    int oct = key / 12;
    key = (oct >= 0) ?
        key - oct * 12 :
        key + 12 - oct * 12;
    --oct;

    static const string keystr[12] = {
        "cn", "cs", "dn", "ds", "en", "fn",
        "fs", "gn", "gs", "an", "as", "bn"
    };
    ostringstream s;
    s << keystr[key];
    if ( oct >= 0) {
        s << oct;
    }
    else {
        s << "m" << -oct;
    }
    return s.str();
}
