﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_CommandFramework.h>

#include "SeqConvInclude.h"
#include "seqconv.h"
#include "parser.h"
#include <sndlib/path.h>
#include <sndlib/utillity.h>
#include <sndlib/nitro.h>
#include <nw/snd/snd_ElementType.h>

using namespace sndlib;

void DefineCommand( Parser& parser );

class CommandlineParameters
{
public:
    CommandlineParameters() : m_IsHelp(false), m_IsUpdate(false), m_IsVerbose(false)
    {
    }

    void Initialize(nn::utilTool::SingleCommandInterface &commandInterface)
    {
        commandInterface.AddHiPriorityFlagArgument(
            nn::utilTool::FlagArgument(
                std::shared_ptr<nn::utilTool::FlagArgumentStore>(new nn::utilTool::BoolFlagArgumentStore(&m_IsHelp)),
                nn::utilTool::KeywordArgumentName('h', "help")),
            nn::utilTool::KeywordArgumentDocument('h', "help", false, "", "Display help.", false));

        commandInterface.AddKeywordArgument(&m_IncludeDirectoryPaths, "include-dir", "add the directory to the #include<> search path", false);
        commandInterface.AddKeywordArgument(&m_OutputFilePath, 'o', "output", "specify output filename", false);
        commandInterface.AddFlagArgument(&m_IsUpdate, 'u', "update", "update only when input file is newer than output file");
        commandInterface.AddFlagArgument(&m_IsVerbose, 'v', "verbose", "explain what is being done");
        commandInterface.SetVariableArguments(&m_InputFilePaths, "files", "intput files", true);
    }

    bool IsHelp() const { return m_IsHelp; }
    bool IsUpdate() const { return m_IsUpdate; }
    bool IsVerbose() const { return m_IsVerbose; }

    const std::vector<std::string>& GetInputFilePaths() const { return m_InputFilePaths; }
    const std::vector<std::string>& GetIncludeDirectoryPaths() const { return m_IncludeDirectoryPaths; }
    const std::string& GetOutputFilePath() const { return m_OutputFilePath; }

private:
    std::vector<std::string> m_InputFilePaths;
    std::vector<std::string> m_IncludeDirectoryPaths;
    std::string m_OutputFilePath;

    bool m_IsHelp;
    bool m_IsUpdate;
    bool m_IsVerbose;
};

void OutputLabelList( const Parser& parser, sndlib::strm::fstream& out, long writebackOffset )
{
    out << strm::binary<u32, TargetEndian>( static_cast<unsigned long>( parser.mGlobalLabelTable.size() ) );

    typedef strm::OffsetWriteback<4, TargetEndian> Offset;
    std::vector<Offset> offsets(parser.mGlobalLabelTable.size());

    for( std::vector<Offset>::iterator p = offsets.begin();
         p != offsets.end() ; ++p )
    {
      #ifdef NW_PLATFORM_RVL
      #else
        // offset とあわせて Reference 型として扱うため、
        // typeId: SequenceSoundFile_LabelInfo, reserved: 0x0000 を挿入しておく
        out << strm::binary<u16, TargetEndian>( nw::snd::internal::ElementType_SequenceSoundFile_LabelInfo )
            << strm::binary<u16, TargetEndian>( 0x0000 ); // u16 reserved
      #endif /* NW_PLATFORM_RVL */
        p->push_cur( out );
    }

    std::vector<Offset>::iterator op = offsets.begin();
    for( Parser::LabelTable::const_iterator p = parser.mGlobalLabelTable.begin() ;
         p != parser.mGlobalLabelTable.end(), op != offsets.end() ;
         ++p, ++op )
    {
        op->writeback( out, writebackOffset );
        out
      #ifdef NW_PLATFORM_RVL
            << strm::binary<u32, TargetEndian>( static_cast<int>(p->second) )
      #else
            // (typeId～offsetをあわせて Reference 型として扱う)
            << strm::binary<u16, TargetEndian>( nw::snd::internal::ElementType_General_ByteStream )
            << strm::binary<u16, TargetEndian>( 0x0000 )                      // u16 reserved
            << strm::binary<u32, TargetEndian>( static_cast<int>(p->second) ) // u32 offset に相当
      #endif /* NW_PLATFORM_RVL */
            << strm::binary<u32, TargetEndian>( static_cast<int>(p->first.length() ) )
            << p->first
            << strm::align(4)
            ;
    }
}

void PrintUsage(nn::utilTool::SingleCommandInterface &commandInterface)
{
    std::string helpText = MakePlainTextCommandDocument(commandInterface.GetDocument());
    std::cout << helpText.c_str();
}

int RunInternal( int argc, char* argv[] )
{
    setlocale( LC_ALL, "" );

    // コマンドライン引数解析
    CommandLineArg cmdinfo;

    {
        nn::utilTool::DefaultCommandLogObserver logObserver;
        nn::utilTool::RegisterCommandLogObserver(logObserver);
        NN_UTIL_SCOPE_EXIT{ UnregisterCommandLogObserver(logObserver); };

        CommandlineParameters params;
        nn::utilTool::SingleCommandInterface commandInterface;

        commandInterface.SetName("SequenceSoundConverter.exe");
        commandInterface.SetSummary("");

        params.Initialize(commandInterface);

        bool result = commandInterface.TryParse(argc - 1, const_cast<const char**>(&argv[1]));
        if (!result)
        {
            PrintUsage(commandInterface);
            return 1;
        }

        if (params.IsHelp())
        {
            PrintUsage(commandInterface);
            return 0;
        }

        for (auto it = params.GetInputFilePaths().begin();
            it != params.GetInputFilePaths().end();
            ++it)
        {
            cmdinfo.files.push_back(*it);
        }

        cmdinfo.out_file = params.GetOutputFilePath();

        for (auto it = params.GetIncludeDirectoryPaths().begin();
            it != params.GetIncludeDirectoryPaths().end();
            ++it)
        {
            cmdinfo.searchPath.push_back(*it);
        }

        cmdinfo.update_flag = params.IsUpdate();
        cmdinfo.verbose_flag = params.IsVerbose();

        if (cmdinfo.verbose_flag)
        {
            logObserver.SetLevel(nn::utilTool::CommandLogLevel_Verbose);
        }
        else
        {
            logObserver.SetLevel(nn::utilTool::CommandLogLevel_Error);
        }
    }

    // メインループ
    CommandLineArg::FileList_t::const_iterator it;
    for( it = cmdinfo.files.begin() ; it != cmdinfo.files.end() ; ++it )
    {
        const std::string infile = *it;
        const std::string basePath = path::GetBasePath( infile );
        const std::string outfile = ( cmdinfo.out_file.empty() ) ? basePath + g_binaryInfo.extesion : cmdinfo.out_file ;

        std::ifstream in( infile.c_str() );
        if ( ! in ) {
            std::stringstream s;
            s << "Cannot open " << infile;
            throw my_exception( s.str() );
        }

        // 更新チェック
        if ( cmdinfo.update_flag &&
             util::CompareFileTime( infile, outfile ) > 0 )
        {
            // 依存ファイルチェックのため空読み
            Lexer lexer( in, infile );
            cmdinfo.AppendSearchPath( lexer );

            while( lexer.ReadToken() != Lexer::END ) {}
            in.clear();
            in.seekg( 0, std::ios_base::beg );

            if ( util::CompareFileTime( lexer.GetDependencyFileList(), outfile ) > 0 )
            {
                if ( cmdinfo.verbose_flag ) {
                    std::cout << "seqconv: " << outfile << " is update." << std::endl;
                }
                continue;
            }
        }

        strm::fstream out;
        Parser parser( cmdinfo );
        // 出力ファイル作成
        if ( ! out.open( outfile.c_str() , "wb" ) )
        {
            std::stringstream s;
            s << "Cannot open " << outfile;
            throw my_exception( s.str() );
        }

        try {
            // ファイル解析
            DefineCommand( parser );

            // ファイル出力
            {
                typedef strm::OffsetWriteback<4, TargetEndian> Offset;
                typedef strm::SizeWriteback<4, TargetEndian> Size;

                Offset dataBlock_offset;
                Offset labelBlock_offset;
                Size dataBlock_size;
                Size labelBlock_size;

                // ヘッダ
                nitro::Header header(g_binaryInfo.signature, g_binaryInfo.version, 2, out );
              #ifdef NW_PLATFORM_RVL
              #else
                out << strm::binary<sndlib::uint16_t, TargetEndian>( nw::snd::internal::ElementType_SequenceSoundFile_DataBlock );
                out << strm::binary<sndlib::uint16_t, TargetEndian>( 0 ); // reserved
              #endif

                dataBlock_offset.push_cur( out );
                dataBlock_size.push_cur( out );

              #ifdef NW_PLATFORM_RVL
              #else
                out << strm::binary<sndlib::uint16_t, TargetEndian>( nw::snd::internal::ElementType_SequenceSoundFile_LabelBlock );
                out << strm::binary<sndlib::uint16_t, TargetEndian>( 0 ); // reserved
              #endif

                labelBlock_offset.push_cur( out );
                labelBlock_size.push_cur( out );

                out << strm::align(32);
                header.EndOfHeader();

                // データブロック
                {
                    dataBlock_offset.writeback( out );
                    dataBlock_size.begin( out );

                    nitro::DataBlock block( 'DATA', out );

                    parser.Parse( infile, in, out );

                    out << strm::binary<sndlib::uint8_t, TargetEndian>( 0xff ); // fin 忘れの保険

                    out << strm::align(32);
                    dataBlock_size.end( out );
                }

                // ラベルブロック
                {
                    labelBlock_offset.writeback( out );
                    labelBlock_size.begin( out );

                    // ブロックヘッダ
                    nitro::DataBlock block( 'LABL', out );
                    const long writebackOffset = out.tell();

                    OutputLabelList( parser, out, writebackOffset );

                    out << strm::align(32);
                    labelBlock_size.end( out );
                }

                out << strm::align(32);
            }
        }
        catch ( ... )
        {
            out.close();
            remove( outfile.c_str() );
            throw;
        }

    }

    return 0;
}

int Run( int argc, char* argv[] )
{
    int result = 0;

    try
    {
        result = RunInternal( argc, argv );
    }
    catch ( const my_exception &e) {
        std::cerr
            << "seqconv: " << e.what() << std::endl
            ;

        result = 1;
    }
    catch ( const Lexer::Exception &e) {
        std::cerr
            << e.what() << std::endl
            ;

        result = 1;
    }
    catch ( const std::exception &e) {
        std::cerr
            << "seqconv: Error: Unexpected exception" << std::endl
            << " message:" << e.what() << std::endl
            ;

        result = 1;
    }
    catch ( ... ) {
        std::cerr
            << "seqconv: Error: Unexpected exception" << std::endl
            ;

        result = 1;
    }

    return result;
}

