﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "seqconv.h"
#include "parser.h"
#include <limits>
#include <nw/snd/snd_MmlCommand.h>

#ifdef max
#undef max
#endif

#ifdef min
#undef min
#endif

using namespace std;
using namespace sndlib;
using namespace nw::snd::internal;
using namespace nw::snd::internal::driver;

struct vmidi {};
struct u24 {};


void Command::Code::Parse( Lexer& lexer, strm::fstream& out, bool testFlag, Parser& parser )
{
    strm::binary<u8, TargetEndian> y = mCode;
    out << y;
}

void Command::Parse( Lexer& lexer, strm::fstream& out, bool testFlag, Parser& parser ) const
{
    if ( mArgs.empty() )
    {
        lexer.ReadToken();
        lexer.VerifyToken( Lexer::RETURN, "newline" );
        return;
    }

    ArgList::const_iterator p;
    for( p = mArgs.begin() ; p != mArgs.end(); ++p )
    {
        (*p)->Parse( lexer, out, testFlag, parser );

        if ( (*p)->IsNeedArg() ) {
            if ( p+1 != mArgs.end() ) {
                lexer.VerifyToken( Lexer::COMMA, "comma" );
            }
        }
    }

    lexer.VerifyToken( Lexer::RETURN, "newline" );
}

Parser::Parser( const CommandLineArg& cmdinfo )
: mFirstFlag( true ),
  mCmdInfo( cmdinfo )
{
}

Parser::~Parser(void)
{
}

// TODO: ExprParser のエラー表示 GetLastTokenString() を用意???

void Parser::Parse( const string& filename, istream& in, strm::fstream& out )
{
    long out_pos = out.tell();
    streamoff in_pos = in.tellg();

    //-----------------------------------------------------------------------------
    // ファーストパス
    //-----------------------------------------------------------------------------
    mFirstFlag = true;

    Lexer lexer( in, filename );
    mCmdInfo.AppendSearchPath( lexer );

    {
        strm::binary<u32, TargetEndian> value_u32 = 0;
        out << value_u32;
    }

    mDataBlockOffset = out.tell();

    ParseBody( lexer, out );

    //-----------------------------------------------------------------------------
    // 巻き戻し処理
    //-----------------------------------------------------------------------------
    mCurGlobalLabel.clear();

    out.seek( out_pos );

    in.clear();
    in.seekg( in_pos, ios_base::beg );


    //-----------------------------------------------------------------------------
    // セカンドパス
    //-----------------------------------------------------------------------------
    mFirstFlag = false;

    Lexer lexer2( in, filename );
    mCmdInfo.AppendSearchPath( lexer2 );

    {
#ifdef NW_PLATFORM_RVL
        strm::binary<u32, TargetEndian> value_u32;
        // ブロックヘッダを起点とする
        value_u32 = mDataBlockOffset - ( out_pos - 8 );
        out << value_u32;
#else
        // オフセットをはさまない
#endif /* NW_PLATFORM_RVL */
    }

    ParseBody( lexer2, out );
}

// 改行チェック
//  @ディレクティブを１行に２つ書いても良いか？

void Parser::ParseBody( Lexer& lexer, strm::fstream& out )
{
    Lexer::Token token;

    lexer.Exceptions( true );

    try {

        while ( 1 )
        {
            token = lexer.ReadToken();

            switch( token ) {

            case Lexer::SYMBOL: {
                const string symbolName = lexer.GetStringValue();

                if ( lexer.PeekToken() == Lexer::COLON ) {
                    // ラベル処理
                    lexer.ReadToken();
                    if ( symbolName[0] != '_' ) {
                        mCurGlobalLabel = symbolName;
                    }

                    if ( mFirstFlag )
                    {
                        LabelTable& labelTable = symbolName[0] == '_'  ?
                                                 mLocalLabelTable[ mCurGlobalLabel ] : mGlobalLabelTable;
                        LabelTable::const_iterator p = labelTable.find( symbolName );
                        if ( p != labelTable.end() ) {
                            stringstream s;
                            s << "Multiple defined label \"" << symbolName << "\"";
                            throw Lexer::Exception( s.str(), lexer );
                        }
                        labelTable[ symbolName ] = out.tell() - mDataBlockOffset;
//                          cout << lexer.GetLineNumber() << ":label:" << symbolName << ":" << out.tell() << endl;
                    }
                    break;
                }

                // コマンド処理
                string command = symbolName;
                if ( symbolName.length() >= 4 && symbolName.substr( symbolName.length() - 3, string::npos ) == "_if" &&
                     symbolName != "envelope_if" )
                {
                    command = symbolName.substr( 0, symbolName.length() - 3 );

                    strm::binary<u8, TargetEndian> y = MmlCommand::MML_IF;
                    out << y;
                }

                string baseCommand = command;
                if ( baseCommand.length() >= 3 && baseCommand.substr( baseCommand.length() - 2, string::npos ) == "_r" )
                {
                    baseCommand = baseCommand.substr( 0, baseCommand.length() - 2 );
                }
                if ( baseCommand.length() >= 3 && baseCommand.substr( baseCommand.length() - 2, string::npos ) == "_v" )
                {
                    baseCommand = baseCommand.substr( 0, baseCommand.length() - 2 );
                }

                int key;
                if ( lexer::ParseKeyString( baseCommand, key ) ) {
                    // ノートコマンド
                    lexer.VerifyMinMaxValue( "note number", key, 0, 127 );
                    strm::binary<u8, TargetEndian> y = static_cast<u8>( key );
                    if ( command.length() >= 3 && command.substr( command.length() - 2, string::npos ) == "_r" )
                    {
                        strm::binary<u8, TargetEndian> c = MmlCommand::MML_RANDOM;
                        out << c << y;
                        mNoteRandCommand.Parse( lexer, out, mFirstFlag, *this );
                    }
                    else if ( command.length() >= 3 && command.substr( command.length() - 2, string::npos ) == "_v" )
                    {
                        strm::binary<u8, TargetEndian> c = MmlCommand::MML_VARIABLE;
                        out << c << y;
                        mNoteVarCommand.Parse( lexer, out, mFirstFlag, *this );
                    }
                    else
                    {
                        out << y;
                        mNoteCommand.Parse( lexer, out, mFirstFlag, *this );
                    }
                }
                else {
                    CommandTable::const_iterator p = mCommandTable.find( command );
                    if ( p == mCommandTable.end() ) {
                        stringstream s;
                        s << "Undefined command \"" << symbolName << "\"";
                        throw Lexer::Exception( s.str(), lexer );
                    }
                    const Command& command = p->second;
                    command.Parse( lexer, out, mFirstFlag, *this );
                }

                break;
            }
            case Lexer::RETURN:
                // skip
                break;

            default:
                stringstream s;
                s << "Syntax error \"" << lexer.GetLastTokenString() << "\"";
                throw Lexer::Exception( s.str(), lexer );
            }
        }
    }
    catch( const Lexer::Failure& )
    {
        // End of File
    }

}

