﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iostream>
#include <fstream>
#include <sstream>
#include <map>
#include <set>
#include <vector>
#include <stack>

#include "smart_ptr.h"

namespace sndlib
{

struct LexerState
{
    std::string fileName;
    int lineNumber;
    std::string lastLine;
};

class Lexer
{
public:
    enum Token
    {
        // character
        RETURN = '\n',
        PLUS='+', MINUS='-',
        MUL='*', DIV='/',
        COLON=':', COMMA=',', PERIOD='.',
        LP = '(', RP = ')',
        LB = '[', RB = ']',
        LBRACE = '{', RBRACE = '}',
        EQUAL = '=',
        OR = '|', AND = '&',

        // state
        END = 256,

        // parameters
        SYMBOL, NUMBER, STRING, DIRECTIVE,

        // 2 charactor operators
        LT, LE, GT, GE, EQ,
        LSHIFT, RSHIFT,
    };

    class Exception;
    class UnexpectedException;
    struct Failure {};

    Lexer( std::istream& in, const std::string& fileName );

    bool Exceptions() const { return mExceptionFlag; }
    void Exceptions( bool flag ) { mExceptionFlag = flag; }

    void VerifyToken( Token token, const std::string& expectedToken ) const;
    void VerifyMinMaxValue( const std::string& name, __int64 value, __int64 min, __int64 max ) const;

    // read token
    Token PeekToken();
    Token ReadToken();

    // get token informations
    Token GetLastToken() const { return mLastToken; }
    std::string GetLastTokenString() const { return mLastTokenString; }
    __int64 GetNumberValue() const;
    std::string GetStringValue() const;
    int GetLineNumber() const;
    std::string GetFileName() const;
    std::string GetLastLine() const;
    LexerState GetState() const;

    bool IsStringValueAvailable() const;
    bool IsNumberValueAvailable() const;

    // search path
    typedef std::vector< std::string > SearchPathList;
    void PushSearchPath( const std::string& path );

    // dependency file
    typedef std::set< std::string > DependencyFileList;
    const DependencyFileList& GetDependencyFileList() const { return mDependencyFileList; }

private:
    struct TokenInfo {
        Token mToken;
        std::string mTokenString;
        __int64 mNumberValue;
        std::string mStringValue;
    };

    struct IfState {
        IfState( bool f, bool elifFlag_=false ) : flag( f ), elseFlag( false ), elifFlag(elifFlag_) {}
        bool flag;
        bool elseFlag;
        bool elifFlag;
    };

    typedef std::vector< TokenInfo > Macro;
    typedef std::map< std::string, SmartPtr<Macro> > MacroTable;
    typedef std::stack< TokenInfo > TokenStack;
    typedef std::vector< IfState > IfStateStack;

    class File {
    public:
        File( std::istream& in, const std::string& fileName ) :
            mInput( in ), mFileName( fileName ),
            mLinePos(0), mLineNumber(0) {}
        bool GetChar( char& ch );
        void UnGet();
        int GetLineNumber() const { return mLineNumber; }
        std::string GetFileName() const { return mFileName; }
        std::string GetLastLine() const { return mLineBuffer; }

    private:
        std::istream& mInput;
        std::string mFileName;
        std::string mLineBuffer;
        size_t mLinePos;
        int mLineNumber;
    };

    class IncludeFile : public File {
    public:
        IncludeFile( const std::string& fileName )
            : File( mFileInput, fileName ),
              mFileInput( fileName.c_str() )
            {}
        bool IsOpen() const { return mFileInput.is_open(); }
    private:
        std::ifstream mFileInput;
    };

    typedef std::vector< SmartPtr<File> > FileStack;

    // member functions
    bool GetChar( char& ch );
    bool GetChar_SkipSpace( char& ch );
    void UnGet();
    Token ParseDirective();

    TokenInfo MakeCurrentTokenInfo() const;
    bool CheckIfState() const;
    static bool IsEqual( const TokenInfo& lhs, const TokenInfo& rhs );

    // member variables
    FileStack mFileStack;

    MacroTable mMacroTable;
    TokenStack mTokenStack;
    SearchPathList mSearchPathList;
    DependencyFileList mDependencyFileList;
    IfStateStack mIfStateStack;

    Token mLastToken;
    std::string mLastTokenString;
    __int64 mNumberValue;
    std::string mStringValue;

    bool mLastTokenAvailable;
    bool mExceptionFlag;
    bool mIgnoreIfState;
};


class Lexer::Exception : public std::exception {
public:
    explicit Exception( const std::string& message, const Lexer& lexer );
    explicit Exception( const std::string& message, const LexerState& state );
    virtual const char *what() const;
private:
    std::string mStr;
};

class Lexer::UnexpectedException : public Lexer::Exception {
public:
    explicit UnexpectedException( const std::string& expectedToken, const Lexer& lexer )
        : Exception( std::string("Expected ") + expectedToken + ", but found \"" + lexer.GetLastTokenString() + "\"", lexer )
        {}
};



namespace lexer
{
    bool ParseKeyString( const std::string& str, int& key );
}

} // namespace sndlib

