﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview.MCS
{
    using System;
    using System.Diagnostics;
    using System.Net;
    using System.Threading;
    using System.Windows.Forms;

    //using NintendoWare.SoundMaker.Configurations;

    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Func;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.Generic.Preview.MCS.Func;
    using NintendoWare.Generic.Preview.MCS.Sync;
    using NintendoWare.Generic.Preview.MCS.Tool;
    using NintendoWare.Generic.Preview.MCS.Viewer;
    using NintendoWare.Generic.Preview.MCS.Ping;

    using NintendoWare.Generic.Preview.MCS.Ctrl;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;

    using NintendoWare.SoundFoundation.Projects;

#if !USE_MCS
    using Nintendo.Alto.Foundation.Communications;
#endif

    /// <summary>
    /// MCSマネージャ
    /// <para>
    /// MCSの管理を行う静的なクラスです。ツールからビューアへパケットを送る接続
    /// （ツール接続）、ビューアからツールへパケットを送る接続（ビューア接続）、
    /// ビューアへのPingを行う接続（Ping接続）の三種類の接続を管理します。
    /// </para>
    /// </summary>
    public class MCSManager : CommManager
    {
        /// <summary>デフォルトホスト名</summary>
        public const string DefaultHostName = "localhost";

#if USE_MCS
        /// <summary>デフォルトポート</summary>
        public const int DefaultPort = 5598;
        /// ポート
        private int _port = DefaultPort;
#endif

        /// ホスト名
        private string _hostName = DefaultHostName;

        ///
        private IPEndPoint ipEndPoint = null;
#if !USE_MCS
        //private IPHostEntry ipHostEntry = null;
        private const int PortBase   = 0x8000 + 0x5344; // 0x8000 + 'SD'
        private const int ToolPort   = PortBase + 0x0;  // 54084
        private const int ViewerPort = PortBase + 0x1;  // 54085
        private const int SyncPort   = PortBase + 0x2;  // 54086
        private const int FuncPort   = PortBase + 0x3;  // 54087
        private const int CtrlPort   = PortBase + 0x4;  // 54088
        private const int PingPort   = PortBase + 0xf;  // 54099

        private const string BaseName = "NW_SND_";
        private string ToolName   = BaseName + "TOOL";
        private string ViewerName = BaseName + "VIEWER";
        private string SyncName   = BaseName + "SYNC";
        private string FuncName   = BaseName + "FUNC";
        private string CtrlName   = BaseName + "CTRL";
        private string PingName   = BaseName + "PING";

        private IComEndPoint currentComEndPoint = null;
#endif
        ///
        delegate bool ConnectToSoundPlayerDelegate();

        public MCSManager()
        {
            this.TargetIsConsole();
        }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="console">MCSのメッセージ出力先となるコンソール</param>
        public override void Initialize( Control invokeControl, ICommConsole console )
        {
            base.Initialize( invokeControl, console);

            ToolConnection   = new MCSToolConnection();
            ViewerConnection = new MCSViewerConnection();
            PingConnection   = new MCSPingConnection();
            SyncConnection = new MCSSyncConnection();
            FuncConnection = new MCSFuncConnection();
            CtrlConnection = new MCSCtrlConnection();

            //
#if !USE_MCS
            TargetIsConsole();
#endif
        }
#if !USE_MCS
        /// <summary>
        ///
        /// </summary>
        public SynchronizationContext SynchronizationContext
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public int SyncPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int FuncPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int CtrlPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int ToolPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int ViewerPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public int PingPortPC { get; set; }

        /// <summary>
        ///
        /// </summary>
        public void TargetIsConsole()
        {
            this.currentComEndPoint = HostIO.Instance;
        }
#endif

#if USE_MCS
        /// <summary>
        /// 接続
        /// </summary>
        private IPEndPoint ConnectToMCS()
        {
            if ( null == InvokeControl ) { throw new Exception( "not initialized." ); }
            if ( null == Console ) { throw new Exception( "not initialized." ); }

            // DNSの解決
            IPEndPoint ipEndPoint = null;
            try
            {
                IPHostEntry entry = Dns.GetHostEntry( _hostName );
                if ( entry.AddressList.Length > 0 )
                {
                    ipEndPoint = new IPEndPoint( entry.AddressList[ 0 ], this._port);
                }
                else
                {
                    ShowError( string.Format(
                        "DNSによる名前解決に失敗しました。{0}:{1}",
                        _hostName, this._port ) );
                    return null;
                }
            }
            catch ( Exception exception )
            {
                ShowError( string.Format(
                    "DNSによる名前解決に問題が発生しました。{0}:{1}\r\n{2}",
                    _hostName, this._port, exception.Message ) );
                return null;
            }

            return ipEndPoint;
        }
#else

        /// <summary>
        ///
        /// </summary>
        private IPHostEntry GetHostEntry( string hostName)
        {
            try
            {
                IPHostEntry entry = Dns.GetHostEntry( hostName);
                if( entry.AddressList.Length <= 0 )
                {
                    ShowError( string.Format( "DNSによる名前解決に失敗しました。{0}", hostName));
                    return null;
                }
                return entry;
            }
            catch( Exception exception)
            {
                ShowError( string.Format
                           ( "DNSによる名前解決に問題が発生しました。{0}\r\n{2}",
                             hostName, exception.Message));
                return null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private object GetConnectPoint( string name)
        {
            return name;
        }

        /// <summary>
        ///
        /// </summary>
        public void ConnectTo6003()
        {
            try
            {
                this.currentComEndPoint.Connect(null);
            }
            catch
            {
            }
        }
#endif
        /// <summary>
        /// 接続します。
        /// </summary>
        public override bool Connect()
        {
#if !USE_MCS
            ConnectTo6003();
#endif
            // 接続されているのか？
            if( IsConnected != false )
            {
                return true;
            }

            // MCSは接続されているが、チャンネル接続がされていないのか？
            if( IsConnecting == false )
            {
#if USE_MCS
                if(( this.ipEndPoint = ConnectToMCS()) == null )
                {
                    return false;
                }
#endif
                SetStateSndEdit( ConnectionState.Connecting);
            }

#if USE_MCS
            // Sync接続の確立
            if( !MCSSyncConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Func接続の確立
            if( !MCSFuncConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Ctrl接続の確立
            if( !MCSCtrlConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }
#else
            //IPAddress ipAddress = this.ipHostEntry.AddressList[0];
            //IPEndPoint ipEndPoint = null;
            //object connectPoint = null;

            // Sync接続の確立
            //ipEndPoint = GetConnectPoint( ipAddress, SyncName);
            MCSSyncConnection.ComEndPoint = this.currentComEndPoint;
            MCSSyncConnection.Port = GetConnectPoint( SyncName);
            if( !MCSSyncConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Func接続の確立
            //ipEndPoint = GetConnectPoint( ipAddress, FuncName);
            MCSFuncConnection.ComEndPoint = this.currentComEndPoint;
            MCSFuncConnection.Port = GetConnectPoint( FuncName);
            if( !MCSFuncConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Ctrl接続の確立
            //ipEndPoint = GetConnectPoint( ipAddress, CtrlName);
            MCSCtrlConnection.ComEndPoint = this.currentComEndPoint;
            MCSCtrlConnection.Port = GetConnectPoint( CtrlName);
            if( !MCSCtrlConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }
#endif

            return true;
        }
#if !USE_MCS
        protected override void OnDisconnected()
        {
            base.OnDisconnected();

            if (this.currentComEndPoint != null)
            {
                this.currentComEndPoint.Disconnect();
            }
        }
#endif
        /// <summary>
        /// SoundPlayerとの接続を行います。
        /// </summary>
        public override bool ConnectToSoundPlayer()
        {
            FormsApplication app = ApplicationBase.Instance as FormsApplication;
            FormsUIService service = app.UIService as FormsUIService;
            MainWindow mainWindow = service.MainWindow;

            if( mainWindow.InvokeRequired != false )
            {
                mainWindow.BeginInvoke
                    ( new ConnectToSoundPlayerDelegate( ConnectToSoundPlayerInternal));
                return true;
            }
            else
            {
                return ConnectToSoundPlayerInternal();
            }
        }

        /// <summary>
        /// SoundPlayerとの接続を行います。
        /// </summary>
        private bool ConnectToSoundPlayerInternal()
        {
            SetState( ConnectionState.Connecting);

#if USE_MCS
            // Tool接続の確立
            if( !MCSToolConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Viewer接続の確立
            if( !MCSViewerConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Ping接続の確立
            if( !MCSPingConnection.Connect( this.ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }
#else

            // Ping接続の確立
            MCSPingConnection.ComEndPoint = this.currentComEndPoint;
            MCSPingConnection.Port = GetConnectPoint( PingName);
            if( !MCSPingConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Tool接続の確立
            MCSToolConnection.ComEndPoint = this.currentComEndPoint;
            MCSToolConnection.Port = GetConnectPoint( ToolName);
            if( !MCSToolConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }

            // Viewer接続の確立
            MCSViewerConnection.ComEndPoint = this.currentComEndPoint;
            MCSViewerConnection.Port = GetConnectPoint( ViewerName);
            if( !MCSViewerConnection.Connect( ipEndPoint, _hostName))
            {
                Disconnect();
                return false;
            }
#endif

            return true;
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public override bool IsWantToExit
        {
            get { return MCSToolConnection.IsWantToExit; }
        }

        /// <summary>
        /// メッセージ表示
        /// </summary>
        protected void ShowMessage( string text )
        {
            Console.WriteMessage( text );
        }

        /// <summary>
        /// エラー表示
        /// </summary>
        protected void ShowError( string text )
        {
            Console.WriteError( text );
        }

        /// <summary>
        /// Tool接続を取得します。
        /// </summary>
        private MCSToolConnection MCSToolConnection
        {
            get
            {
                MCSToolConnection connection = ToolConnection as MCSToolConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Viewer接続を取得します。
        /// </summary>
        private MCSViewerConnection MCSViewerConnection
        {
            get
            {
                MCSViewerConnection connection = ViewerConnection as MCSViewerConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Ping接続を取得します。
        /// </summary>
        private MCSPingConnection MCSPingConnection
        {
            get
            {
                MCSPingConnection connection = PingConnection as MCSPingConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Sync接続を取得します。
        /// </summary>
        private MCSSyncConnection MCSSyncConnection
        {
            get
            {
                MCSSyncConnection connection = SyncConnection as MCSSyncConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Func接続を取得します。
        /// </summary>
        private MCSFuncConnection MCSFuncConnection
        {
            get
            {
                MCSFuncConnection connection = FuncConnection as MCSFuncConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Ctrl接続を取得します。
        /// </summary>
        private MCSCtrlConnection MCSCtrlConnection
        {
            get
            {
                MCSCtrlConnection connection = CtrlConnection as MCSCtrlConnection;
                Debug.Assert( connection != null);
                return connection;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class MCSConsole : ICommConsole
    {
        public void WriteMessage(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }

        public void WriteError(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }
    }
}
